#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2021:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.
import os
import time

from ..log import LoggerFactory
from ..misc.type_hint import TYPE_CHECKING
from ..util import force_memory_trimming

if TYPE_CHECKING:
    from ..misc.type_hint import Any, Optional
    
    ProcessMainInstance = Any

AFTER_FORK_CLEANUP_METHOD = 'after_fork_cleanup'
AFTER_FORK_CLEANUP_PART_LOGGER = 'CLEAN AFTER FORK'


class AfterForkCleanup(object):
    def __init__(self):
        self.top_instance = None
        self.new_instance = None
        self.logger = LoggerFactory.get_logger().get_sub_part(AFTER_FORK_CLEANUP_PART_LOGGER)
    
    
    def register_top_instance(self, inst):
        # type: (ProcessMainInstance) -> None
        # SEF-9050 when forking, we must free useless data from father process
        # get the main daemon / module instance from where cleanup will start
        if self.top_instance is None:
            self.top_instance = inst
        elif self.top_instance != inst:
            self.logger.debug('top_instance:〖 %s 〗 already registered, discarding:〖 %s 〗' % (self.top_instance, inst))
    
    
    def after_fork_can_cleanup(self, inst):
        # type: (ProcessMainInstance) -> bool
        return not (self.new_instance and self.new_instance == inst)
    
    
    def do_after_fork_cleanup(self, after_fork_new_top_instance=None):
        # type: (Optional[ProcessMainInstance]) -> None
        self.new_instance = after_fork_new_top_instance
        LoggerFactory.after_fork_cleanup()
        if hasattr(after_fork_new_top_instance, 'logger'):
            _logger = after_fork_new_top_instance.logger
        else:
            _logger = LoggerFactory.get_logger()
        _logger = _logger.get_sub_part(AFTER_FORK_CLEANUP_PART_LOGGER)
        self.logger = _logger.get_sub_part('pid:%s' % os.getpid(), register=False)
        start = time.time()
        self.logger.info('cleanup is starting')
        top_instance_cleanup = getattr(self.top_instance, 'do_after_fork_cleanup', None)
        if callable(top_instance_cleanup):
            top_instance_cleanup(after_fork_new_top_instance)
        del self.top_instance
        self.top_instance = after_fork_new_top_instance
        self.new_instance = None
        force_memory_trimming(context='because a new process was started')
        self.logger.info('cleanup done in %.3fs' % (time.time() - start))


after_fork_cleanup = AfterForkCleanup()
