#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import collections
import sys

from shinken.misc.type_hint import TYPE_CHECKING
from ..compat import cPickle, cStringIO as StringIO

if TYPE_CHECKING:
    from shinken.misc.type_hint import Str, Any, Type, IO, Union, Iterable, Optional

from ._pickledb_dict import _SAFE_PICKLEABLE_CLASSES_DATABASE


class ShinkenPickleableMeta(type):
    
    def __new__(mcs, name, bases, namespace):
        cls = super(ShinkenPickleableMeta, mcs).__new__(mcs, name, bases, namespace)
        
        if not getattr(cls, '__module__', None):
            raise TypeError('%s does not have __module__ defined' % cls)
        return cls


def shinken_namedtuple(typename, field_names, verbose=False, module=None):
    # type: (Str, Union[Str, Iterable[Str]], bool, Optional[str]) -> Type[tuple]
    # For pickling to work, the __module__ variable needs to be set to the frame
    # where the named tuple is created.
    cls_base = collections.namedtuple(typename, field_names, verbose=verbose)
    if module is None:
        try:
            module = sys._getframe(1).f_globals.get('__name__', '__main__')  # noqa: I need sys._getframe()
        except (AttributeError, ValueError):
            pass
    if module is not None:
        cls_base.__module__ = module
    
    cls = ShinkenPickleableMeta(cls_base.__name__, (cls_base,), {'__module__': cls_base.__module__})  # type: Type[tuple]  # noqa: cls_base is a tuple
    return cls


class ShinkenSecurityUnpicklingError(cPickle.UnpicklingError):
    pass


class ShinkenUnpickler(object, metaclass=ShinkenPickleableMeta):
    __slots__ = ('__file', '__weakref__')
    
    _BYPASS_PICKLE_SAFE_CHECK = False  # Do not touch this variable outside the tests
    
    
    def __init__(self, file):  # noqa: I want the 'file' parameter name
        # type: (IO[str]) -> None
        self.__file = file
    
    
    @classmethod
    def static_find_class(cls, module, name):
        # type: (str, str) -> Any
        if module not in _SAFE_PICKLEABLE_CLASSES_DATABASE or name not in _SAFE_PICKLEABLE_CLASSES_DATABASE[module]:
            raise ShinkenSecurityUnpicklingError('Unpickle [ %s.%s ] is forbidden' % (module, name))
        return getattr(__import__(module, fromlist=['']), name)
    
    
    def find_class(self, module, name):  # noqa: overridable
        # type: (str, str) -> Any
        return self.static_find_class(module, name)
    
    
    # NOTE: we will load the pickle string but important:
    # * cPickle.Unpickler is a C object, that cannot be extended
    def load(self):
        # type: () -> Any
        unpickler = cPickle.Unpickler(self.__file)
        unpickler.find_global = self.find_class
        try:
            return unpickler.load()
        finally:
            delattr(unpickler, 'find_global')
            del unpickler
    
    
    @classmethod
    def loads(cls, pickle_string):
        # type: (str) -> Any
        return cls(StringIO(pickle_string)).load()
