#!/usr/bin/env python
# -*- coding: utf-8 -*-

import threading
import os
import time

from shinken.log import logger


class DrainStopInProgressException(Exception):
    pass


# The transaction Protector help us to know when it's OK to exit a daemon/processus
# * it will +1/-1 current transactions
# * when the daemon ask for exit, it call the drain_stop() that:
#   * do not allow new transaction calls (raise exception)
#   * wait (30s) until all transactions are finish
# IMPORTANT: the transaction protector is useful ACROSS THE SAME PROCESSUS ONLY
class TransactionProtector(object):
    def __init__(self):
        self.__cur_pid = None
        self.__lock = None
        self.__counter = 0
        self.__drain_stop_in_progress = False


    def __enter__(self):
        # Oups, someone want to do a transaction, but we are stopping, do not allow it
        if self.__drain_stop_in_progress:
            raise DrainStopInProgressException()

        current_pid = os.getpid()
        # if was never initialized, do it
        if self.__cur_pid is None:
            self.__cur_pid = current_pid
            self.__lock = threading.RLock()
        # maybe this is a new process, if so, the old lock & counter is dangerous,
        # and must be reinitialized
        if self.__cur_pid != current_pid:
            self.__lock = threading.RLock()
            self.__counter = 0

        with self.__lock:
            self.__counter += 1


    def __exit__(self, exc_type, value, traceback):
        with self.__lock:
            self.__counter -= 1
        return False


    def get_counter(self):
        return self.__counter


    # Will lock until we do not have anymore transaction
    def drain_stop(self, timeout=30):
        # So no new transaction will be done
        self.__drain_stop_in_progress = True

        start = time.time()
        while True:
            # Manage timeout
            if (time.time() - start) > timeout:
                logger.error('Drain stop did fail, there are still %d transactions in progress' % self.__counter)
                return False
            # ok all transactions are done
            if self.__counter == 0:
                return True
            # still loop
            time.sleep(0.1)


    def is_drain_stop_in_progress(self):
        return self.__drain_stop_in_progress


transaction_protecter = TransactionProtector()
