#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2021:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.
import logging
import threading

from .daemon import Interface
from .log import logger, LoggerFactory
from .misc.type_hint import TYPE_CHECKING
from .satellite import BaseSatellite

if TYPE_CHECKING:
    from .configuration_incarnation import ConfigurationIncarnation
    from .misc.type_hint import Dict, Optional
    from .objects.inventory import ElementsInventory


# Interface for method from the arbiter to the daemon with the inventory
class IArbiterToInventorySatellite(Interface):
    # Used by the arbiter to push us the realm inventory
    def push_realm_inventory(self, realm_name, realm_inventory):
        return self.app.push_realm_inventory(realm_name, realm_inventory)
    
    
    push_realm_inventory.doc = '(internal) push a realm inventory to'
    push_realm_inventory.method = 'POST'
    push_realm_inventory.need_lock = False  # the daemon function manage with a lock
    push_realm_inventory.display_name = 'Arbiter server send element inventory to the daemon'
    
    
    # Used by the arbiter to know if we already have an inventory for a realm, and if so avoid
    # pushing it again
    def already_have_realm_inventory(self, realm_name, inventory_hash):
        return self.app.already_have_realm_inventory(realm_name, inventory_hash)
    
    
    already_have_realm_inventory.doc = '(internal) returns if the daemon already have the realm inventory based on its hash'
    already_have_realm_inventory.need_lock = False  # the daemon function manage with a lock
    
    
    # Used by the arbiter to push us inventories
    def push_inventories(self, push_inventories, configuration_incarnation):
        return self.app.push_inventories(push_inventories, configuration_incarnation)
    
    
    push_inventories.doc = '(internal) push a realm inventory to'
    push_inventories.method = 'post'
    push_inventories.need_lock = False  # the daemon function manage with a lock
    
    
    # Used by the arbiter to know if we already have an inventory for each realm, and if so, avoid pushing them again
    def already_have_inventories(self, inventories_hash):
        return self.app.already_have_inventories(inventories_hash)
    
    
    already_have_inventories.doc = '(internal) returns if the daemon already have the inventory based on its hash'
    already_have_inventories.method = 'post'
    already_have_inventories.need_lock = False  # the daemon function manage with a lock


# Class for daemons with inventories, currently Broker & Receiver
class WithInventorySatellite(BaseSatellite):
    properties = BaseSatellite.properties.copy()
    
    
    def __init__(self, name, config_file, is_daemon, do_replace, debug, debug_file, daemon_id=0):
        super(WithInventorySatellite, self).__init__(name, config_file, is_daemon, do_replace, debug, debug_file, daemon_id)
        # Will have one inventory for each realm
        self._realms_inventory = {}  # type: Dict[unicode, ElementsInventory]
        # each time we receive an inventory, compute differences with previous one (if any, so modules can load it faster)
        self._realms_inventory_differences = {}  # type: Dict[unicode, Dict]
        self._realms_inventory_lock = threading.RLock()
        self._realms_inventory_configuration_incarnation = None  # type: Optional[ConfigurationIncarnation]
        # The arbiter let us know about the realms that are allowed to talk to us,
        # telling also if a realm that was present before, and need to be deleted
        self.known_realms = []
        self.logger_inventory = LoggerFactory.get_logger('INVENTORY')
    
    
    # We will give all receiver modules a way to have the inventory if they want
    def assert_module_inventory_are_updated(self):
        with self._realms_inventory_lock:
            for instance in self.modules_manager.get_all_alive_instances():
                f = getattr(instance, 'update_inventory_from_daemon', None)
                if f is None:
                    continue
                try:
                    f(self._realms_inventory, self._realms_inventory_differences, self._realms_inventory_configuration_incarnation)
                except Exception as exp:
                    logger.warning('The mod %s raise an exception: %s, I\'m tagging it to restart later' % (instance.get_name(), str(exp)))
                    logger.warning('Exception type: %s' % type(exp))
                    logger.print_stack(level=logging.WARN)
                    self.modules_manager.did_crash(instance, 'The mod %s raise an exception: %s' % (instance.get_name(), str(exp)))
    
    
    # When a realm is renamed/deleted on the arbiter side, we need to
    # clean it on our inventory list, so modules will be updated too.
    # NOTE: you WILL have realms that are not already in the inventory
    #       because realms are pushing it after schedulers, so it takes time
    def _clean_old_realms_in_inventory(self):
        with self._realms_inventory_lock:
            in_inventory = set(self._realms_inventory.keys())
            known_realms = set(self.known_realms)
            for realm_name in in_inventory - known_realms:
                logger.info('Removing the realm %s from the inventory' % realm_name)
                del self._realms_inventory[realm_name]
    
    
    def already_have_realm_inventory(self, realm_name, checked_inventory_hash):
        with self._realms_inventory_lock:
            realm_inventory = self._realms_inventory.get(realm_name, None)
            if realm_inventory is None:
                return False
            current_realm_inventory_hash = realm_inventory.get_hash()
            return current_realm_inventory_hash == checked_inventory_hash
    
    
    def push_realm_inventory(self, realm_name, realm_inventory):
        with self._realms_inventory_lock:
            old_inventory = self._realms_inventory.get(realm_name, None)
            self._realms_inventory[realm_name] = realm_inventory
            logger.info('The realm %s inventory is now updated with %s elements and the hash %s' % (realm_name, realm_inventory.get_len(), realm_inventory.get_hash()))
            # Now compute differences for this realm from the old one
            # * new elements
            # * updated elements
            # * removed elements
            inventory_differences = None
            if old_inventory is not None:
                inventory_differences = old_inventory.compute_differences_from_new_inventory(realm_inventory, only_hosts=False)
            # set or reset differences
            self._realms_inventory_differences[realm_name] = inventory_differences
            logger.debug('realm[%s] DID COMPUTED DIFFERENCES: %s' % (realm_name, inventory_differences))
    
    
    def already_have_inventories(self, inventories_hash):
        # type: (Dict[unicode,unicode]) ->  Dict[unicode,bool]
        with self._realms_inventory_lock:
            
            already_have_inventories = {}
            for realm_name, inventory_hash in inventories_hash.items():
                already_have_inventories[realm_name] = False
                
                current_realm_inventory = self._realms_inventory.get(realm_name, None)
                if current_realm_inventory:
                    current_realm_inventory_hash = current_realm_inventory.get_hash()
                    inventory_is_up_to_date = current_realm_inventory_hash == inventory_hash
                    already_have_inventories[realm_name] = inventory_is_up_to_date
                    if not inventory_is_up_to_date:
                        self.logger_inventory.debug('must update inventory for realm %s (curr:%s new:%s)' % (realm_name, current_realm_inventory_hash, inventory_hash))
                else:
                    self.logger_inventory.debug('must update inventory for realm %s' % realm_name)
        
        return already_have_inventories
    
    
    def push_inventories(self, push_inventories, configuration_incarnation):
        # type: (Dict[unicode, ElementsInventory], ConfigurationIncarnation) -> None
        with self._realms_inventory_lock:
            self._realms_inventory_configuration_incarnation = configuration_incarnation
            for realm_name, realm_inventory in push_inventories.items():
                old_inventory = self._realms_inventory.get(realm_name, None)
                self._realms_inventory[realm_name] = realm_inventory
                self.logger_inventory.info('The realm %s inventory is now updated with %s elements and the hash %s' % (realm_name, realm_inventory.get_len(), realm_inventory.get_hash()))
                # Now compute differences for this realm from the old one
                # * new elements
                # * updated elements
                # * unchanged elements
                # * removed elements
                inventory_differences = None
                if old_inventory is not None:
                    inventory_differences = old_inventory.compute_differences_from_new_inventory(realm_inventory, only_hosts=False)
                # set or reset differences
                self._realms_inventory_differences[realm_name] = inventory_differences
                self.logger_inventory.debug('realm[%s] COMPUTED DIFFERENCES: %s' % (realm_name, inventory_differences))
    
    
    def _set_daemon_id_of_scheduler(self, daemon, daemon_id):
        raise NotImplementedError()
    
    
    def get_jobs_from_distant(self, e):
        raise NotImplementedError()
    
    
    def do_loop_turn(self):
        raise NotImplementedError()
    
    
    def _set_default_values_to_scheduler_entry(self, entry):
        raise NotImplementedError()
