#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#
# Copyright (C) 2009-2022:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import time
from ctypes import c_double, c_int
from multiprocessing import Value

from .action import ACTION_TYPES
from .load import AvgInRange
from .misc.type_hint import TYPE_CHECKING
from .util import from_int_micro_sec_to_float_sec

if TYPE_CHECKING:
    from .misc.type_hint import Optional
    from .log import PartLogger
    from .action import Action

NO_CPU_DIFFERENCE_TIME = -9999.0  # Need a float as we will share between process


class InWorkerStats(object):
    def __init__(self):
        self.logger_launching = None  # type: Optional[PartLogger]
        
        # Current read to launch actions need to have a counter that stay across loops
        # _enqueued_actions = _enqueued_actions(loop -1) + new_actions_loop1 + new_actions_loop2 - finish_actions_loop1 - finish_actions_loop2
        # NOTE: When asked for stats, we will return _enqueued_actions(loop -1) + new_actions_loop1 + new_actions_loop2 because that was
        #       which actions where in a point in time ready to run
        self._enqueued_actions = 0
        # IMPORTANT: We will keep the trace of enqueue cpu time, BUT in an μs/int way, to not sum float and have rounding deviation in time
        self._enqueued_actions_cpu_time_us = 0
        # stack across the loop for finish cpu_time, in int too! (because here we cannot just sum(float) and expect valid value
        self._finished = 0
        self._finished_estimated_cpu_time_us = 0
        
        # For execution stats
        self._cpu_estimation_differences = Value(c_double, NO_CPU_DIFFERENCE_TIME)
        self._cpu_estimation_differences_this_turn = NO_CPU_DIFFERENCE_TIME
        self._nb_launch_actions_this_turn = 0
        self._nb_launch_actions_cpu_time_this_turn = 0
        self._wait_time_for_resource_this_turn = 0.0
        self._last_wait_time_for_resource_this_turn_save = 0.0
        self._nb_finished_actions_this_turn = 0
        self._nb_finished_actions_cpu_time_this_turn = 0.0
        self._nb_running_actions_this_turn = 0
        self._nb_running_actions_cpu_time_this_turn = 0.0
        self._nb_finished_actions = Value(c_int, 0)
        self._nb_finished_actions_cpu_time = Value(c_double, 0.0)
        self._nb_running_actions = Value(c_int, 0)
        self._nb_running_actions_cpu_time = Value(c_double, 0.0)
        self._nb_launch_actions = Value(c_int, 0)  # exposed value for self._nb_launch_actions_this_turn
        self._nb_launch_actions_cpu_time = Value(c_double, 0.0)  # exposed value for self._nb_launch_actions_cpu_time_this_turn
        self._wait_time_for_resource = Value(c_double, 0.0)  # exposed value for self._wait_time_for_resource_this_turn
        self._nb_ready_to_launch_actions = Value(c_int, 0)  # exposed value for self._enqueued_actions
        self._nb_ready_to_launch_actions_cpu_time = Value(c_double, 0.0)  # exposed value for self._enqueued_actions_cpu_time_us
    
    
    def add_logger(self, logger):
        self.logger_launching = logger.get_sub_part('LAUNCHING')
    
    
    # The worker have a new action to stack, so take the stats from it
    def add_ready_to_launch(self, action):
        # type: (Action) -> None
        self._enqueued_actions += 1
        self._enqueued_actions_cpu_time_us += action.average_cpu_time_in_micro
    
    
    def add_finished_action(self, action):
        # type: (Action) -> None
        self._finished += 1
        self._finished_estimated_cpu_time_us += action.average_cpu_time_in_micro
    
    
    def increase_wait_time_for_resource(self, wait_time):
        # type: (float) -> None
        self._wait_time_for_resource_this_turn += wait_time
    
    
    def increase_nb_launch_actions(self, nb_action_launched, sum_launched_expected_cpu_time):
        self._nb_launch_actions_this_turn += nb_action_launched
        self._nb_launch_actions_cpu_time_this_turn += sum_launched_expected_cpu_time
    
    
    def increase_finished_and_running_stats(self, nb_finished, nb_finished_real_cpu_time, nb_finished_estimated_cpu_time, nb_running, nb_running_cpu_time):
        # We want to have the real differences time, and don't set 0.0 if we have no value, we want real
        # average, so not put 0.0 that will make wrong
        self._nb_finished_actions_this_turn += nb_finished
        self._nb_finished_actions_cpu_time_this_turn += from_int_micro_sec_to_float_sec(self._finished_estimated_cpu_time_us)  # nb_finished_real_cpu_time
        if nb_finished != 0:
            _cpu_difference = nb_finished_estimated_cpu_time - nb_finished_real_cpu_time
            self.logger_launching.debug('[ CPU ESTIMATION ] Have a difference of %.3fs on %d executions' % (_cpu_difference, nb_finished))
            if self._cpu_estimation_differences_this_turn == NO_CPU_DIFFERENCE_TIME:
                self._cpu_estimation_differences_this_turn = _cpu_difference
            else:
                self._cpu_estimation_differences_this_turn += _cpu_difference
        
        # Save launched stats
        self._nb_running_actions_this_turn = nb_running
        self._nb_running_actions_cpu_time_this_turn = nb_running_cpu_time
    
    
    # The executor have 2 loops in 1s, so we need to compute real /1s stats every 2 loops
    def compute_stats_after_two_loops_before_manage_returns(self):
        # As we have all finished, we will have a way to update the enqueue counter
        self._nb_ready_to_launch_actions.value = self._enqueued_actions
        self._enqueued_actions = self._enqueued_actions - self._nb_finished_actions_this_turn
        self._nb_ready_to_launch_actions_cpu_time.value = from_int_micro_sec_to_float_sec(self._enqueued_actions_cpu_time_us)
        self._enqueued_actions_cpu_time_us = self._enqueued_actions_cpu_time_us - self._finished_estimated_cpu_time_us
        
        self._nb_finished_actions.value = self._nb_finished_actions_this_turn
        self._nb_finished_actions_cpu_time.value = self._nb_finished_actions_cpu_time_this_turn
        if self._nb_finished_actions.value == 0:
            self._cpu_estimation_differences.value = NO_CPU_DIFFERENCE_TIME
        else:
            self._cpu_estimation_differences.value = self._cpu_estimation_differences_this_turn
        self._nb_running_actions.value = self._nb_running_actions_this_turn
        self._nb_running_actions_cpu_time.value = self._nb_running_actions_cpu_time_this_turn
        
        # Now reset it
        self._nb_finished_actions_this_turn = 0
        self._nb_finished_actions_cpu_time_this_turn = 0.0
        self._cpu_estimation_differences_this_turn = NO_CPU_DIFFERENCE_TIME
        self._nb_running_actions_this_turn = 0
        self._nb_running_actions_cpu_time_this_turn = 0.0
        self._finished_estimated_cpu_time_us = 0
        self._finished = 0
    
    
    # The executor have 2 loops in 1s, so we need to compute real /1s stats every 2 loops
    def compute_stats_after_two_loops_after_manage_returns(self):
        self._nb_launch_actions.value = self._nb_launch_actions_this_turn
        self._nb_launch_actions_cpu_time.value = self._nb_launch_actions_cpu_time_this_turn
        # Wait time: set a /1s average
        if self._last_wait_time_for_resource_this_turn_save == 0.0:  # First save, let's say we did wait 1s
            _diff_time_since_last_wait_time = 1.0
        else:
            _diff_time_since_last_wait_time = abs(time.time() - self._last_wait_time_for_resource_this_turn_save)  # NOTE: beware of time jump
        
        self._wait_time_for_resource.value = self._wait_time_for_resource_this_turn / _diff_time_since_last_wait_time
        self._last_wait_time_for_resource_this_turn_save = time.time()
        # And reset them
        self._nb_launch_actions_this_turn = 0
        self._nb_launch_actions_cpu_time_this_turn = 0
        self._wait_time_for_resource_this_turn = 0.0
    
    
    def print_loop_debug_stats(self, loop_time, was_loop_aborted, nb_action_launched_by_type, nb_action_launched, nb_action_ready_to_launch, sum_launched_expected_cpu_time, all_action_expected_cpu_time):
        is_loop_aborted_str = '-LIMITED-' if was_loop_aborted else ''
        self.logger_launching.debug(
            '[ Total= %.3f %s ]s [ Waiting CPU/Memory availability (2 loops)= %.3fs ] [ Launched: %d checks, %d notifications, %d event handlers => Total launched: %d / Was ready to be launched: %d] [ Total Expected CPU Time launched: %.3fs / Was ready expected CPU time: %.3fs ] ' % (
                loop_time,
                is_loop_aborted_str,
                self._wait_time_for_resource_this_turn,
                nb_action_launched_by_type[ACTION_TYPES.CHECK],
                nb_action_launched_by_type[ACTION_TYPES.NOTIFICATION],
                nb_action_launched_by_type[ACTION_TYPES.EVENTHANDLER],
                nb_action_launched,
                nb_action_ready_to_launch,
                sum_launched_expected_cpu_time,
                all_action_expected_cpu_time,
            ))
    
    
    # Call from master process, get all worker stats
    def get_stats_from_master_process(self):
        stats = {
            'nb_running'                  : self._nb_running_actions.value,
            'nb_running_cpu_time'         : self._nb_running_actions_cpu_time.value,
            'nb_ready_to_launchs'         : self._nb_ready_to_launch_actions.value,
            'nb_ready_to_launchs_cpu_time': self._nb_ready_to_launch_actions_cpu_time.value,
            'nb_launchs'                  : self._nb_launch_actions.value,
            'nb_launchs_cpu_time'         : self._nb_launch_actions_cpu_time.value,
            'wait_time_for_resource'      : self._wait_time_for_resource.value,
            # default value for these one, fill just after
            'cpu_estimation_difference'   : 0.0,
            'nb_finished_actions'         : 0,
            'nb_finished_actions_cpu_time': 0.0,
        }
        # Now cpu difference with estimation
        _cpu_estimation_difference = self._cpu_estimation_differences.value
        _nb_finished_actions = self._nb_finished_actions.value
        if _nb_finished_actions != 0 and _cpu_estimation_difference != NO_CPU_DIFFERENCE_TIME:
            stats['cpu_estimation_difference'] = _cpu_estimation_difference
            stats['nb_finished_actions'] = _nb_finished_actions
            stats['nb_finished_actions_cpu_time'] = self._nb_finished_actions_cpu_time.value
        return stats


class InMainProcessStatsAggregateAndPrinter(object):
    def __init__(self, managed_actions_str):
        # type: (str) -> None
        self.managed_actions_str = managed_actions_str  # what the daemon is managing (checks, notif, etc), as string
        # Stats of the current workers working (number of launch, cpu time, etc)
        self._in_worker_nb_ready_to_launch_actions = AvgInRange(60)
        self._in_worker_nb_launched_actions = AvgInRange(60)
        self._in_worker_nb_launched_actions_cpu_time = AvgInRange(60)
        self._in_worker_wait_time_for_resource = AvgInRange(60)
        self._in_worker_nb_running_actions = AvgInRange(60)
        self._in_worker_nb_running_actions_cpu_time = AvgInRange(60)
        self._in_worker_nb_finished_actions = AvgInRange(60)
        self._in_worker_nb_finished_actions_cpu_time = AvgInRange(60)
        self._in_worker_cpu_estimation_difference = AvgInRange(60)
        self._overall_workers_cpu_usage_pct1 = AvgInRange(60)
        
        self._enqueued_in_daemon_nb_avg = AvgInRange(60)
        self._enqueued_in_daemon_cpu_time_avg = AvgInRange(60)
    
    
    def get_nb_finished_actions_average(self):
        return self._in_worker_nb_finished_actions.get_avg(0.0)
    
    
    def get_nb_finished_actions_cpu_time_average(self):
        return self._in_worker_nb_finished_actions_cpu_time.get_avg(0.0)
    
    
    def _print_worker_missing_resource_stats(self, logger, nb_ready_to_launchs, nb_ready_to_launchs_cpu_time, nb_launchs, wait_time_for_resource):
        # Display if the Worker did not have enough CPU/RAM/Load average to launch all pending actions
        _nb_not_launched = nb_ready_to_launchs - nb_launchs
        if _nb_not_launched != 0:
            logger.info('[ ALL WORKERS ] [ MISSING RESOURCE ] Must launch: %2d %s  ( Expected CPU Time: %.2fs ) => Launched: %2d/%-2d %s => Wait %.2fs CPU/Memory availability' % (
                nb_ready_to_launchs, self.managed_actions_str, nb_ready_to_launchs_cpu_time, nb_launchs, nb_ready_to_launchs, self.managed_actions_str, wait_time_for_resource))
        else:
            logger.info('[ ALL WORKERS ] Launched all %2d/%2d %s  ( Expected CPU Time: %.2fs )' % (
                nb_ready_to_launchs, nb_launchs, self.managed_actions_str, nb_ready_to_launchs_cpu_time))
    
    
    def _print_worker_stats_immediate(self, logger, nb_to_give, to_give_cpu_time, nb_launchs, nb_launchs_cpu_time, nb_running, nb_running_cpu_time, nb_finished, nb_finished_cpu_time, cpu_estimation_difference,
                                      in_daemon_to_give_back, in_daemon_to_give_back_cpu_time):
        # Global stats for workers
        logger.info('[ PERFS ] [ FOR SCHEDULERS/SYNCHRONIZERS ] [ THIS TURN   ] [ Daemon ]       [ Waiting to be push to workers]   %2d %s, Expected CPU Time: %.2fs' % (nb_to_give, self.managed_actions_str, to_give_cpu_time))
        logger.info(
            '[ PERFS ]                                  [ THIS TURN   ] => [ Workers ]   [ New launch = %2d %s, Expected CPU Time: %.2fs ] [ Executing = %2d %s, Expected CPU Time: %.3fs ] [ Just finished = %2d %s, Consumed CPU Time: %.2fs, %+.2fs from Expected CPU Time ]'
            %
            (nb_launchs, self.managed_actions_str, nb_launchs_cpu_time,
             nb_running, self.managed_actions_str, nb_running_cpu_time,
             nb_finished, self.managed_actions_str, nb_finished_cpu_time, cpu_estimation_difference,
             ))
        logger.info('[ PERFS ]                                  [ THIS TURN   ] [ Daemon ]       [ Waiting to be returned ]         %2d %s, Consumed CPU Time: %.2fs ' % (
            in_daemon_to_give_back, self.managed_actions_str, in_daemon_to_give_back_cpu_time))
    
    
    def _print_worker_stats_average(self, logger, in_daemon_to_give_back_avg, in_daemon_to_give_back_cpu_time_avg):
        # Global stats for workers
        logger.info('[ PERFS ] [ FOR SCHEDULERS/SYNCHRONIZERS ] [ 1min AVERAGE ] [ Daemon ]       [ Waiting to be push to workers ]   %.2f %s/s, Expected CPU Time: %.2fs ' % (
            self._enqueued_in_daemon_nb_avg.get_avg(0.0), self.managed_actions_str, self._enqueued_in_daemon_cpu_time_avg.get_avg(0.0)))
        logger.info(
            '[ PERFS ]                                  [ 1min AVERAGE ] => [ Workers ]   [ New launch = %.2f %s/s, Expected CPU Time: %.2fs ] [ Executing = %.2f %s/s, Expected CPU Time: %.3fs ] [ Just finished = %.2f %s/s, Consumed CPU Time (total): %.2fs, %+.2fs from Expected CPU Time ]'
            %
            (self._in_worker_nb_launched_actions.get_avg(0.0), self.managed_actions_str, self._in_worker_nb_launched_actions_cpu_time.get_avg(0.0),
             self._in_worker_nb_running_actions.get_avg(0.0), self.managed_actions_str, self._in_worker_nb_running_actions_cpu_time.get_avg(0.0),
             self._in_worker_nb_finished_actions.get_avg(0.0), self.managed_actions_str, self._in_worker_nb_finished_actions_cpu_time.get_avg(0.0), self._in_worker_cpu_estimation_difference.get_avg(0.0),
             ))
        logger.info('[ PERFS ]                                  [ 1min AVERAGE ] [ Daemon ]       [ Waiting to be returned ]         %.2f %s/s, Consumed CPU Time: %.2fs ' % (
            in_daemon_to_give_back_avg.get_avg(0.0), self.managed_actions_str, in_daemon_to_give_back_cpu_time_avg.get_avg(0.0)))
    
    
    def print_stats(self, logger, workers,
                    enqueued_in_daemon_nb, enqueued_in_daemon_cpu_time,
                    in_daemon_to_give_back_actions_nb, in_daemon_to_give_back_actions_nb_avg,
                    in_daemon_to_give_back_actions_cpu_time, in_daemon_to_give_back_actions_cpu_time_avg):
        # Now print stats from all workers
        _nb_ready_to_launchs = 0
        _nb_ready_to_launchs_cpu_time = 0.0
        _nb_launchs = 0
        _nb_launchs_cpu_time = 0.0
        _wait_time_for_resource = 0.0
        _nb_running = 0
        _nb_running_cpu_time = 0.0
        _nb_finished = 0
        _nb_finished_cpu_time = 0.0
        _cpu_estimation_difference = 0.0
        
        nb_workers = len(workers)
        for worker in workers:
            worker_stats = worker.get_stats_from_master_process()
            _nb_ready_to_launchs += worker_stats['nb_ready_to_launchs']
            _nb_ready_to_launchs_cpu_time += worker_stats['nb_ready_to_launchs_cpu_time']
            _nb_launchs += worker_stats['nb_launchs']
            _nb_launchs_cpu_time += worker_stats['nb_launchs_cpu_time']
            _wait_time_for_resource += worker_stats['wait_time_for_resource']
            
            _nb_running += worker_stats['nb_running']
            _nb_running_cpu_time += worker_stats['nb_running_cpu_time']
            _nb_finished += worker_stats['nb_finished_actions']
            _nb_finished_cpu_time += worker_stats['nb_finished_actions_cpu_time']
            if _nb_finished != 0:
                _cpu_estimation_difference += worker_stats['cpu_estimation_difference']
        
        _overall_worker_cpu_time = _nb_launchs_cpu_time
        
        _overall_workers_pct_usage = _overall_worker_cpu_time / nb_workers
        _wait_time_for_resource = _wait_time_for_resource / nb_workers  # To have a /s sleep time
        self._overall_workers_cpu_usage_pct1.update_avg(_overall_workers_pct_usage)
        
        self._in_worker_nb_ready_to_launch_actions.update_avg(_nb_ready_to_launchs)
        self._in_worker_nb_launched_actions.update_avg(_nb_launchs)
        self._in_worker_nb_launched_actions_cpu_time.update_avg(_nb_launchs_cpu_time)
        self._in_worker_wait_time_for_resource.update_avg(_wait_time_for_resource)
        self._in_worker_nb_running_actions.update_avg(_nb_running)
        self._in_worker_nb_running_actions_cpu_time.update_avg(_nb_running_cpu_time)
        
        self._in_worker_nb_finished_actions.update_avg(_nb_finished)
        self._in_worker_nb_finished_actions_cpu_time.update_avg(_nb_finished_cpu_time)
        # A 0.0 difference is only meaning full if there are finished actions
        if _nb_finished != 0:
            self._in_worker_cpu_estimation_difference.update_avg(_cpu_estimation_difference)
        
        self._enqueued_in_daemon_nb_avg.update_avg(enqueued_in_daemon_nb)
        self._enqueued_in_daemon_cpu_time_avg.update_avg(enqueued_in_daemon_cpu_time)
        
        self._print_worker_missing_resource_stats(logger, _nb_ready_to_launchs, _nb_ready_to_launchs_cpu_time, _nb_launchs, _wait_time_for_resource)
        
        self._print_worker_stats_immediate(
            logger,
            enqueued_in_daemon_nb,
            enqueued_in_daemon_cpu_time,
            _nb_launchs,
            _nb_launchs_cpu_time,
            _nb_running,
            _nb_running_cpu_time,
            _nb_finished,
            _nb_finished_cpu_time,
            _cpu_estimation_difference,
            in_daemon_to_give_back_actions_nb,
            in_daemon_to_give_back_actions_cpu_time
        )
        
        self._print_worker_stats_average(logger, in_daemon_to_give_back_actions_nb_avg, in_daemon_to_give_back_actions_cpu_time_avg)
