#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

import itertools

from shinken.misc.type_hint import Dict, List, Optional, NoReturn
from shinkensolutions.api.synchronizer import ITEM_TYPE, ItemType
from shinkensolutions.api.synchronizer.source.compute_sync_keys import SyncKeysManager
from shinkensolutions.api.synchronizer.source.display_origin_item.display_origin_item_format import DisplayOriginItemFormat
from shinkensolutions.api.synchronizer.source.item.source_item import SourceItem
from shinkensolutions.api.synchronizer.source.rules_application_template import SourceRulesManager


class ItemsContainers(object):
    
    def __init__(self):
        self.display_origin_item_format = None  # type: Optional[DisplayOriginItemFormat]
        self.source_rules_manager = None  # type: Optional[SourceRulesManager]
        self.sync_keys_service = None  # type: Optional[SyncKeysManager]
        
        self._items = {
            ITEM_TYPE.HOSTS                    : [],
            ITEM_TYPE.HOSTTPLS                 : [],
            ITEM_TYPE.HOSTGROUPS               : [],
            ITEM_TYPE.CLUSTERS                 : [],
            ITEM_TYPE.CLUSTERTPLS              : [],
            ITEM_TYPE.SERVICESHOSTS            : [],
            ITEM_TYPE.SERVICESHOSTTPLS         : [],
            ITEM_TYPE.SERVICESCLUSTERS         : [],
            ITEM_TYPE.SERVICESCLUSTERTPLS      : [],
            ITEM_TYPE.SERVICETPLS              : [],
            ITEM_TYPE.CONTACTS                 : [],
            ITEM_TYPE.CONTACTTPLS              : [],
            ITEM_TYPE.CONTACTGROUPS            : [],
            ITEM_TYPE.ESCALATIONS              : [],
            ITEM_TYPE.NOTIFICATIONWAYS         : [],
            ITEM_TYPE.COMMANDS                 : [],
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS: [],
            ITEM_TYPE.MACROMODULATIONS         : [],
            ITEM_TYPE.RESULTMODULATIONS        : [],
            ITEM_TYPE.TIMEPERIODS              : [],
        }  # type: Dict[ItemType, List[SourceItem]]
    
    
    def set_display_origin_item_format(self, display_origin_item_format):
        # type: (DisplayOriginItemFormat) -> NoReturn
        for i in itertools.chain.from_iterable(iter(self.get_items().values())):
            i.set_display_origin_item_format(display_origin_item_format)
        self.display_origin_item_format = display_origin_item_format
    
    
    def set_sync_key_service(self, sync_keys_service):
        # type: (SyncKeysManager) -> NoReturn
        self.sync_keys_service = sync_keys_service
        for i in itertools.chain.from_iterable(iter(self.get_items().values())):
            self.sync_keys_service.compute_sync_keys(i)
    
    
    def set_source_rules_manager(self, source_rules_manager):
        # type: (SourceRulesManager) -> NoReturn
        self.source_rules_manager = source_rules_manager
    
    
    def apply_host_template_binding_rules(self):
        if not self.source_rules_manager:
            return
        for source_item in itertools.chain.from_iterable(iter(self.get_items().values())):
            template_names = self.source_rules_manager.match(source_item.raw_data)
            if template_names:
                as_str_template_names = ','.join(template_names)
                source_item.use += ',%s' % as_str_template_names if source_item.use else as_str_template_names
    
    
    def add_item(self, item_type, item):
        # type: (ItemType, SourceItem)->NoReturn
        if self.display_origin_item_format:
            item.set_display_origin_item_format(self.display_origin_item_format)
        
        if self.sync_keys_service:
            self.sync_keys_service.compute_sync_keys(item)
        
        self._items[item_type].append(item)
    
    
    def get_items(self):
        return self._items
    
    
    def add_host(self, p_host):
        self.add_item(ITEM_TYPE.HOSTS, p_host)
    
    
    def add_host_template(self, p_host_template):
        self.add_item(ITEM_TYPE.HOSTTPLS, p_host_template)
    
    
    def add_host_groups(self, p_host_group):
        self.add_item(ITEM_TYPE.HOSTGROUPS, p_host_group)
    
    
    def add_cluster(self, p_cluster):
        self.add_item(ITEM_TYPE.CLUSTERS, p_cluster)
    
    
    def add_cluster_template(self, p_cluster_template):
        self.add_item(ITEM_TYPE.CLUSTERTPLS, p_cluster_template)
    
    
    def add_check(self, p_check):
        self.add_item(ITEM_TYPE.SERVICESHOSTS, p_check)
    
    
    def add_check_template(self, p_check_template):
        self.add_item(ITEM_TYPE.SERVICESHOSTTPLS, p_check_template)
    
    
    def add_services_clusters(self, p_service_clusters):
        self.add_item(ITEM_TYPE.SERVICESCLUSTERS, p_service_clusters)
    
    
    def add_services_clusters_templates(self, p_services_clusters_templates):
        self.add_item(ITEM_TYPE.SERVICESCLUSTERTPLS, p_services_clusters_templates)
    
    
    def add_services_templates(self, p_services_templates):
        self.add_item(ITEM_TYPE.SERVICETPLS, p_services_templates)
    
    
    def add_contacts(self, p_contacts):
        self.add_item(ITEM_TYPE.CONTACTS, p_contacts)
    
    
    def add_contacts_templates(self, p_contacts_templates):
        self.add_item(ITEM_TYPE.CONTACTTPLS, p_contacts_templates)
    
    
    def add_contacts_groups(self, p_contacts_groups):
        self.add_item(ITEM_TYPE.CONTACTGROUPS, p_contacts_groups)
    
    
    def add_escalations(self, p_escalations):
        self.add_item(ITEM_TYPE.ESCALATIONS, p_escalations)
    
    
    def add_notification_ways(self, p_notification_ways):
        self.add_item(ITEM_TYPE.NOTIFICATIONWAYS, p_notification_ways)
    
    
    def add_commands(self, p_commands):
        self.add_item(ITEM_TYPE.COMMANDS, p_commands)
    
    
    def add_business_impact_modulations(self, p_business_impact_modulations):
        self.add_item(ITEM_TYPE.BUSINESSIMPACTMODULATIONS, p_business_impact_modulations)
    
    
    def add_macro_modulations(self, p_macro_modulations):
        self.add_item(ITEM_TYPE.MACROMODULATIONS, p_macro_modulations)
    
    
    def add_result_modulations(self, p_result_modulations):
        self.add_item(ITEM_TYPE.RESULTMODULATIONS, p_result_modulations)
    
    
    def add_time_periods(self, p_time_periods):
        self.add_item(ITEM_TYPE.TIMEPERIODS, p_time_periods)
    
    
    def prepare_for_synchronizers(self):
        pass
