#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import NoReturn

from shinkensolutions.api.synchronizer import METADATA, ITEM_TYPE
from shinkensolutions.api.synchronizer.source.display_origin_item.display_origin_item_format import DisplayOriginItemFormat

PRIVATE_PROP = {'_metadata', '_SYNC_KEYS', '_SE_UUID', '__slots__', 'raw_data', 'macros'}


class SourceItem(object):
    __slots__ = ['raw_data', '_metadata', 'item_type', 'use', '_SYNC_KEYS', 'display_name', '_SE_UUID', 'macros', 'enabled', 'pack']
    
    
    def __setattr__(self, key, value):
        if key.startswith('_') and key not in PRIVATE_PROP:
            self.macros[key] = value
        else:
            super(SourceItem, self).__setattr__(key, value)
    
    
    def __getattribute__(self, key):
        if key.startswith('_') and key not in PRIVATE_PROP:
            try:
                return self.macros[key]
            except KeyError as e:
                # we raise a AttributeError here because macros are like properties
                # it make the default value for getattr work
                raise AttributeError(e.message)
        else:
            return super(SourceItem, self).__getattribute__(key)
    
    
    def __init__(self):
        self.macros = {}
        self._metadata = {}
        self.use = ''
        self._SYNC_KEYS = ''
    
    
    def __str__(self):
        display_attr = self.__slots__[:]
        display_attr.remove('item_type')
        display_attr.remove('raw_data')
        all_values = ['%s:[%s]' % (attr, getattr(self, attr, '')) for attr in display_attr if getattr(self, attr, '')]
        return 'SourceItem(%s){%s}' % (self.item_type, '| '.join(all_values))
    
    
    def get_item_type(self):
        return self.item_type
    
    
    def set_metadata(self, metadata_key_name, value):
        self._metadata[metadata_key_name] = value
    
    
    def set_display_origin_item_format(self, format_info):
        # type: (DisplayOriginItemFormat) -> NoReturn
        
        origin_item_info = {
            'format'  : format_info.template_format,
            'raw_data': self.raw_data,
            'title'   : format_info.title,
        }
        
        self.set_metadata(METADATA.ORIGIN_ITEM_INFO, origin_item_info)
    
    
    def compute_sync_keys(self, properties_used_as_synckey):
        sync_keys = set()
        _properties_used_as_synckey = properties_used_as_synckey.get(self.item_type, [])
        if not _properties_used_as_synckey:
            return
        
        for _property_used_as_synckey in _properties_used_as_synckey:
            sync_key_value = getattr(self, _property_used_as_synckey, '')
            if not sync_key_value:
                continue
            
            sync_key_value = sync_key_value.lower()
            
            must_add_tpl_suffix = ITEM_TYPE.is_template(self.item_type) and 'name' == _property_used_as_synckey
            if must_add_tpl_suffix:
                sync_key_value = '%s-tpl' % sync_key_value
            
            sync_keys.add(sync_key_value)
        self._SYNC_KEYS = (','.join(sync_keys))
