#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

from shinkensolutions.api.synchronizer.http_lib_external.v01_00 import Messages, MESSAGE
from shinkensolutions.api.synchronizer.http_lib_external.v01_00 import SELECTOR_CONST, DefineBy, CounterSelect, Counter

from shinken.misc.type_hint import NoReturn
from shinkensolutions.api.synchronizer import RulesComponent, SourceTranslatePart, ComponentManagerSynchronizer, ComponentManagerSyncui
from shinkensolutions.api.synchronizer.source.api_item_properties import ApiItemProperties, CHECK_PROPERTY
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString

DISABLE_VALUE = 'DISABLE'


# TODO : BMAR -> remove cette BGL
def get_component_manager():
    try:
        ComponentManagerSyncui.get_instance()
        return ComponentManagerSyncui
    except Exception:
        pass
    
    try:
        ComponentManagerSynchronizer.get_instance()
        return ComponentManagerSynchronizer
    except Exception:
        pass


class MappingElement(object):
    def __init__(self, item_type, source_prop, origin_prop, description='', is_set_by_user=DefineBy.BY_DEFAULT, mapper_name='', is_valid=True, validation_messages=None, translator=None, is_mapped=True, api_item_properties=None):
        # type: (str, unicode, str, str, str, unicode, bool, Messages, SourceTranslatePart, bool, ApiItemProperties) -> None
        self.translator = translator
        self.item_type = None
        self.item_type_label = ''
        self.source_prop = None
        self.source_prop_label = ''
        self.source_prop_desc = ''
        self.is_set_by_user = None
        self.is_set_by_user_label = ''
        self.description = description
        self.mapper_name = None
        self.mapper_name_filter = SELECTOR_CONST.EMPTY
        self.is_valid = is_valid
        self.is_mapped = is_mapped
        self.validation_messages = validation_messages or Messages()
        self.counter_filter = CounterSelect.EMPTY
        self.counter_html = ''
        self.error_counter_number = 0
        self.warning_counter_number = 0
        self.filter_value = ''
        self.invalid_item_type = False
        self.origin_prop = ''
        self.origin_prop_label = ''
        self.enable = True
        self.set_mapper_name(mapper_name)
        self.set_item_type(item_type)
        self.set_property_name(source_prop)
        self.set_origin_prop(origin_prop, api_item_properties)
        self.set_define_by(is_set_by_user)
        self.compute_counter_html()
        self._sort_key = ''
    
    
    # *****************************   GETTER SETTER  ************************************************************#
    def get_sort_key(self, origin_prop_name):
        # type: (str) -> unicode
        if not self._sort_key:
            self._sort_key = '%s-%s-%s-%s-%s-%s' % (
                1 if self.is_set_by_user == DefineBy.BY_USER_AND_DISABLE else 0, self.is_valid, 1 if self.is_set_by_user == DefineBy.BY_DEFAULT else 0, self.mapper_name if self.mapper_name else '~~~~~', self.item_type_label, origin_prop_name)
        return self._sort_key
    
    
    def set_mapper_name(self, to_set):
        # type: (unicode) -> NoReturn
        self.mapper_name = to_set
        if self.mapper_name:
            self.mapper_name_filter = self.mapper_name
    
    
    def set_item_type(self, to_set):
        # type: (str) -> NoReturn
        self.item_type = to_set
        _validation = RulesComponent.is_unknown_type(self.item_type)
        _validation = [i for i in _validation if i.message != RulesComponent.IS_UNKNOWN_PROPERTY.EMPTY_PROPERTY]
        _is_valid = not bool(len(_validation))
        if not _is_valid:
            self.invalid_item_type = True
            self.is_valid = False
            self.warning_counter_number += 1
            for _current in _validation:
                self.validation_messages.add_message_txt(MESSAGE.STATUS_WARNING, self.translator.translate('mapping_origin_to_source.errors.%s' % _current.message) % _current.params)
        
        if _is_valid and self.translator.is_translate_existing('type.%s' % self.item_type):
            self.item_type_label = self.translator.translate('type.%s' % self.item_type)
        elif not _is_valid:
            self.item_type_label = '''<span class='shinken-format-warning'>%s</span>''' % ToolsBoxString.escape_XSS(to_set)
        else:
            self.item_type_label = ToolsBoxString.escape_XSS(to_set)
    
    
    def set_property_name(self, to_set):
        # type: (unicode) -> NoReturn
        _is_valid = True
        if not isinstance(to_set, str):
            _is_valid = False
        elif to_set.upper() == DISABLE_VALUE:
            self.source_prop = ''
            self.enable = False
            return
        
        self.source_prop = to_set
        if not self.is_mapped:
            return
        
        if self.source_prop == '_SE_UUID':
            _is_valid = False
            self.is_valid = False
            self.validation_messages.add_message_txt(MESSAGE.STATUS_WARNING, self.translator.translate('mapping_origin_to_source.errors.mapping_on_se_uuid'))
            self.warning_counter_number += 1
        
        if not self.invalid_item_type and _is_valid:
            _validation = RulesComponent.is_unknown_property(self.source_prop, self.item_type)
            if len(_validation):
                _is_valid = False
                self.is_valid = False
                self.warning_counter_number += 1
                for _current in _validation:
                    self.validation_messages.add_message_txt(MESSAGE.STATUS_WARNING, self.translator.translate('mapping_origin_to_source.errors.%s' % _current.message))
        
        if _is_valid:
            translate_key = '%s.%s' % (self.item_type, self.source_prop)
            self.source_prop_label = ToolsBoxString.escape_XSS(to_set)
            if self.translator.is_translate_existing(translate_key):
                self.source_prop_desc = self.translator.translate(translate_key)
            else:
                self.source_prop_desc = self.source_prop_label
        else:
            self.source_prop_desc = ''
            self.source_prop_label = '''<span class='shinken-format-warning'>%s</span>''' % ToolsBoxString.escape_XSS(to_set)
    
    
    def set_origin_prop(self, to_set, api_item_properties):
        # type: (str, ApiItemProperties) -> NoReturn
        self.origin_prop = to_set
        
        _is_valid = api_item_properties.check_property(self.item_type, to_set)
        if _is_valid == CHECK_PROPERTY.IS_UNKNOWN:
            self.is_valid = False
            self.warning_counter_number += 1
            self.validation_messages.add_message_txt(MESSAGE.STATUS_WARNING, self.translator.translate('mapping_origin_to_source.errors.unknown_source_key', api_item_properties.user_file.path))
        if _is_valid == CHECK_PROPERTY.IS_DISABLE:
            self.is_valid = False
            self.warning_counter_number += 1
            self.validation_messages.add_message_txt(MESSAGE.STATUS_WARNING, self.translator.translate('mapping_origin_to_source.errors.disable_source_key', api_item_properties.user_file.path))
        
        if _is_valid:
            self.origin_prop_label = ToolsBoxString.escape_XSS(self.origin_prop)
        else:
            self.origin_prop_label = '''<span class='shinken-format-warning'>%s</span>''' % ToolsBoxString.escape_XSS(self.origin_prop)
    
    
    def set_define_by(self, to_set):
        # type: (str) -> NoReturn
        if to_set == DefineBy.BY_USER and not self.enable:
            to_set = DefineBy.BY_USER_AND_DISABLE
        self.is_set_by_user = to_set
        self.is_set_by_user_label = get_component_manager().get_translate_component().translator().translate('common.%s' % self.is_set_by_user)
    
    
    # *****************************   HTML  ************************************************************#
    
    def compute_counter_html(self):
        # type: () -> NoReturn
        if self.is_valid:
            self.counter_html = ''
        else:
            self.counter_filter = CounterSelect.WARNING
            _warning_counter = Counter('shinken-warning-counter', number=str(self.warning_counter_number))
            _warning_counter.add_tooltip()
            _warning_counter.tooltip.set_type('shinken-warning-border')
            _warning_counter.tooltip.set_html(self.validation_messages.get_html())
            self.counter_html = _warning_counter.get_html()
