#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved

from shinken.basemodule import SOURCE_STATE
from shinken.basesubprocess import EventHandler
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.api.synchronizer import ComponentManagerSyncui
from shinkensolutions.api.synchronizer.http_lib_external.v01_00 import OptionSelectComp, SelectComp, SELECTOR_CONST, instance_selector_distributor, HtmlDistributor, LabelHtml
from shinkensolutions.api.synchronizer.source.route.route import AbstractRoute
from shinkensolutions.api.synchronizer.source.tab.tab import AbstractTab

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Optional
    from shinken.log import PartLogger
    from shinkensolutions.api.synchronizer import SourceTranslatePart, ItemType
    from shinkensolutions.api.synchronizer.source.mapping.origin_to_source.mapper import MapperOriginToSource


class TabMappingOriginToSource(AbstractTab):
    def __init__(self, logger, translator, source_name, mapper_origin_to_source, type_to_import):
        # type: (PartLogger,SourceTranslatePart, unicode, MapperOriginToSource,List[ItemType] ) -> None
        super(TabMappingOriginToSource, self).__init__(logger, translator, source_name, 'mapping_origin_to_source', translator.translate('mapping_origin_to_source.tab_display_name'))
        self.mapper_origin_to_source = mapper_origin_to_source
        self.type_to_import = type_to_import
        self.html_distributor = None  # type: Optional[HtmlDistributor]
    
    
    def controller(self):
        common_translator = ComponentManagerSyncui.get_translate_component().translator()
        
        mapping_rules = []
        for mapper_name, mapping_by_type in self.mapper_origin_to_source.get_mapping().items():
            for item_type, item_mapping in mapping_by_type.items():
                for origin_prop_name, mapping_rule in item_mapping.items():
                    mapping_rules.append((mapping_rule.get_sort_key(origin_prop_name), item_type, origin_prop_name, mapping_rule))
        mapping_rules.sort(key=lambda x: x[0])
        
        have_item_class_col = len(self.type_to_import) != 1
        have_mappers_name_col = self.mapper_origin_to_source.have_multi_mapper()
        user_mapping_files = [user_mapping_file for user_mapping_file in self.mapper_origin_to_source.user_mapping_files if 'user_mapping' in user_mapping_file['path']]
        source_exclusive_file = [exclusive_file for exclusive_file in self.mapper_origin_to_source.user_mapping_files if exclusive_file not in user_mapping_files]
        return {
            '__shinken_lang__'   : self.translator.get_lang(),
            'user_mapping_files'   : user_mapping_files,
            'source_exclusive_file': source_exclusive_file,
            'files_configuration'  : self.mapper_origin_to_source.user_mapping_files,
            'validation_state'     : self.mapper_origin_to_source.validation_state.as_html(),
            'mapping_rules'        : mapping_rules,
            'have_item_class_col'  : have_item_class_col,
            'have_mappers_name_col': have_mappers_name_col,
            'source_name'          : self.source_name,
            'translate'            : self.translator.translate,
            'common_translator'    : common_translator,
            'html_distributor'     : self.init_html_object(),
            'tab_info'             : self
        }
    
    
    def init_html_object(self):
        # type: ()-> HtmlDistributor
        if not self.html_distributor:
            _to_return = HtmlDistributor(self.translator)
            # BUTTONS
            _to_return.add_button('reload_rules', 'reload_rules', LabelHtml('''<span class="shinicon-undo"></span>'''))
            # FILTERS
            _to_return.add_list_filter('mapper-name', type_filter='select-single', content=self._build_select_mapper_name())
            _to_return.add_list_filter('item-type', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_ITEM_TYPES))
            _to_return.add_list_filter('source-key', placeholder='--')
            _to_return.add_list_filter('description', placeholder='--')
            _to_return.add_list_filter('shinken-key', placeholder='--')
            _to_return.add_list_filter('shinken-name', placeholder='--')
            _to_return.add_list_filter('define_by', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_DEFINE_BY))
            _to_return.add_list_filter('counters', type_filter='select-single', content=instance_selector_distributor.find_element(SELECTOR_CONST.FILTER_COUNTERS))
            self.html_distributor = _to_return
        return self.html_distributor
    
    
    def _build_select_mapper_name(self):
        select_mapper_name_options = [OptionSelectComp('select-mapper-name', mapper_name) for mapper_name in self.mapper_origin_to_source.mappers_name]
        select_mapper_name_options.insert(0, OptionSelectComp('select-mapper-name', SELECTOR_CONST.EMPTY, label=self.translator.translate('common.%s' % SELECTOR_CONST.EMPTY)))
        select_mapper_name_options.insert(0, OptionSelectComp('select-mapper-name', SELECTOR_CONST.ALL, label=self.translator.translate('common.%s' % SELECTOR_CONST.ALL)))
        select_mapper_name = SelectComp(self.translator, options=select_mapper_name_options, _id='select-mapper-name')
        return select_mapper_name


class RouteReloadMappingOriginToSource(AbstractRoute):
    
    def __init__(self, logger, source_name, mapper_origin_to_source, mapper_event):
        # type: (PartLogger, unicode, MapperOriginToSource, EventHandler) -> None
        super(RouteReloadMappingOriginToSource, self).__init__(logger, source_name, 'mapping_origin_to_source/reload', have_http_template=False)
        self.mapper_origin_to_source = mapper_origin_to_source
        self.mapper_event = mapper_event
    
    
    def controller(self):
        _source_info = ComponentManagerSyncui.get_source_info_component().get_source_info(self.source_name)
        if _source_info.is_import_in_progress():
            return SOURCE_STATE.RUNNING
        
        self.mapper_origin_to_source.reload_mapper()
        self.mapper_event.send_event()
        return SOURCE_STATE.OK
