# !/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

import os
import re

from shinken.compat import unicode_to_bytes
from shinken.misc.type_hint import List, Dict

NOT_FOUND = -1


def _update_conf_file_search_for_patterns(cfg_file_path, conf_name, new_path):
    # type: (unicode, unicode, unicode) -> (List, bool, bool)
    line_found = []
    has_global_data = False
    has_cfg_file = False
    with open(cfg_file_path, 'rb') as cfg_file:
        # Let's do some "grep" config file
        # Looking for "conf_name" occurrences in config file,
        # keeping its positions in file for later processing
        pattern_new_path = re.compile(r'^[^#]+%s' % new_path)
        pattern_global_data = re.compile(r'^[^#]+global-data')
        pattern = re.compile('^[^#]+%s' % conf_name)
        pattern_comment = re.compile(r'^\s*(#.*)*$')
        cfg_file.seek(0, os.SEEK_SET)
        while True:
            begin_offset = cfg_file.tell()
            line = cfg_file.readline()
            end_offset = cfg_file.tell()
            
            if not line:
                break
            if has_global_data and not pattern_comment.match(line):
                # must be at end of file
                has_global_data = False
            if pattern_new_path.match(line):
                has_cfg_file = True
            elif pattern.match(line):
                cur_file = re.sub(r'^.+=([^\s]+)\s*$', '\\1', line)
                if not os.path.isfile(cur_file):
                    line_found.append({'begin_offset': begin_offset, 'end_offset': end_offset, 'replacement': re.sub('^([^=]+=).*$', '\\1%s' % new_path, line), 'global-data': False})
                else:
                    has_cfg_file = True
            elif pattern_global_data.match(line):
                line_found.append({'begin_offset': begin_offset, 'end_offset': end_offset, 'replacement': '', 'global-data': True})
                has_global_data = True
    
    return line_found, has_cfg_file, has_global_data


def _update_conf_file_write_changes(cfg_file_path, new_path, line_found, has_cfg_file, has_global_data, cfg_property_name='cfg_file'):
    # type: (unicode, unicode, List, bool, bool, unicode) -> bool
    
    if has_global_data and has_cfg_file and len(line_found) <= 2:
        return False
    
    with open(cfg_file_path, 'rb+') as cfg_file:
        cfg_file.seek(0, os.SEEK_END)
        if line_found:
            # browse elements from line_found in reversed order
            # and get chunks of file between them
            for elt in line_found[::-1]:
                last_offset = cfg_file.tell()
                cfg_file.seek(elt['end_offset'], os.SEEK_SET)
                elt.update({'file_chunk': cfg_file.read(last_offset - elt['end_offset'])})
                cfg_file.seek(elt['begin_offset'], os.SEEK_SET)
            
            cfg_file.seek(line_found[0]['begin_offset'], os.SEEK_SET)
            
            for elt in line_found:
                if elt['global-data'] is False:
                    if not has_cfg_file:
                        has_cfg_file = True
                        cfg_file.write(elt['replacement'])
                elif has_global_data:
                    # must be at end of file
                    has_global_data = False
                
                cfg_file.write(elt['file_chunk'])
            cfg_file.truncate()
        cfg_file.seek(-1, os.SEEK_END)
        if cfg_file.read(1) != '\n':
            cfg_file.seek(0, os.SEEK_END)
            cfg_file.write('\n')
        cfg_file.seek(0, os.SEEK_END)
        if not has_cfg_file:
            cfg_file.write("%s=%s\n" % (cfg_property_name, new_path))
        if not has_global_data:
            cfg_file.write("cfg_dir=/etc/shinken-user/configuration/global-data\n")
    return True


def update_conf_file_with_new_path(cfg_file_path, conf_name, new_path, new_cfg_property_name='cfg_file'):
    # type: (unicode, unicode, unicode, unicode) -> bool
    
    # Add or replace incorrect settings for "conf_name" in file
    # Add cfg_dir to global data if not present
    # Param
    #  cfg_file_path : (string) path to configuration file
    #  conf_name : (string) cfg_file to update (ex: arbiter_cfg_overload)
    #  new_path : (string) path of new configuration file
    (line_found, has_cfg_file, has_global_data) = _update_conf_file_search_for_patterns(cfg_file_path, conf_name, new_path)
    
    return _update_conf_file_write_changes(cfg_file_path, new_path, line_found, has_cfg_file, has_global_data, new_cfg_property_name)


def _search_for_patterns(file_path, patterns):
    # type: (unicode, List[unicode]) -> Dict
    
    with open(file_path, 'rb') as cfg_file:
        index_patterns = {}
        patterns_compiled = {}
        for pattern in patterns:
            index_patterns[pattern] = NOT_FOUND
            patterns_compiled[pattern] = re.compile('^[^#]*%s' % pattern)
        cfg_file.seek(0, os.SEEK_SET)
        index = 0
        while True:
            line = cfg_file.readline()
            
            if not line:
                break
            
            for pattern, regex in patterns_compiled.items():
                if regex.match(line):
                    index_patterns[pattern] = index
            index += 1
    
    return index_patterns


# Add new parameter in file like shinken.cfg or synchronizer.cfg
def add_new_parameters_to_cfg_file_that_have_no_define(new_lines, file_path, add_after='', add_limit=''):
    # type: (List[unicode], unicode, unicode, unicode) -> None
    patterns = [i for i in (add_after, add_limit) if i]
    index_patterns = _search_for_patterns(file_path, patterns)
    
    with open(file_path, 'r') as cfg_file:
        cfg_file.seek(0, os.SEEK_SET)
        content = cfg_file.readlines()
    
    index_for_new_lines = len(content)
    
    must_write_after = index_patterns.get(add_after, NOT_FOUND)
    if must_write_after != NOT_FOUND and must_write_after < index_for_new_lines:
        index_for_new_lines = must_write_after + 1
    
    limit = index_patterns.get(add_limit, NOT_FOUND)
    if limit != NOT_FOUND and limit < index_for_new_lines:
        index_for_new_lines = limit
    
    for line in reversed(new_lines):
        content.insert(index_for_new_lines, '%s%s' % (unicode_to_bytes(line), '\n'))
    
    with open(file_path, 'w') as cfg_file:
        cfg_file.writelines(content)
