#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.type_hint import TYPE_CHECKING
from ..object.shinken_object import ShinkenObject
from ..constant import FRONT_HTML_FORMAT
from ..component.radio_list_element import RadioListElement

if TYPE_CHECKING:
    from shinken.misc.type_hint import NoReturn, List, Optional, Dict
    from ..object.messages import Messages


class RadioListComp(ShinkenObject):
    def __init__(self, options, _id=''):
        # type: ([RadioListElement], str)-> None
        super(RadioListComp, self).__init__()
        self._options = options[:]  # type: List[RadioListElement]
        self._id = _id
        self.selected_value = None  # type: Optional[str]
        self._mapping_options = None  # type: Optional[Dict(str)]
        self.compute_options()
    
    
    # *****************************   GETTER SETTER  ************************************************************#
    def get_id(self):
        # type: ()-> str
        return self._id
    
    
    def set_selected_value(self, to_set, validation):
        # type: (str, Messages) -> NoReturn
        self.selected_value = to_set
        if self.selected_value not in self._mapping_options:
            _to_add = RadioListElement(self.selected_value, label=self.escape_xss(self.selected_value), extra_class='shinken-error')
            _to_add.add_tooltip()
            _to_add.tooltip.set_html(validation.get_html())
            _to_add.tooltip.set_type('shinken-error-border')
            self._options.insert(0, _to_add)
            self._mapping_options[_to_add.get_value()] = _to_add
    
    
    # *****************************   COMPUTE  ************************************************************#
    def compute_options(self):
        # type: ()-> NoReturn
        self._mapping_options = {}
        for value in self._options:
            self._mapping_options[value.get_value()] = value
    
    
    def add_option(self, to_add):
        # type: (RadioListElement)-> NoReturn
        self._options.append(to_add)
        self._mapping_options[to_add.get_value()] = to_add
    
    
    # *****************************   HTML  ************************************************************#
    
    def get_html(self, html_format=FRONT_HTML_FORMAT.NORMAL, is_lock=False):
        # type: (str, bool) -> str
        _to_return = ['''<div class="shinken-btn-radio-container"><table class="shinken-btn-group-table"><tr>''']
        if is_lock:
            _to_return.append(self.get_html_lock())
        for option in self._options:
            _class_to_add = 'shinken-active' if option.is_name(self.selected_value) else ''
            _to_return.append(option.get_html(class_to_add=_class_to_add, html_format=html_format))
        _to_return.append('''</tr></table></div>''')
        return ''.join(_to_return)


class RadioListCumulativeComp(RadioListComp):
    def __init__(self, options, _id=''):
        # type: ([RadioListElement], str)-> None
        super(RadioListCumulativeComp, self).__init__(options, _id=_id)
    
    
    def get_html(self, html_format=FRONT_HTML_FORMAT.NORMAL, is_lock=False):
        # type: (str, bool) -> str
        _is_selected_found = False
        _index = 0
        _class_to_add_by_option = {}
        for _option in reversed(self._options):
            _class_to_add = ''
            if _option.is_name(self.selected_value):
                _is_selected_found = True
                _class_to_add = 'shinken-active'
            if _is_selected_found:
                if _index:
                    _class_to_add = '%s %s shinken-active-cumulative' % (_class_to_add, 'shinken-active-%s' % _index)
                _index += 1
            _class_to_add_by_option[_option.get_value()] = _class_to_add
        
        _to_return = ['''<div class="shinken-btn-radio-container"><table class="shinken-btn-group-table"><tr>''']
        if is_lock:
            _to_return.append(self.get_html_lock())
        
        for option in self._options:
            _to_return.append(option.get_html(class_to_add=_class_to_add_by_option[option.get_value()], html_format=html_format))
        _to_return.append('''</tr></table></div>''')
        return ''.join(_to_return)
