# !/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.fast_copy import fast_deepcopy, NO_COPY
from shinken.misc.type_hint import TYPE_CHECKING
from .shinken_object import ShinkenObject
from ..constant import FRONT_HTML_FORMAT

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict, Optional, Callable, Any, Union, Tuple
    from shinkensolutions.component.abstract_translate_component import TranslateKey


class MESSAGE(object):
    ICON_CRITICAL = '''<span class='shinken-icon-error shinkon-warning-circle'></span>'''
    ICON_ERROR = '''<span class='shinken-icon-error shinkon-warning'></span>'''
    ICON_WARNING = '''<span class='shinken-icon-warning shinkon-warning'></span>'''
    ICON_BEWARE = '''<span class='shinken-icon-warning shinkon-warning-light'></span>'''
    ICON_INFO = '''<span class='shinken-icon-info shinkon-info-circle'></span>'''
    MAX_ERROR = 5
    MAX_WARNING = 5
    STATUS_NONE = 'shinken-no-status'
    STATUS_INFO = 'shinken-info'
    STATUS_ERROR = 'shinken-error-status'
    STATUS_CRITICAL = 'shinken-critical-status'
    STATUS_WARNING = 'shinken-warning-status'
    STATUS_BEWARE = 'shinken-beware-status'
    WEIGHT_HIGH = 0
    WEIGHT_LOW = 100


class Messages(ShinkenObject):
    def __init__(self, translate):
        # type: (Callable[[TranslateKey, Any],unicode]) -> None
        super(Messages, self).__init__(translate)
        self.messages_critical = []  # type: List[Message]
        self.messages_error = []  # type: List[Message]
        self.messages_warning = []  # type: List[Message]
        self.messages_beware = []  # type: List[Message]
        self.messages_info = []  # type: List[Message]
    
    
    def add_message_txt(self, type_message, text):
        # type: (str, str) -> None
        self.add_message(Message(type_message, text, self.translate))
    
    
    def add_message(self, message):
        # type: (Message) -> None
        if message.get_type() == MESSAGE.STATUS_CRITICAL:
            self.messages_critical.append(message)
        elif message.get_type() == MESSAGE.STATUS_ERROR:
            self.messages_error.append(message)
        elif message.get_type() == MESSAGE.STATUS_WARNING:
            self.messages_warning.append(message)
        elif message.get_type() == MESSAGE.STATUS_BEWARE:
            self.messages_beware.append(message)
        else:
            self.messages_info.append(message)
    
    
    def add_messages(self, messages):
        # type: (List[Message]) -> None
        for _message in messages:
            self.add_message(_message)
    
    
    def has_message(self):
        # type: () -> bool
        return self.has_critical() or self.has_error() or self.has_warning() or self.has_beware() or self.has_info()
    
    
    def get_nb_critical_and_errors(self):
        # type: () -> int
        return self.get_nb_critical() + self.get_nb_error()
    
    
    def get_critical_copy(self):
        # type: () -> List
        return fast_deepcopy(self.messages_critical, additional_dispatcher=ADDITIONAL_DISPATCHER)
    
    
    def get_errors_copy(self):
        # type: () -> List
        return fast_deepcopy(self.messages_error, additional_dispatcher=ADDITIONAL_DISPATCHER)
    
    
    def get_warnings_copy(self):
        # type: () -> List
        return fast_deepcopy(self.messages_warning, additional_dispatcher=ADDITIONAL_DISPATCHER)
    
    
    def get_nb_critical(self):
        # type: () -> int
        return len(self.messages_critical)
    
    
    def get_nb_error(self):
        # type: () -> int
        return len(self.messages_error)
    
    
    def get_nb_warning(self):
        # type: () -> int
        return len(self.messages_warning)
    
    
    def get_nb_beware(self):
        # type: () -> int
        return len(self.messages_beware)
    
    
    def get_nb_info(self):
        # type: () -> int
        return len(self.messages_info)
    
    
    def has_critical(self):
        # type: () -> bool
        return bool(self.get_nb_critical())
    
    
    def has_error(self):
        # type: () -> bool
        return bool(self.get_nb_error())
    
    
    def has_warning(self):
        # type: () -> bool
        return bool(self.get_nb_warning())
    
    
    def has_beware(self):
        # type: () -> bool
        return bool(self.get_nb_beware())
    
    
    def has_info(self):
        # type: () -> bool
        return bool(self.get_nb_info())
    
    
    def get_status(self):
        # type: () -> str
        if self.has_critical():
            return MESSAGE.STATUS_CRITICAL
        if self.has_error():
            return MESSAGE.STATUS_ERROR
        if self.has_warning():
            return MESSAGE.STATUS_WARNING
        if self.has_beware():
            return MESSAGE.STATUS_BEWARE
        if self.has_info():
            return MESSAGE.STATUS_INFO
        return MESSAGE.STATUS_NONE
    
    
    def get_html(self, default_text='', ignored_status=None):
        # type: (unicode, Optional[List[unicode]]) -> unicode
        
        if ignored_status is None:
            ignored_status = []
        if self.get_status() == MESSAGE.STATUS_NONE:
            return default_text
        
        _to_return = ['''<div class='shinken-tooltip-message text-left'>''']
        if self.has_critical():
            if MESSAGE.STATUS_CRITICAL not in ignored_status:
                _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_CRITICAL, self._('element.tooltip_message_tag_critical'), self._build_message_for_list(self.messages_critical)))
        if self.has_error():
            if MESSAGE.STATUS_ERROR not in ignored_status:
                _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_ERROR, self._('element.tooltip_message_tag_error'), self._build_message_for_list(self.messages_error)))
        if self.has_warning():
            if MESSAGE.STATUS_WARNING not in ignored_status:
                _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_WARNING, self._('element.tooltip_message_tag_warning'), self._build_message_for_list(self.messages_warning)))
        if self.has_beware():
            if MESSAGE.STATUS_BEWARE not in ignored_status:
                _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_BEWARE, self._('element.tooltip_message_tag_beware'), self._build_message_for_list(self.messages_beware)))
        if self.has_info():
            if MESSAGE.STATUS_INFO not in ignored_status:
                _to_return.append('''%s%s<br><ol>%s</ol>''' % (MESSAGE.ICON_INFO, self._('element.tooltip_message_tag_info'), self._build_message_for_list(self.messages_info)))
        _to_return.append('''</div>''')
        return ''.join(_to_return)
    
    
    def extract_critical_messages(self):
        # type: () -> Messages
        _to_return = Messages(self.translate)
        _to_return.add_messages(self.messages_critical)
        return _to_return
    
    
    def extract_error_messages(self):
        # type: () -> Messages
        _to_return = Messages(self.translate)
        _to_return.add_messages(self.messages_error)
        return _to_return
    
    
    def extract_warning_messages(self):
        # type: () -> Messages
        _to_return = Messages(self.translate)
        _to_return.add_messages(self.messages_warning)
        return _to_return
    
    
    @staticmethod
    def _build_message_for_list(message_list, max_to_show=999):
        # type: (List[Message], int) -> str
        _size = min(len(message_list), max_to_show)
        sorted_list_by_weight = sorted(message_list, key=lambda m: m.weight)
        return ''.join((m.get_html() for m in sorted_list_by_weight[:_size]))


class Message(ShinkenObject):
    def __init__(self, type_message, text, translate=None, weight=MESSAGE.WEIGHT_LOW):
        # type: (unicode, unicode, Callable[[TranslateKey, Any],unicode], Optional[int]) -> None
        super(Message, self).__init__(translate=translate)
        self._type = type_message
        self.add_class(self._type)
        self._text = text
        self.weight = weight
    
    
    def __eq__(self, other):
        # type: (Message) -> bool
        if not isinstance(other, Message):
            return NotImplemented
        
        return self._type == other._type and self._text == other._text and self.weight == other.weight and self._class == other._class
    
    
    def __ne__(self, other):
        # type: (Message) -> bool
        return not self.__eq__(other)
    
    
    def __repr__(self):
        return '%s : %s' % (self._type, self._text)
    
    
    def get_type(self):
        # type: () -> unicode
        return self._type
    
    
    def get_html(self):
        # type: () -> unicode
        return '''<li %s>%s</li>''' % (self.get_object_tag_html(), self._text)
    
    
    def get_text(self):
        # type: () -> unicode
        return self._text
    
    
    def as_dict(self):
        return self.get_text()


class ValidatorMessage(Message):
    TEXT_KEY = 'text'
    VALIDATION_LIST_KEYS_KEY = 'list_keys'
    
    
    def __init__(self, type_message, text, translate=None, weight=MESSAGE.WEIGHT_LOW, list_keys=None):
        # type: (unicode, unicode, Callable[[TranslateKey, Any],unicode], Optional[int], Optional[unicode]) -> None
        super(ValidatorMessage, self).__init__(type_message, text, translate=translate, weight=weight)
        self.list_keys = list_keys
    
    
    def __repr__(self):
        return '%s : %s' % (self._type, self._text)
    
    
    def get_type(self):
        # type: () -> unicode
        return self._type
    
    
    def get_html(self):
        # type: () -> unicode
        return '''<li %s>%s</li>''' % (self.get_object_tag_html(), self._text)
    
    
    def get_text(self):
        # type: () -> unicode
        return self._text
    
    
    def as_dict(self):
        return {
            self.TEXT_KEY                : self.get_text(),
            self.VALIDATION_LIST_KEYS_KEY: self.list_keys
        }


ADDITIONAL_DISPATCHER = {ValidatorMessage: NO_COPY}


class ValidatorMessages(Messages):
    KEYS_WARNINGS = 'warnings'
    KEYS_ERRORS = 'errors'
    KEYS_CRITICAL = 'critical'
    KEY_ROOT = 'validation_messages'
    
    
    def __init__(self, translate, validation=None, html_format=FRONT_HTML_FORMAT.NORMAL):
        # type: (Optional[Callable[[TranslateKey, Any]], unicode], Dict, unicode) -> None
        super(ValidatorMessages, self).__init__(translate)
        
        if validation:
            warnings = [m.as_dict() for m in validation['warning_messages']]
            errors = [m.as_dict() for m in validation['errors_messages']]
            critical = [m.as_dict() for m in validation['critical_messages']]
            self.init_messages(warnings, errors, critical)
    
    
    def init_messages(self, warnings, errors, critical):
        # type: (List[unicode], List[unicode], List[unicode]) -> None
        self.add_messages([ValidatorMessage(MESSAGE.STATUS_WARNING, message, translate=self.translate) for message in warnings])
        self.add_messages([ValidatorMessage(MESSAGE.STATUS_ERROR, message, translate=self.translate) for message in errors])
        self.add_messages([ValidatorMessage(MESSAGE.STATUS_CRITICAL, message, translate=self.translate) for message in critical])
    
    
    def as_dict(self):
        # type: () -> Union[None, Dict[unicode, List[unicode]]]
        if not self.messages_warning and not self.messages_error and not self.messages_critical:
            return None
        return {
            self.KEYS_WARNINGS: [m.as_dict() for m in self.messages_warning],
            self.KEYS_ERRORS  : [m.as_dict() for m in self.messages_error],
            self.KEYS_CRITICAL: [m.as_dict() for m in self.messages_critical]
        }
    
    
    def add_trad_message(self, status, trad_key, trad_args=(), css_cls=None, list_keys=None):
        # type: (unicode, unicode, Tuple, Optional[unicode], Optional[Union[tuple,unicode]]) -> None
        pass
    
    
    def __add__(self, validator_message):
        # type: (ValidatorMessages) -> ValidatorMessages
        new_validator_message = ValidatorMessages(self.translate, html_format=self.html_format)
        
        new_validator_message.messages_warning.extend(self.messages_warning)
        new_validator_message.messages_error.extend(self.messages_error)
        new_validator_message.messages_critical.extend(self.messages_critical)
        
        new_validator_message.messages_warning.extend(validator_message.messages_warning)
        new_validator_message.messages_error.extend(validator_message.messages_error)
        new_validator_message.messages_critical.extend(validator_message.messages_critical)
        
        return new_validator_message
    
    
    def __iadd__(self, validator_message):
        # type: (ValidatorMessages) -> ValidatorMessages
        self.messages_warning.extend(validator_message.messages_warning)
        self.messages_error.extend(validator_message.messages_error)
        self.messages_critical.extend(validator_message.messages_critical)
        
        return self
    
    
    def clear(self):
        del self.messages_warning[:]
        del self.messages_error[:]
        del self.messages_critical[:]
