#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from .table_body import TableBody
from .table_line import TableLine, TableLineHeader
from ..object.shinken_object import ShinkenObject
from ..object.array import ShinkenArray

if TYPE_CHECKING:
    from shinken.misc.type_hint import NoReturn, List, Optional
    from shinkensolutions.api.synchronizer import TranslatePart
    from .table_cell import TableCell
    

class TableGeneric(ShinkenObject):
    def __init__(self, translate=None, data=None, title='', class_name='', header=None):
        # type: (TranslatePart, List[List[unicode]], unicode, unicode, List[unicode])-> NoReturn
        super(TableGeneric, self).__init__(translate=translate)
        self._class = 'shinken-width-100 shinken-list-table'
        self._title = title
        self._colspan = None  # type: Optional[int]
        self._lines = []  # type: List[TableLine]
        self.add_class(class_name)
        self._headers = None  # type: Optional[ShinkenArray]
        self.compute_data(data)
        self.compute_header(header)
    
    
    def compute_data(self, to_compute):
        # type: (List[List[unicode]])-> NoReturn
        if to_compute is None:
            return
        self._lines = []
        for _line in to_compute:
            _to_add = TableLine(_line, self.translate)
            self._lines.append(_to_add)
        self._colspan = self._lines[0].get_nb_cell()
    
    
    def compute_header(self, to_compute):
        # type: (List[unicode])-> NoReturn
        if to_compute is None:
            return
        self._headers = ShinkenArray(translate=self.translate)
        self._headers.add_element(TableLineHeader(line_data=to_compute, translate=self.translate))
    
    # ****************************************************** GETTER  ***************************************
    
    def get_line(self, index):
        # type: (int)-> TableLine
        return self._lines[index]
    
    
    # ****************************************************** SETTER  ***************************************
    
    def add_line(self, table_line):
        # type: (TableLine)-> NoReturn
        self._lines.append(table_line)
    
    
    def init_header(self, index=0):
        # type: (int)-> NoReturn
        if not self._headers:
            self._headers = ShinkenArray(translate=self.translate)
        
        if self._headers.get_size() < (index + 1):
            self._headers.add_element(TableLineHeader(translate=self.translate))
            return self.init_header(index)
    
    def add_header(self, to_add, index=0):
        # type: (TableCell, int)-> NoReturn
        self.get_line_header(index).add_cell(to_add)

    def get_line_header(self, index=0):
        # type: (int)-> TableLineHeader
        self.init_header(index)
        return self._headers.get_element(index)
    
    
    # ****************************************************** HTML  ***************************************
    
    def get_html(self):
        # type: () -> unicode
        
        _to_return = ['''<div class="shinken-list-container %s"><table %s>''' % (self._class_container, self.get_object_tag_html()),
                      self.get_html_title(),
                      self.get_html_thead(),
                      self.get_html_tbody(),
                      '''</table></div>''']
        return ''.join(_to_return)
    
    
    def get_html_title(self):
        # type: () -> unicode
        if not self._title:
            return ''
        _to_return = [
            '''<thead><tr class="shinken-title-line">''',
            '''<th class="shinken-title-cell" colspan="%s">''' % self._colspan,
            '''<div class="shinken-title-cell-container">%s</div>''' % self._title,
            '''</th></tr></thead>'''
        ]
        return ''.join(_to_return)
    
    
    def get_html_thead(self):
        # type: ()-> str
        if not self._headers:
            return ''
        _to_return = ['''<thead>''', self._headers.get_html(), '''</thead>''']
        return ''.join(_to_return)
    
    
    def get_html_tbody(self):
        # type: ()-> str
        _to_return = ['''<tbody class="shinken-list-tbody">''']
        for _line in self._lines:
            _to_return.append(_line.get_html())
        _to_return.append('''</tbody>''')
        return ''.join(_to_return)


class TableWithTbody(TableGeneric):
    def __init__(self, translate=None, title='', class_name='', header=None):
        # type: (TranslatePart, str, str, List[unicode])-> NoReturn
        super(TableWithTbody, self).__init__(translate=translate, title=title, class_name=class_name, header=header)
        # self._data = data
        self._tbodies = []  # type: List[TableBody]
    
    
    # ****************************************************** GETTER ***************************************
    def get_tbody(self, index):
        # type: (int)-> TableBody
        return self._tbodies[index]
    
    
    # ****************************************************** SETTER  ***************************************
    
    def add_tbody(self, to_add):
        # type: (TableBody)-> NoReturn
        self._tbodies.append(to_add)
    
    
    def get_html_tbody(self):
        # type: ()-> str
        _to_return = []
        for _tbody in self._tbodies:
            _to_return.append(_tbody.get_html())
        return ''.join(_to_return)


class TableWithFixHeaders(TableGeneric):
    def __init__(self, translate=None, class_name='', header=None):
        # type: (TranslatePart, unicode, List[unicode])-> NoReturn
        super(TableWithFixHeaders, self).__init__(translate=translate, class_name=class_name, header=header)


    # ****************************************************** HTML  ***************************************

    def get_html(self):
        # type: () -> unicode
        _to_return = ['''<div class="shinken-list-container %s">''' % self._class_container]
        _to_return.append('''<div class="shinken-list-container-fixed-part">''')
        _to_return.append('''<table %s>''' % self.get_object_tag_html())
        _to_return.append('''%s''' % self.get_html_thead())
        _to_return.append('''</table>''')
        _to_return.append('''</div>''')
        _to_return.append('''<div class="shinken-list-container-scrolled-part">''')
        _to_return.append('''<table %s>''' % self.get_object_tag_html())
        _to_return.append('''%s''' % self.get_html_thead())
        _to_return.append('''%s''' % self.get_html_tbody())
        _to_return.append('''</table>''')
        _to_return.append('''</div>''')
        _to_return.append('''</div>''')
        return ''.join(_to_return)
