#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING, cast
from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriver, AbstractDataHubDriverConfig
from shinkensolutions.data_hub.data_hub_exception.data_hub_exception import DataHubItemNotFound
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Any
    from shinken.log import PartLogger
    from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriverConfig
    from shinkensolutions.data_hub.data_hub import DataHubConfig


class DataHubMetaDriverConfigReplication(AbstractDataHubDriverConfig):
    def __init__(self, config_drivers):
        # type: (List[AbstractDataHubDriverConfig]) -> None
        super(DataHubMetaDriverConfigReplication, self).__init__('REPLICATION')
        self.config_drivers = config_drivers


def data_hub_meta_driver_replication_factory(logger, driver_config, data_hub_config):
    # type: (PartLogger, AbstractDataHubDriverConfig, DataHubConfig) -> DataHubMetaDriverReplication
    driver_config = cast(DataHubMetaDriverConfigReplication, driver_config)
    drivers = [DataHubFactory.build_driver(logger, driver_conf, data_hub_config) for driver_conf in driver_config.config_drivers]
    return DataHubMetaDriverReplication(logger, driver_config, drivers)


DataHubFactory.register_driver_factory(DataHubMetaDriverConfigReplication, data_hub_meta_driver_replication_factory)


class DataHubMetaDriverReplication(AbstractDataHubDriver):
    def __init__(self, logger, driver_config, drivers):
        # type: (PartLogger, DataHubMetaDriverConfigReplication, List[AbstractDataHubDriver]) -> None
        super(DataHubMetaDriverReplication, self).__init__(logger, driver_config)
        
        self.drivers = drivers
    
    
    def get_size_of(self, data_id):
        raise NotImplementedError()
    
    
    def get_lock(self, data_id):
        raise NotImplementedError()
    
    
    def init(self):
        # type: () -> None
        for driver in self.drivers:
            driver.init()
    
    
    def get_all_data_id(self):
        # type: () -> List[unicode]
        all_data_id = set()
        for driver in self.drivers:
            all_data_id.update(driver.get_all_data_id())
        return list(all_data_id)
    
    
    def is_data_correct(self, data_id):
        # type: (unicode) -> bool
        for driver in self.drivers:
            try:
                if driver.is_data_correct(data_id):
                    return True
            except Exception:
                pass
        
        return False
    
    
    def write(self, data_id, data):
        # type: (unicode, Any) -> None
        for driver in self.drivers:
            driver.write(data_id, data)
    
    
    def write_raw(self, data_id, raw_data):
        # type: (unicode, Any) -> None
        for driver in self.drivers:
            driver.write_raw(data_id, raw_data)
    
    
    def read(self, data_id, log_error=True):
        # type: (unicode, bool) -> Any
        for driver in self.drivers:
            try:
                data = driver.read(data_id, log_error)
                return data
            except Exception:
                pass
        
        raise DataHubItemNotFound(self._data_type, data_id)
    
    
    def read_raw(self, data_id, log_error=True):
        # type: (unicode, bool) -> Any
        for driver in self.drivers:
            try:
                data = driver.read_raw(data_id, log_error)
                return data
            except Exception:
                pass
        
        raise DataHubItemNotFound(self._data_type, data_id)
    
    
    def remove(self, data_id):
        # type: (unicode) -> None
        for driver in self.drivers:
            driver.remove(data_id)
    
    
    def get_last_modification_date(self, data_id):
        # type: (unicode) -> int
        for driver in self.drivers:
            try:
                last_file_modification = driver.get_last_modification_date(data_id)
                return last_file_modification
            except Exception:
                pass
        
        raise DataHubItemNotFound(self._data_type, data_id)
    
    
    def destroy(self):
        # type: () -> None
        for driver in self.drivers:
            driver.destroy()
    
    
    def stop(self):
        # type: () -> None
        for driver in self.drivers:
            driver.stop()
    
    
    def check_right_access_issues_then_fix_them(self):
        # type: () -> None
        for driver in self.drivers:
            driver.check_right_access_issues_then_fix_them()
    
    
    def get_total_size(self):
        # type: () -> int
        for driver in self.drivers:
            try:
                return driver.get_total_size()
            except Exception:
                pass
    
    
    def get_number_of_stored_data(self):
        # type: () -> int
        for driver in self.drivers:
            try:
                return driver.get_number_of_stored_data()
            except Exception:
                pass
