import os
import hashlib
import shutil
import uuid

from shinken.log import logger, get_chapter_string, get_section_string


CHECK_PLUGIN_STYLES_DIRECTORY = '/var/lib/shinken-user/external-resources'
CHECK_PLUGIN_STYLES_CSS_DIRECTORY = os.path.join(CHECK_PLUGIN_STYLES_DIRECTORY, 'css')

CHECK_PLUGIN_STYLES_OUTPUT_FILE = '/var/lib/shinken/shinken_check_plugin_style_combined.css'

CHECK_PLUGIN_STYLES_CSS_PREFIX = 'shinken_check_plugin_style_'

CHAPTER_NAME = get_chapter_string('EXTERNAL-RESSOURCES')
SECTION_NAME = get_section_string('PLUGIN-STYLE-LOADING')


class External_Resources(object):
    def __init__(self):
        pass
    
    
    # We will load all shinken_check_plugin_style_*.css files and aggregate them
    # into a unique file
    # Give back :
    #  * the file path of the combined css
    #  * the sha1 sum of the file
    def load_and_combine_css(self):
        # UNCOMMENT AS SOON AS WE HAVE MORE THAN 1 FILE
        # css_pattern = '%s*.css' % CHECK_PLUGIN_STYLES_CSS_PREFIX
        # css_files = glob.glob('%s/%s' % (CHECK_PLUGIN_STYLES_CSS_DIRECTORY, css_pattern))
        # nb_css_files = len(css_files)
        
        # TODEL: AS SOON AS NO MORE 1 FILE
        unique_css_file = '%s/%sspecific.css' % (CHECK_PLUGIN_STYLES_CSS_DIRECTORY, CHECK_PLUGIN_STYLES_CSS_PREFIX)
        logger.debug('UNIQ CSS FILE: %s' % unique_css_file)
        if os.path.exists(unique_css_file):
            logger.info('%s %s The file %s will be exported to the UIs' % (CHAPTER_NAME, SECTION_NAME, unique_css_file))
            css_files = [unique_css_file]
        else:
            logger.info('%s %s The file %s is missing, so no custom css will be exported to the UIs' % (CHAPTER_NAME, SECTION_NAME, unique_css_file))
            css_files = []
        
        nb_css_files = len(css_files)
        
        if nb_css_files:
            # Uncomment as soon as the 1 file limit is no more
            # logger.info('%s %s All the css code will be aggregated in a single css file' % (CHAPTER_NAME, SECTION_NAME))
            # logger.info('%s %s The alphabetic order of the files will give the order in the final single css file' % (CHAPTER_NAME, SECTION_NAME))
            logger.info('%s %s - Make sure your styles always begin by the class ".shinken-check-container"' % (CHAPTER_NAME, SECTION_NAME))
            logger.info('%s %s   EXAMPLE: .shinken-check-container .label-error{color:red}' % (CHAPTER_NAME, SECTION_NAME))
            logger.info('%s %s   in order to avoid any custom css overriding with the Shinken UI css styles' % (CHAPTER_NAME, SECTION_NAME))
        # all_lines = [u'/*  SHINKEN CHECK PLUGIN STYLES CSS. file with pattern %s from directory %s */\n\n' % (css_pattern, CHECK_PLUGIN_STYLES_CSS_DIRECTORY)]
        all_lines = ['/*  SHINKEN CHECK PLUGIN STYLES CSS. */']
        for css_file in css_files:
            all_lines.append('/*  SHINKEN CHECK PLUGIN STYLES CSS FILE: START :: %s */' % css_file)
            try:
                # Open in Universal way for Windows, Mac, Linux
                with open(css_file, 'rU') as fd:
                    css_lines = fd.readlines()
            except IOError as exp:
                logger.error('%s %s Cannot read the file %s : %s' % (CHAPTER_NAME, SECTION_NAME, css_file, exp))
                # TODO: raise?
                raise
            for (line_nb, line) in enumerate(css_lines):
                line_nb += 1
                try:
                    line = line.decode('utf8').strip()
                    all_lines.append(line)
                except UnicodeDecodeError as exp:
                    logger.error('%s %s Cannot read the file %s at line %s, this is not a utf8 file: %s' % (CHAPTER_NAME, SECTION_NAME, css_file, line_nb, exp))
                    # TODO: raise?
                    raise
            all_lines.append('/*  SHINKEN CHECK PLUGIN STYLES CSS FILE: END :: %s */\n\n' % css_file)
        
        full_css_output = '\n'.join(all_lines)
        
        # Finally write the file, and not in place as maybe someone is reading it
        compiled_css_path = CHECK_PLUGIN_STYLES_OUTPUT_FILE
        try:
            # Note: using a uuid4 because we can have multiple webui that are doing it on the same time
            tmp_file_path = compiled_css_path + uuid.uuid4().hex + '.tmp'
            logger.debug('%s %s Using temporary file path: %s' % (CHAPTER_NAME, SECTION_NAME, tmp_file_path))
            with open(tmp_file_path, 'wb') as f:
                f.write(full_css_output)
            shutil.move(tmp_file_path, compiled_css_path)
        except Exception as exp:
            logger.error('%s %s Cannot write the combined the file %s : %s' % (CHAPTER_NAME, SECTION_NAME, compiled_css_path, exp))
            # TODO: raise?
            raise
        
        compiled_css_hash = hashlib.sha1(full_css_output).hexdigest()
        return compiled_css_path, compiled_css_hash


external_resources = External_Resources()
