#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.exceptions.business import ShinkenExceptionKeyError
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.filter.list.items.abstract_list_filter_item import AbstractListFilterItem

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Dict, List


class SORT_FILTER_CONSTS(object):
    ORDER = 'order'  # type: unicode
    ORDER__ASCENDANT = 'asc'  # type: unicode
    ORDER__DESCENDANT = 'desc'  # type: unicode


class AbstractListFilterItemSort(AbstractListFilterItem):
    def __init__(self, order=SORT_FILTER_CONSTS.ORDER__ASCENDANT):
        if order not in (SORT_FILTER_CONSTS.ORDER__ASCENDANT, SORT_FILTER_CONSTS.ORDER__DESCENDANT):
            raise ShinkenExceptionKeyError(text="Unknown order value [ %s ]" % order)
        self.order = order
    
    
    def apply_on(self, list_to_filter):
        # type: (List[Any]) -> List[Any]
        list_to_filter.sort(key=self._filter_key, reverse=(self.order == SORT_FILTER_CONSTS.ORDER__DESCENDANT))
        return list_to_filter
    
    
    def _filter_key(self, obj):
        # type: (Any) -> Any
        raise NotImplementedError()
    
    
    @classmethod
    def get_valid_keys(cls):
        # type: () -> List[unicode]
        return [SORT_FILTER_CONSTS.ORDER]
    
    
    @classmethod
    def from_dict(cls, filter_dict):
        # type: (Dict[unicode, Any]) -> AbstractListFilterItemSort
        order = filter_dict.get(SORT_FILTER_CONSTS.ORDER, None)
        if order is None:
            raise ShinkenExceptionKeyError(text='Missing [ %s ] key' % SORT_FILTER_CONSTS.ORDER)
        return cls(order=order)


class ListFilterItemSort(AbstractListFilterItemSort):
    def _filter_key(self, obj):
        # type: (Any) -> Any
        return obj
