#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.logs.logger_abstract import LoggerAbstract

if TYPE_CHECKING:
    from typing import Optional


class LoggerAuthentication(LoggerAbstract):
    FORMATS = [
        ('log_time', '[ %(log_time)s ]'),
        ('mode', '[ %(mode)-5s ]'),
        ('type_request', '[ %(type_request)-6s ]'),
        ('result_code', '[ RESULT:%(result_code)-3s ]'),
        ('run_time', '[ TIME:   %(run_time)6.0fms ]'),
        ('user_info', '[ USER:%(user_info)-9s ]'),
        ('requester_ip', '[ CALL_BY:%(requester_ip)-15s ]'),
        ('authentication_status', '[ AUTHENTICATED:%(authentication_status)-6s ]'),
        ('requester_module', '[ BY:%(requester_module)s ]'),
        ('auth_module', '[ AUTHENTICATED BY THE MODULE:%(auth_module)s ]'),
        ('error', '[ ERROR:%(error)-9s ]'),
    ]
    
    
    def login(self, user_uuid, user_name, requester_ip, run_time, result_code=200, requester_module=None, auth_module=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode], Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
            auth_module=auth_module,
        )
        
        log_data.update({
            'type_request'         : 'LOGIN',
            'authentication_status': 'OK',
        })
        
        self._log(self._build_message(log_data, self.FORMATS))
    
    
    def login_failed(self, user_uuid, user_name, requester_ip, run_time, result_code=401, requester_module=None, auth_module=None, error=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode], Optional[unicode], Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
            auth_module=auth_module,
            error=error,
        )
        
        log_data.update({
            'type_request'         : 'LOGIN',
            'authentication_status': 'FAILED',
        })
        
        self._log(self._build_message(log_data, self.FORMATS))
    
    
    def logout(self, user_uuid, user_name, requester_ip, run_time, result_code=200, requester_module=None):
        # type: (unicode, unicode, unicode, float, int, Optional[unicode]) -> None
        if not self.log_enable or not self._logger:
            return
        log_data = self._build_common_log_data(
            user_uuid=user_uuid,
            user_name=user_name,
            requester_ip=requester_ip,
            run_time=run_time,
            result_code=result_code,
            requester_module=requester_module,
        )
        
        log_data.update({
            'type_request': 'LOGOUT',
        })
        
        self._log(self._build_message(log_data, self.FORMATS))
