#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# From https://gist.github.com/lemiant/10660092

import threading
# from shinken.log import logger
# import inspect

try:
    from shinken.synchronizer.dao.transactions.transactions import get_transaction_object
except ImportError:
    try:
        # for TU
        from synchronizer.dao.transactions.transactions import get_transaction_object
    except ImportError:
        print('not in a synchronizer install')
        get_transaction_object = lambda: ''


def read_sync(func):
    """ Reader decorator. """
    
    
    def wrapper(self, *args, **kwargs):
        self.rw_lock.acquire_read()
        try:
            return func(self, *args, **kwargs)
        finally:
            self.rw_lock.release_read()
    
    
    return wrapper


def write_sync(func):
    """ Writer decorator. """
    
    
    def wrapper(self, *args, **kwargs):
        self.rw_lock.acquire_write()
        try:
            return func(self, *args, **kwargs)
        finally:
            self.rw_lock.release_write()
    
    
    return wrapper


class ReadWriteLock(object):
    """ An implementation of a read-write lock for Python.
    Any number of readers can work simultaneously but they
    are mutually exclusive with any writers (which can
    only have one at a time).

    This implementation is reader biased. This can be harmful
    under heavy load because it can starve writers.
    However under light load it will be quite performant since
    reads are usually much less resource intensive than writes,
    and because it maximizes concurrency.
    """
    
    
    def __init__(self):
        self.__reader_lock = threading.Lock()
        self.__write_lock = threading.Lock()
        self.readers = 0
    
    
    def acquire_read(self):
        in_transaction = bool(get_transaction_object())
        # caller_name = inspect.stack()[3][3]
        # thread_id = threading.current_thread().ident
        # logger.debug('[LOCK][%s]%s Ask Acquire Read Lock [%s]' % (thread_id, '[in_transaction]' if in_transaction else '', caller_name))
        if in_transaction:
            # logger.debug('[LOCK][%s][in_transaction] Read Lock Acquire !  [%s]' % (thread_id, caller_name))
            return
        with self.__reader_lock:
            # logger.debug('[LOCK][%s] Read Lock Acquire ! [%s]' % (thread_id, caller_name))
            self.readers += 1
            if self.readers == 1:
                self.__write_lock.acquire()
                # logger.debug('[LOCK][%s] R Write Lock Acquire ! [%s]' % (thread_id, caller_name))
    
    
    def release_read(self):
        in_transaction = bool(get_transaction_object())
        # thread_id = threading.current_thread().ident
        # caller_name = inspect.stack()[3][3]
        # logger.debug('[LOCK][%s]%s Ask Released Read Lock [%s]' % (thread_id, '[in_transaction]' if in_transaction else '', caller_name))
        if in_transaction:
            # logger.debug('[LOCK][%s][in_transaction] Read Lock Released !  [%s]' % (thread_id, caller_name))
            return
        with self.__reader_lock:
            self.readers -= 1
            # logger.debug('[LOCK][%s] Read Lock Released ! [%s]' % (thread_id, caller_name))
            if self.readers == 0:
                self.__write_lock.release()
                # logger.debug('[LOCK][%s] R Write Lock Released ! [%s]' % (thread_id, caller_name))
    
    
    def acquire_write(self):
        in_transaction = bool(get_transaction_object())
        # caller_name = inspect.stack()[3][3]
        # thread_id = threading.current_thread().ident
        # logger.debug('[LOCK][%s]%s Ask Acquire Write Lock [%s]' % (thread_id, '[in_transaction]' if in_transaction else '', caller_name))
        if in_transaction:
            # logger.debug('[LOCK][%s][in_transaction] Write Lock Acquire ! [%s]' % (thread_id, caller_name))
            return
        self.__write_lock.acquire()
        # logger.debug('[LOCK][%s] Write Lock Acquired ! [%s]' % (thread_id, caller_name))
    
    
    def release_write(self):
        in_transaction = bool(get_transaction_object())
        # thread_id = threading.current_thread().ident
        # caller_name = inspect.stack()[3][3]
        # logger.debug('[LOCK][%s]%s Ask Released Write Lock [%s]' % (thread_id, '[in_transaction]' if in_transaction else '', caller_name))
        if in_transaction:
            # logger.debug('[LOCK][%s][in_transaction] Write Lock Released ! [%s]' % (thread_id, caller_name))
            return
        self.__write_lock.release()
        # logger.debug('[LOCK][%s] Write Lock Released ! [%s]' % (thread_id, caller_name))
    
    
    def open_transaction(self):
        # caller_name = inspect.stack()[3][3]
        # thread_id = threading.current_thread().ident
        # logger.debug('[LOCK][%s][in_transaction] Ask Acquire Write Transaction [%s]' % (thread_id, caller_name))
        self.__write_lock.acquire()
        # logger.debug('[LOCK][%s][in_transaction] Write Transaction Acquired [%s]' % (thread_id, caller_name))
    
    
    def close_transaction(self):
        # caller_name = inspect.stack()[3][3]
        # thread_id = threading.current_thread().ident
        # logger.debug('[LOCK][%s][in_transaction] Ask Released Write Transaction [%s]' % (thread_id, caller_name))
        self.__write_lock.release()
        # logger.debug('[LOCK][%s][in_transaction] Released Write Transaction [%s]' % (thread_id, caller_name))
