#!/usr/bin/python-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import sys

from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import TYPE_CHECKING
from .migrations.migrator import WeatherJSONMigrator
from ..version import WEATHER_VERSION

sys.path.append('/var/lib/shinken/modules')

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional
    from shinken.log import PartLogger


# WEATHER CONSTANTS ARE FORBIDDEN HERE

def get_weather_format_version(weather_data):
    # type: (Dict) -> Optional[int]
    weather_version = weather_data.get('identification', {}).get('weather_format_version', None)  # type: Optional[int]
    if weather_version:
        weather_version = int(weather_version)
    return weather_version


def get_weather_uuid(weather_data):
    # type: (Dict) -> Optional[unicode]
    return weather_data.get('identification', {}).get('weather_uuid', None)


def update_weather_format(weather_data, logger, weather_id=None):
    # type: (Dict, PartLogger, Optional[unicode]) -> Optional[Dict]
    if weather_id is None:
        weather_id = get_weather_uuid(weather_data)
    weather_version = get_weather_format_version(weather_data)
    if weather_version is None:
        logger.error('Weather [ %s ] does not have a weather_format_version. Sanitize cannot migrate json of this weather.' % weather_id)
        return None
    if weather_version >= WEATHER_VERSION.WEATHER_FORMAT_VERSION:
        logger.info('Weather [ %s ] is up-to-date' % weather_id)
        return None
    logger.info('Starting migration of the old weather [ %s ] ( version %d to version %d )' % (weather_id, weather_version, WEATHER_VERSION.WEATHER_FORMAT_VERSION))
    
    migrator = WeatherJSONMigrator()
    migrated_weather = fast_deepcopy(weather_data)  # Do not alter given weather (for safety)
    for target_version in range(weather_version + 1, WEATHER_VERSION.WEATHER_FORMAT_VERSION + 1):  # WEATHER_FORMAT_VERSION + 1 because WEATHER_FORMAT_VERSION must be included in loop
        migrator.apply_migrations(migrated_weather, target_version, logger=logger)
    
    logger.info('Migration of the old weather [ %s ] done ( version %d to version %d )' % (weather_id, weather_version, WEATHER_VERSION.WEATHER_FORMAT_VERSION))
    return migrated_weather
