#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import time
import uuid

from shinken.log import PartLogger
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from .misc.type_hint import Optional, Dict, List

# Purely random uuid4 for regenerator that do not have currently a configuration to manage
DEFAULT_CONFIGURATION_UUID = '6dbc00f706ca4a3b9a60a843a370a634'


class SCHEDULER_MANAGED_CONFIGURATION_KEYS:
    SHARD_ID = 'shard_id'
    CONFIGURATION_INCARNATION_DUMP = 'configuration_incarnation_dump'
    SCHEDULER_NAME = 'scheduler_name'
    IS_ACTIVE = 'is_active'


class ConfigurationIncarnation:
    def __init__(self):
        self._uuid = uuid.uuid4().hex
        self._creation_date = int(time.time())
        self._author = '(unknown)'
        self._architecture_name = '(unknown)'
    
    
    def set_from_json_obj(self, dict_obj):
        _uuid = dict_obj.get('uuid', None)
        creation_date = dict_obj.get('creation_date', None)
        author = dict_obj.get('author', None)
        architecture_name = dict_obj.get('architecture_name', None)
        
        # NOTE: Architecture name is for display only currently, but will be mandatory in the future (after 02.07.06++)
        if _uuid is None or creation_date is None or author is None:
            raise ValueError('Cannot create a ConfigurationIncarnation because value is missing')
        
        self._uuid = _uuid
        self._creation_date = creation_date
        self._author = author
        self._architecture_name = architecture_name
        if self._architecture_name is None:
            self._architecture_name = '(unknown)'
    
    
    def set_author(self, author):
        self._author = author
    
    
    def get_author(self):
        return self._author
    
    
    def set_architecture_name(self, architecture_name):
        self._architecture_name = architecture_name
    
    
    def get_architecture_name(self):
        # Maybe we are given an old architecture name
        # TODO: remove after 02.08.02 (included)
        return getattr(self, '_architecture_name', '(unknown)')
    
    
    def get_uuid(self):
        return self._uuid
    
    
    def get_creation_date(self):
        return self._creation_date
    
    
    def reset_creation_date(self):
        self._creation_date = int(time.time())
    
    
    def __str__(self):
        return '[configuration_uuid=%s, arbiter=%s, architecture=%s, date=%s]' % (self._uuid, self._author, self.get_architecture_name(), time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._creation_date)))
    
    
    def __repr__(self):
        return self.__str__()
    
    
    def __eq__(self, other) -> bool:
        try:
            return self.get_uuid() == other.get_uuid()
        except:
            return False
    
    
    def build_log_message(self):
        return 'Monitoring incarnation 〖%s〗 given by arbiter:〖%s〗 at:〖%s〗' % (
            self.get_uuid(),
            self.get_author(),
            PartLogger.format_time(self.get_creation_date())
        )
    
    
    # Is a configuration_incarnation is the same as another?
    def is_equal(self, other_configuration_incarnation: 'Optional[ConfigurationIncarnation]') -> bool:
        if other_configuration_incarnation is None:
            return False
        
        # uuid should be the same, if so, other are the same too
        return self._uuid == other_configuration_incarnation.get_uuid()
    
    
    def dump_as_json(self):
        dict_obj = {
            'uuid'             : self._uuid,
            'creation_date'    : self._creation_date,
            'author'           : self._author,
            'architecture_name': self.get_architecture_name(),
        }
        return dict_obj
    
    
    @staticmethod
    def create_from_json(dict_obj: 'Dict') -> 'ConfigurationIncarnation|None':
        configuration_incarnation = ConfigurationIncarnation()
        try:
            configuration_incarnation.set_from_json_obj(dict_obj)
        except ValueError:  # missing value
            return None
        return configuration_incarnation
    
    
    def is_newer_than(self, configuration_incarnation: 'ConfigurationIncarnation') -> bool:
        return self.get_creation_date() > configuration_incarnation.get_creation_date()


class PartConfigurationIncarnation:
    def __init__(self, configuration_incarnation: 'ConfigurationIncarnation', shard_id: int, scheduler_name: str, is_active: bool = True) -> None:
        self._configuration_incarnation = configuration_incarnation
        self._shard_id = shard_id
        self.scheduler_name = scheduler_name
        self._is_active = is_active  # know if the scheduler manage a real shard and is not a sleeping spare
    
    
    def __str__(self):
        # Shard with id = 0 means idle spare
        shard_str = '-none-' if self._shard_id == 0 else self._shard_id
        return '[shard_id=%s, scheduler=%s, configuration_uuid=%s, arbiter=%s, architecture=%s, date=%s, active=%s]' % (
            shard_str, self.scheduler_name, self._configuration_incarnation.get_uuid(), self._configuration_incarnation.get_author(),
            self._configuration_incarnation.get_architecture_name(), time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._configuration_incarnation.get_creation_date())), self._is_active)
    
    
    def __repr__(self):
        return self.__str__()
    
    
    def __eq__(self, other):
        try:
            return self._shard_id == other.get_part_id() and self.get_uuid() == other.get_uuid()
        except:
            return False
    
    
    def build_log_message(self):
        return 'Monitoring configuration part 〖%s〗-〖%s〗 from scheduler:〖%s〗 provided by arbiter:〖%s〗 at:〖%s〗' % (
            self._shard_id,
            self.get_uuid(),
            self.scheduler_name,
            self.get_author(),
            PartLogger.format_time(self._configuration_incarnation.get_creation_date())
        )
    
    
    def get_configuration_incarnation(self):
        return self._configuration_incarnation
    
    
    def get_uuid(self):
        return self._configuration_incarnation.get_uuid()
    
    
    def get_creation_date(self):
        return self._configuration_incarnation.get_creation_date()
    
    
    def get_author(self):
        return self._configuration_incarnation.get_author()
    
    
    def get_part_id(self):
        return self._shard_id
    
    
    def get_architecture_name(self):
        return self._configuration_incarnation.get_architecture_name()
    
    
    def dump_as_json(self):
        dict_obj = {
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID                      : self._shard_id,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.SCHEDULER_NAME                : self.scheduler_name,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.IS_ACTIVE                     : self._is_active,
            SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIGURATION_INCARNATION_DUMP: self._configuration_incarnation.dump_as_json(),
        }
        return dict_obj
    
    
    @staticmethod
    def create_from_json(dict_obj: 'Dict') -> 'PartConfigurationIncarnation|None':
        configuration_incarnation_dump = dict_obj.get(SCHEDULER_MANAGED_CONFIGURATION_KEYS.CONFIGURATION_INCARNATION_DUMP, {})
        try:
            configuration_incarnation = ConfigurationIncarnation()
            configuration_incarnation.set_from_json_obj(configuration_incarnation_dump)
        except ValueError:  # missing value
            return None
        shard_id = dict_obj.get(SCHEDULER_MANAGED_CONFIGURATION_KEYS.SHARD_ID, None)
        scheduler_name = dict_obj.get(SCHEDULER_MANAGED_CONFIGURATION_KEYS.SCHEDULER_NAME, None)
        is_active = dict_obj.get(SCHEDULER_MANAGED_CONFIGURATION_KEYS.IS_ACTIVE, None)
        if shard_id is None or scheduler_name is None or is_active is None:
            return None
        partial_configuration_incarnation = PartConfigurationIncarnation(configuration_incarnation, shard_id, scheduler_name, is_active=is_active)
        return partial_configuration_incarnation
    
    
    def is_newer_or_equal_than(self, _date: int) -> bool:
        return self._configuration_incarnation.get_creation_date() >= _date
    
    
    def is_newer_than(self, _date: int) -> bool:
        return self._configuration_incarnation.get_creation_date() > _date


class PartConfigurationIncarnationContainer:
    
    def __init__(self):
        self._parts_loaded: 'Dict[int, PartConfigurationIncarnation]' = {}
        self._configuration_loaded_uuid = DEFAULT_CONFIGURATION_UUID
        self._configuration_loaded_date = 0
        self.last_part_configuration_incarnation_loaded: 'Optional[PartConfigurationIncarnation]' = None
        self.last_part_configuration_incarnation_received: 'Optional[PartConfigurationIncarnation]' = None
        self.last_configuration_incarnation_received: 'ConfigurationIncarnation|None' = None
    
    
    def add_part_loaded(self, part_configuration_incarnation: 'PartConfigurationIncarnation') -> None:
        self._parts_loaded[part_configuration_incarnation.get_part_id()] = part_configuration_incarnation
        self.last_part_configuration_incarnation_loaded = part_configuration_incarnation
        self._configuration_loaded_uuid = part_configuration_incarnation.get_uuid()
        self._configuration_loaded_date = part_configuration_incarnation.get_creation_date()
    
    
    def add_part_received(self, part_configuration_incarnation: 'PartConfigurationIncarnation') -> None:
        self.last_part_configuration_incarnation_received = part_configuration_incarnation
        
        configuration_incarnation = part_configuration_incarnation.get_configuration_incarnation()
        if self.last_configuration_incarnation_received is None or configuration_incarnation.is_newer_than(self.last_configuration_incarnation_received):
            self.last_configuration_incarnation_received = configuration_incarnation
    
    
    def check_configuration_incarnation_came_from_last_configuration_incarnation_received(self, configuration_incarnation: 'ConfigurationIncarnation') -> bool:
        # If we never received a configuration_incarnation, we allow the new configuration
        return self.last_configuration_incarnation_received is None or self.last_configuration_incarnation_received == configuration_incarnation
    
    
    def check_part_is_loaded(self, part_configuration_incarnation: 'PartConfigurationIncarnation') -> 'tuple[bool, str]':
        current_part_configuration_incarnation = self._parts_loaded.get(part_configuration_incarnation.get_part_id(), None)
        
        i_have_this_part = current_part_configuration_incarnation is not None
        not_already_handle = not i_have_this_part or (i_have_this_part and part_configuration_incarnation.get_uuid() != current_part_configuration_incarnation.get_uuid())
        if_already_have_this_part__new_conf_must_be_newer_than_the_one_i_have = i_have_this_part and part_configuration_incarnation.is_newer_than(current_part_configuration_incarnation.get_creation_date())
        if_do_not_have_this_part__new_conf_must_be_newer_or_same_date = not i_have_this_part and part_configuration_incarnation.is_newer_or_equal_than(self._configuration_loaded_date)
        
        if not not_already_handle:
            cause = 'I already loaded it %s' % part_configuration_incarnation
        elif i_have_this_part:
            cause = 'The configuration sent is older than the one we have. Current loaded configuration %s. Sent configuration : %s' % (current_part_configuration_incarnation, part_configuration_incarnation)
        else:
            cause = 'The configuration sent is too old. Current loaded configuration %s. Sent configuration : %s' % (time.strftime('%d-%m-%Y %H:%M:%S', time.localtime(self._configuration_loaded_date)), part_configuration_incarnation)
        
        return not_already_handle and (if_already_have_this_part__new_conf_must_be_newer_than_the_one_i_have or if_do_not_have_this_part__new_conf_must_be_newer_or_same_date), cause
    
    
    def get_configuration_incarnation_part_from_loaded_configuration(self) -> 'List[PartConfigurationIncarnation]':
        return [i for i in self._parts_loaded.values() if i.get_uuid() == self._configuration_loaded_uuid]
    
    
    def get_configuration_loaded_uuid(self):
        return self._configuration_loaded_uuid
    
    
    def get_part_configuration_incarnation_loaded(self, part_id):
        return self._parts_loaded.get(part_id, None)
