#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import get_window_of_a_sorted_list_with_weights

from .inventoryhasher import get_hasher
from .itemsummary import HostSummary

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Union
    from .itemsummary import CheckSummary


class ElementsInventory:
    def __init__(self):
        self._reset()
    
    
    def __iter__(self):
        return iter(self._inventory.values())
    
    
    def _reset(self):
        self._inventory = {}
        self._hash = None
    
    
    def have_host_uuid(self, host_uuid):
        return host_uuid in self._inventory
    
    
    def have_item_uuid(self, item_uuid):
        host_uuid = item_uuid
        check_uuid = ''
        if '-' in item_uuid:
            host_uuid = item_uuid.split('-')[0]
            check_uuid = item_uuid.split('-')[1]
        if not self.have_host_uuid(host_uuid):
            return False
        elif check_uuid:
            host = self.get_host(item_uuid)
            if check_uuid not in host.get_checks():
                return False
        
        return True
    
    
    def get_all_hosts(self):
        return self._inventory
    
    
    def get_all(self):
        # type: () -> List[Union[HostSummary, CheckSummary]]
        _all = []
        for host in self._inventory.values():
            _all.append(host)
            for check in host.get_checks().values():
                _all.append(check)
        return _all
    
    
    def get_host(self, host_uuid):
        host = self._inventory[host_uuid]
        return host
    
    
    def get_host_or_check(self, uuid):
        # type: (str) -> Union[HostSummary,CheckSummary]
        if '-' in uuid:
            host_uuid = uuid.split('-')[0]
            return self._inventory[host_uuid].get_checks()[uuid]
        else:
            return self._inventory[uuid]
    
    
    def get_host_uuids_and_hashes(self):
        r = {}
        for (host_uuid, host) in self._inventory.items():
            r[host_uuid] = host.get_hash()
        return r
    
    
    def get_hosts_and_checks_uuids_and_hashes(self):
        r = {}
        for (host_uuid, host) in self._inventory.items():
            r[host_uuid] = host.get_hash()
            for (check_uuid, check) in host.get_checks().items():
                r[check_uuid] = check.get_hash()
        return r
    
    
    def get_hosts_uuids(self):
        return list(self._inventory.keys())
    
    
    def update_from_sharded_inventory(self, sharded_inventory):
        for host_in_receiver in sharded_inventory:
            self.add_host_in_receiver(host_in_receiver)
    
    
    def compute_hash(self):
        hasher = get_hasher()
        # IMPORTANT: always be sure to have sorted hosts, if not, it's random for each restart...
        sorted_host_uuids = sorted(list(self._inventory.keys()))
        for host_uuid in sorted_host_uuids:
            host_entry = self._inventory[host_uuid]
            hasher.update(host_entry.get_hash())  # NOTE: the host hash was already computed by a sub process
        self._hash = hasher.hexdigest()
    
    
    def get_hash(self):
        if self._hash is None:
            self.compute_hash()
        return self._hash
    
    
    def get_len(self):
        return len(self._inventory)
    
    
    def add_host_in_receiver(self, host_in_receiver):
        self._inventory[host_in_receiver.get_uuid()] = host_in_receiver
        # Hash is now invalid
        self._hash = None
    
    
    def _del_host_in_receiver(self, host_in_receiver):
        if host_in_receiver.get_uuid() in self._inventory:
            del self._inventory[host_in_receiver.get_uuid()]
            # Hash is now invalid
            self._hash = None
    
    
    def _del_host_uuid_in_receiver(self, host_uuid):
        if host_uuid in self._inventory:
            del self._inventory[host_uuid]
            # Hash is now invalid
            self._hash = None
    
    
    # In the inventory, we will need to have information about:
    # * host & checks name & uuids
    # * templates & data templates but only for the ones need by the receivers
    # (brokers don't care about templates or DATA)
    def create_from_sharded_configuration(self, shard_config, inventory_filters):
        self._reset()  # be sure to be void
        # Now look for all hosts if their templates, and if the host templates match the
        # receivers' data filtering, take the template DATA from the host and put them into the inventory
        for host in shard_config.hosts:
            # We only want to have realm hosts here
            if host.is_tpl():
                continue
            # If there are filters, respect them
            # * with filters: for receivers
            # * without filters: for brokers (need all)
            if inventory_filters is not None:
                # Try to see if the host templates match any of the filter
                # NOTE: yes I know it's possible to one-line it with 2 any(). But No. Just no.
                founded = False
                for tpl in host.templates:
                    tpl_name = tpl.get_name()
                    for inventory_filter in inventory_filters:
                        if inventory_filter.template_match_template_filter(tpl_name):
                            founded = True
                            break
                    if founded:
                        break
                # The host do not have any template for receivers
                if not founded:
                    continue
            # Ok it matches at least one receiver
            host_in_receiver = HostSummary()
            host_in_receiver.create_from_host(host, inventory_filters)
            self.add_host_in_receiver(host_in_receiver)
        
        # NOTE: This inventory is used to exchange between worker-> main process and this last one
        # will merge several of inventory, so we do not need to compute inventory hash
        # as the main process will have to re-compute it for all hosts (hosts with valid order!)
    
    
    # The arbiter is asking us a new inventory that is an extract of us with only a shard of hosts
    def get_new_inventory_from_us_and_take_only_a_shard_part(self, previous_elements_weight_sum, receiver_weight, sum_of_receiver_weights, receiver_index, receiver_weights):
        all_host_uuids = list(self._inventory.keys())
        
        # NOTE: receiver_shard_idx starts at 0
        receiver_only_host_uuids = get_window_of_a_sorted_list_with_weights(all_host_uuids, previous_elements_weight_sum, receiver_weight, sum_of_receiver_weights, receiver_index, receiver_weights)
        
        extracted_inventory = ElementsInventory()
        
        for host_uuid in receiver_only_host_uuids:
            host_in_receiver = self._inventory[host_uuid]
            extracted_inventory.add_host_in_receiver(host_in_receiver)
        
        # This inventory need a hash as the receiver can maybe have it, and we need to be sure before resent it
        extracted_inventory.compute_hash()
        return extracted_inventory
    
    
    # In the receiver we need to have the new/updated/deleted list between the newer inventory and us (we own the old one)
    def compute_differences_from_new_inventory(self, newer_inventory, only_hosts=True):
        differences = {'new': {}, 'changed': {}, 'unchanged': {}, 'deleted': {}}
        if only_hosts:
            newer_elements = newer_inventory.get_host_uuids_and_hashes()
            older_elements = self.get_host_uuids_and_hashes()
            get_item_from_new_inventory = newer_inventory.get_host
        else:
            newer_elements = newer_inventory.get_hosts_and_checks_uuids_and_hashes()
            older_elements = self.get_hosts_and_checks_uuids_and_hashes()
            get_item_from_new_inventory = newer_inventory.get_host_or_check
        
        newer_elements_set = set(newer_elements.keys())
        older_elements_set = set(older_elements.keys())
        
        new_uuids = newer_elements_set - older_elements_set
        deleted_uuids = older_elements_set - newer_elements_set
        in_both_uuids = newer_elements_set.intersection(older_elements_set)
        
        logger.debug('Computing DIFFERENCES:')
        logger.debug('NEWS: %s  DELETED: %s  in_both: %s' % (new_uuids, deleted_uuids, in_both_uuids))
        # Save both new and changed objects, and only the uuids of deleted
        for element_uuid in new_uuids:
            differences['new'][element_uuid] = get_item_from_new_inventory(element_uuid)
        for element_uuid in in_both_uuids:
            status = 'unchanged' if older_elements[element_uuid] == newer_elements[element_uuid] else 'changed'
            differences[status][element_uuid] = get_item_from_new_inventory(element_uuid)
        differences['deleted'] = list(deleted_uuids)
        return differences
    
    
    # The other inventory did create us a difference, apply it
    def update_from_differences(self, differences):
        for (host_uuid, host_in_receiver) in differences['new'].items():
            self.add_host_in_receiver(host_in_receiver)
        
        for (host_uuid, host_in_receiver) in differences['changed'].items():
            self.add_host_in_receiver(host_in_receiver)
        
        for host_uuid in differences['deleted']:
            self._del_host_uuid_in_receiver(host_uuid)


# In the receiver module, we have a unique ElementInventory that will
# in fact consume classic ElementsInventory that are limited to one realm only
# So this one just have methods to load/delete from other ElementInventory
class ElementsInventoryInReceiverModule(ElementsInventory):
    def add_elements_from_realm_inventory(self, realm_name, realm_inventory):
        logger.info('Adding %s new elements from the realm %s' % (realm_inventory.get_len(), realm_name))
        for host_in_receiver in realm_inventory:
            logger.debug('Adding new host: %s' % host_in_receiver.get_name())
            self.add_host_in_receiver(host_in_receiver)
    
    
    def delete_elements_from_realm_inventory(self, realm_name, realm_inventory):
        logger.info('We are removing %s new elements from the realm %s' % (realm_inventory.get_len(), realm_name))
        for host_in_receiver in realm_inventory:
            logger.debug('Deleting host from realm: %s' % host_in_receiver.get_name())
            self._del_host_in_receiver(host_in_receiver)
    
    
    def update_elements_from_realm_inventory(self, realm_name, realm_inventory):
        logger.info('We are updating %s elements from the realm %s' % (realm_inventory.get_len(), realm_name))
        for host_in_receiver in realm_inventory:
            logger.debug('Updating host: %s' % host_in_receiver.get_name())
            self.add_host_in_receiver(host_in_receiver)


# In the receiver module, we have a unique ElementInventory that will
# in fact consume classic ElementsInventory that are limited to one realm only
# So this one just have methods to load/delete from other ElementInventory
class ElementsInventoryInBrokerModule(ElementsInventory):
    def add_elements_from_realm_inventory(self, realm_name, realm_inventory):
        logger.info('Adding %s new elements from the realm %s' % (realm_inventory.get_len(), realm_name))
        for host_in_receiver in realm_inventory:
            logger.debug('Adding new host: %s' % host_in_receiver.get_name())
            self.add_host_in_receiver(host_in_receiver)
    
    
    def delete_elements_from_realm_inventory(self, realm_name, realm_inventory):
        logger.info('We are removing %s new elements from the realm %s' % (realm_inventory.get_len(), realm_name))
        for host_in_receiver in realm_inventory:
            logger.debug('Deleting host from realm: %s' % host_in_receiver.get_name())
            self._del_host_in_receiver(host_in_receiver)
    
    
    def update_elements_from_realm_inventory(self, realm_name, realm_inventory):
        logger.info('We are updating %s elements from the realm %s' % (realm_inventory.get_len(), realm_name))
        for host_in_receiver in realm_inventory:
            logger.debug('Updating host: %s' % host_in_receiver.get_name())
            self.add_host_in_receiver(host_in_receiver)
