#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.


from .satellitelink import SatelliteLinks
from .property import BoolProp, IntegerProp, StringProp, ListProp
from .http_client import HTTPExceptions
from .withinventorysatellitelink import WithInventorySatelliteLink


# IMPORTANT: When adding a new property, think about the retention in the arbiter SPARE!
class ReceiverFilter:
    def __init__(self, filter_template, data_templates):
        self._filter_template = filter_template
        self._data_templates = set(data_templates)
    
    
    def template_match_data_filter(self, template_name):
        return template_name in self._data_templates
    
    
    def template_match_template_filter(self, template_name):
        return template_name == self._filter_template


class ReceiverLink(WithInventorySatelliteLink):
    """Please Add a Docstring to describe the class here"""
    
    id = 0
    my_type = 'receiver'
    properties = WithInventorySatelliteLink.properties.copy()
    properties.update({
        'receiver_name'                          : StringProp(fill_brok=['full_status'], to_send=True),
        'port'                                   : IntegerProp(default='7772', fill_brok=['full_status']),
        'manage_sub_realms'                      : BoolProp(default='1', fill_brok=['full_status']),
        'direct_routing'                         : BoolProp(default='0', fill_brok=['full_status'], to_send=True),
        
        'elements_sharding_enabled'              : BoolProp(default='0', to_send=True),
        'elements_sharding_filter_by_template'   : StringProp(default='', to_send=True),
        'elements_sharding_add_data_of_templates': ListProp(default='', to_send=True),
        'elements_sharding_weight'               : IntegerProp(default='1', to_send=True),
        
    })
    
    
    def __init__(self, *args, **kwargs):
        super(ReceiverLink, self).__init__(*args, **kwargs)
    
    
    def get_name(self):
        return getattr(self, 'receiver_name', 'unknown')
    
    
    def register_to_my_realm(self):
        self.realm.receivers.append(self)
    
    
    # We will loop over all realms and all their shards, and look if the
    def assert_valid_host_mapping(self):
        # Only direct routing receivers need to have host name mapping
        if not self.direct_routing:
            return
        # If we are dead, skip this part, we cannot known
        if not self.is_alive():
            return
        
        for realm in self.get_all_my_managed_realms():
            for (shard_id, shard) in realm.shards.items():
                # If the shard id not assigned, we cant' do anything about it
                if not shard.is_assigned:
                    continue
                # Check it
                if not self.have_host_names_mapping(shard_id):
                    shard_host_names = shard.get_managed_host_names()  # get the list of the host names (not uuid) that this shard manage
                    self.logger_dispatch.info("HOSTS TO SCHEDULER MAPPING Sending %s hostnames from the shard=%-4d to the receiver" % (len(shard_host_names), shard.get_id()))
                    self.push_host_names(shard_id, shard_host_names)
    
    
    def push_host_names(self, shard_id, hnames):
        try:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            self.con.get('ping')
            # NOTE: maybe the receiver is not already up to date for it's configuration, so also send the current configuration uuid
            self.con.post('push_host_names', {'configuration_incarnation_uuid': self.configuration_incarnation.get_uuid(),
                                              'shard_id'                      : shard_id,
                                              'hnames'                        : hnames}, wait='long')
        except HTTPExceptions as exp:
            self.logger_dispatch.error('Cannot call /push_host_names : %s' % (exp))
            # Other errors will be managed by the ping call (like DEAD, and so on)
    
    
    def have_host_names_mapping(self, shard_id):
        try:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return False
            
            self.con.get('ping')
            have_it = self.con.get('have_host_names_mapping', {'configuration_incarnation_uuid': self.configuration_incarnation.get_uuid(), 'shard_id': shard_id})
            return have_it
        except HTTPExceptions as exp:
            self.logger_dispatch.error('Cannot call /have_host_names_mapping : %s' % (exp))
            return False
    
    
    # We give a filter object with our data but only if we are a manage sub element receiver
    def get_inventory_templates_filter(self):
        if not self.elements_sharding_enabled:
            return None
        return ReceiverFilter(self.elements_sharding_filter_by_template, self.elements_sharding_add_data_of_templates)


class ReceiverLinks(SatelliteLinks):
    """Please Add a Docstring to describe the class here"""
    
    name_property = "receiver_name"
    inner_class = ReceiverLink
    
    
    # Receivers can have properties about interesting host templates
    # - elements_sharding_enabled = 1          => should give/have templates
    # - elements_sharding_add_data_of_templates            => will have only the host with this template
    # - elements_sharding_weight => hosts will have data about theses templates only
    def get_inventory_filters_by_receivers(self):
        res = []
        for receiver in self:
            receiver_filter = receiver.get_inventory_templates_filter()
            if receiver_filter:
                res.append(receiver_filter)
        return res
