#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import operator
import re
import threading
import time
import uuid
from collections import OrderedDict
from collections.abc import Iterable

import shinkensolutions.shinkenjson as json
from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.webui import bottlewebui as bottle
from shinken.webui.bottlewebui import request, response, parse_auth, HTTPResponse
from shinken.webui.cherrypybackend import CherryPyServerHTTP
from shinkensolutions.api.synchronizer import ITEM_STATE, ITEM_TYPE, NOT_TO_LOOK, DEF_ITEMS, METADATA, DataProviderMongo, get_type_item_from_class, get_name_from_type, ComponentManagerSynchronizer
from shinkensolutions.api.synchronizer.source.abstract_module.listener_module import ListenerModule
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration, read_string_in_configuration, read_bool_in_configuration
from shinkensolutions.system_tools import get_linux_local_addresses

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict, Any, Callable, List, Tuple
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from shinken.log import PartLogger
    
    ListenerRawItem = dict[str, str | float]


# This module will open an HTTP service, where a user can send a host (for now)
class BaseRESTListener(ListenerModule):
    url_base_path = '/shinken/listener-rest'
    
    _configuration_fields = None
    
    
    def __init__(self, configuration):
        # type: (ShinkenModuleDefinition) -> None
        super(BaseRESTListener, self).__init__(configuration)
        self.srv = None
        self.srv_lock = threading.RLock()
        self.provider_mongo = None
        self.thread = None
        self.server_opts = {}
        self.logger = LoggerFactory.get_logger().get_sub_part(self.get_name())
        self.logger_creation = self.logger.get_sub_part('CREATION REQUEST')
        self.logger_update = self.logger.get_sub_part('UPDATE REQUEST')
        self.logger_delete = self.logger.get_sub_part('DELETE REQUEST')
        self.logger_get = self.logger.get_sub_part('GET REQUEST')
        self.logger_list = self.logger.get_sub_part('LIST REQUEST')
        self.logger_listening = self.logger.get_sub_part('LISTENING')
        self.logger_init = self.logger.get_sub_part('INITIALISATION')
        self.logger_authentication = self.logger.get_sub_part('AUTHENTICATION')
        
        try:
            self.logger_init.debug('Configuration starting ...')
            self.port = read_int_in_configuration(configuration, 'port', '7761')
            self.host = read_string_in_configuration(configuration, 'host', '0.0.0.0')
            self.module_name = read_string_in_configuration(configuration, 'module_name', 'module-listener_test_auto')
            self.enable_cors = read_bool_in_configuration(configuration, 'enable_cors', False)
            if self.module_name.startswith('module-'):
                self.module_name = self.module_name[7:]
            self.logger_init.debug('Configuration done, the configuration summary will be logged as soon as the source_controller is initialized, and the module can access its stocked configuration in mongo')
        except AttributeError:
            if configuration.imported_from != 'internal_source':
                self.logger_init.error('The module is missing a property, check module declaration in %s' % configuration.imported_from)
            else:
                self.logger_init.error('The module is missing a property, please contact Shinken Solutions support')
            
            raise
        except Exception as e:
            self.logger_init.error('Exception : %s' % str(e))
            raise
        
        self.items_lock = threading.RLock()
    
    
    def source_start(self, source_is_enabled):
        # type: (bool) -> None
        if not self._is_auth_configuration_valid():
            self.logger_authentication.warning('Authentication is activated, but username or password parameters are not defined. Please check their values or deactivate authentication.')
        self.logger_init.info('Configuration is OK :')
        self.log_configuration(self.logger_init)
        listening_message = 'The source is enabled, it will start listen.' if source_is_enabled else 'The source is disabled, it will not listen until it is enabled.'
        
        self.logger_init.info(listening_message)
        
        if source_is_enabled:
            self.start_listener()
    
    
    def _is_auth_configuration_valid(self):
        # type: () -> bool
        conf = self.get_my_configuration()
        if not conf.get('configuration', {}).get('authentication'):
            # This means the authentication is not activated, and it's ok
            return True
        auth_conf = conf.get('configuration', {})
        username = auth_conf.get('login', '')
        password = auth_conf.get('password', '')
        # If the authentication is ON, both username and password needs to be activated
        return username and password
    
    
    def log_configuration(self, logger=None):
        if logger is None:
            logger = self.logger.get_sub_part('CONFIGURATION')
        conf = self.get_my_configuration().get('configuration', {})
        logger.info('   - Host --------------------------------------- : %s' % self.host)
        logger.info('   - Port --------------------------------------- : %s' % self.port)
        logger.info('   - Authentication required ( login/password ) - : %s' % ('Yes' if conf.get('authentication') else 'No'))
        logger.info('      - login ----------------------------------- : %s' % ('********' if conf.get('login') else ''))
        logger.info('      - password -------------------------------- : %s' % ('********' if conf.get('password') else ''))
        logger.info('   - SSL required ( HTTPS ) --------------------- : %s' % ('Yes' if conf.get('use_ssl') else 'No'))
        logger.info('      - ssl_cert -------------------------------- : %s' % conf.get('ssl_cert'))
        logger.info('      - ssl_key --------------------------------- : %s' % conf.get('ssl_key'))
        
        listen_on_all_interfaces = self.host == '0.0.0.0'
        
        url = '%(protocol)s://%(host)s:%(port)s/shinken/%(source_name)s/v%(listener_version)s/hosts' % {
            'protocol'        : 'https' if conf.get('use_ssl') else 'http',
            'host'            : 'SYNCHRONIZER_IP' if listen_on_all_interfaces else self.host,
            'port'            : self.port,
            'source_name'     : self.get_name(),  # The source_name is hardcoded in self.url_base_path, must be self.get_name() in future
            'listener_version': '1',  # the listener version is hardcoded in _init_routes :(
        }
        logger.info('The listener URL is : %s' % url)
        if listen_on_all_interfaces:
            all_interfaces = get_linux_local_addresses(logger)
            logger.info('The source listen an all interfaces, SYNCHRONIZER_IP can be one of these address found on this system :')
            for interface in all_interfaces:
                logger.info('   - %s' % interface)
            logger.info('   - 127.0.0.1')  # The local interface is not in the list of address, but it can be called when listen on 0.0.0.0
    
    
    def get_configuration_fields(self):
        if self._configuration_fields is None:
            self._configuration_fields = OrderedDict([
                ('configuration', OrderedDict([
                    ('authentication', {
                        'display_name': self._('analyzer.conf_authentication'),
                        'default'     : False,
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'checkbox',
                        'display_bind': ('login', 'password')
                    }),
                    ('login', {
                        'display_name': self._('analyzer.conf_login'),
                        'default'     : '',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('password', {
                        'display_name': self._('analyzer.conf_password'),
                        'default'     : '',
                        'protected'   : True,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('use_ssl', {
                        'display_name': self._('analyzer.conf_use_ssl'),
                        'default'     : False,
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'checkbox',
                        'display_bind': ('ssl_key', 'ssl_cert')
                    }),
                    ('ssl_key', {
                        'display_name': self._('analyzer.conf_ssl_key'),
                        'default'     : '',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('ssl_cert', {
                        'display_name': self._('analyzer.conf_ssl_cert'),
                        'default'     : '',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                
                ])
                 ),
            ])
        return self._configuration_fields
    
    
    def _init_http(self):
        if self.srv:
            # we need to stop it to restart it right after that that’s why we don’t launch it in a thread
            self.srv.stop()
        try:
            conf = self.get_my_configuration()
        except Exception:
            self.logger.print_stack()
            raise
        
        ssl_conf = conf.get('configuration', {})
        use_ssl = ssl_conf.get('use_ssl', '')
        ssl_key = ssl_conf.get('ssl_key', '')
        ssl_cert = ssl_conf.get('ssl_cert', '')
        try:
            # instantiate a new Bottle object, don't use the default one otherwise all module will share the same
            app = bottle.Bottle()
            app = self._init_routes(app)
            self.srv = app.run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=use_ssl, ssl_key=ssl_key, ssl_cert=ssl_cert, quiet=True, **self.server_opts)
        except Exception as e:
            self.logger.error('Exception : %s' % str(e))
            raise
        self.logger.debug('Server loaded')
    
    
    # set CORS headers for browsers to be happy
    @staticmethod
    def _set_cors_headers():
        response.headers['Access-Control-Allow-Origin'] = '*'
        response.headers['Access-Control-Allow-Methods'] = 'GET, POST, PUT, OPTIONS, DELETE, PATCH'
        response.headers['Access-Control-Allow-Headers'] = 'Origin, Accept, Content-Type, X-Requested-With, X-CSRF-Token, X-Shinken-Token, Authorization'
        response.headers['Access-Control-Allow-Credentials'] = 'true'
    
    
    # As the listener will be called from the UI, we need to add some headers to every route, so we create a wrapper
    def cors_wrapper(self, func: 'Callable'):
        def _wrap(*args, **kwargs):
            
            # the browser need CORS headers to be set
            if self.enable_cors:
                self._set_cors_headers()
            
            # OPTIONS calls are for CORS, and don't need more than void return
            if bottle.request.method == 'OPTIONS':
                return
            
            return func(*args, **kwargs)
        
        
        return _wrap
    
    
    def log_exception_wrapper(self, func: 'Callable') -> 'Callable':
        def _wrap(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except:
                self.logger.print_stack()
                raise
        
        
        return _wrap
    
    
    def _init_routes(self, app: 'bottle.Bottle'):
        
        def base_rest_item(item_type: str):
            response.content_type = 'application/json'
            self._query_check_auth()
            if request.method == 'PUT':
                return self.create_item(item_type)
            elif request.method == 'GET':
                return self.list_items(item_type)
        
        
        def specific_rest_item(item_type: str, item_uuid: str):
            response.content_type = 'application/json'
            self._query_check_auth(item_uuid)
            import_needed = bottle.request.GET.get('import_needed', True)
            if request.method == 'DELETE':
                return self.delete_item(item_uuid, item_type, import_needed)
            elif request.method == 'GET':
                return self.get_item(item_uuid, item_type)
            
            elif request.method in ('POST', 'PUT', 'PATCH'):
                return self.update_item(item_uuid, item_type)
        
        
        base_rest_item = self.log_exception_wrapper(base_rest_item)
        specific_rest_item = self.log_exception_wrapper(specific_rest_item)
        if self.enable_cors:
            base_rest_item = self.cors_wrapper(base_rest_item)
            specific_rest_item = self.cors_wrapper(specific_rest_item)
        
        app.route('%s/v1/:item_type/' % self.url_base_path, callback=base_rest_item, method=('PUT', 'GET', 'OPTIONS'))
        app.route('%s/v1/:item_type' % self.url_base_path, callback=base_rest_item, method=('PUT', 'GET', 'OPTIONS'))
        app.route('%s/v1/:item_type/:item_uuid' % self.url_base_path, callback=specific_rest_item, method=('DELETE', 'GET', 'POST', 'PUT', 'PATCH', 'OPTIONS'))
        return app
    
    
    def _init_data_provider_mongo(self):
        # type: () -> None
        mongo_component = ComponentManagerSynchronizer.get_mongo_component()
        database_cipher = ComponentManagerSynchronizer.get_database_cipher_component()
        self.provider_mongo = DataProviderMongo(mongo_component, database_cipher)
    
    
    def get_dataprovider(self):
        # type: () -> DataProviderMongo
        if self.provider_mongo is None:
            self._init_data_provider_mongo()
        return self.provider_mongo
    
    
    def start_listener(self, reason=None):
        # type: (Optional[str]) -> None
        self.get_dataprovider()
        # We must protect against a user that spam the enable/disable button
        with self.srv_lock:
            # We already did start, skip it
            if self.thread is not None:
                if self.thread.is_alive():
                    return
                
                self.thread.join()
                self.thread = None
            
            self._init_http()
            self._start_thread(reason)
    
    
    def stop_listener(self, reason=None):
        # type: (Optional[str]) -> None
        with self.srv_lock:
            if self.thread is None:
                return
            self.srv.stop()
            self.thread.join(1)
            self.thread = None
            self.logger_listening.info('Closing connections on %s:%s %s' % (self.host, self.port, reason if reason else ''))
    
    
    def _start_thread(self, reason):
        # type: (Optional[str]) -> None
        self.thread = threading.Thread(None, target=self._http_start, name='Listener-REST', args=(reason,))
        self.thread.daemon = True
        self.thread.start()
    
    
    def _http_start(self, reason=None):
        # type: (Optional[str]) -> None
        self.logger_listening.info('Opening connections on %s:%s %s' % (self.host, self.port, reason if reason else ''))
        self.srv.start()
    
    
    # WARNING: do not call it check_auth, or it will be used for Configuration UI auth!
    def _query_check_auth(self, item_uuid=None):
        # type: (Optional[str]) -> None
        conf = self.get_my_configuration()
        if conf.get('configuration', {}).get('authentication'):
            auth_conf = conf.get('configuration', {})
            username = auth_conf.get('login', '')
            password = auth_conf.get('password', '')
            basic = parse_auth(request.environ.get('HTTP_AUTHORIZATION', ''))
            # Maybe the user not even ask for user/pass. If so, bail out
            if not basic:
                error_message = ['Refuse unauthenticated request']
                if item_uuid:
                    error_message.append('for item se_uuid : "%s"' % item_uuid)
                error_message.append(self._get_request_information())
                
                self.logger_authentication.error(' '.join(error_message))
                raise HTTPResponse(self._('listener.error_auth_required'), 401)
            # Maybe he doesn't give the good credential?
            bad_user_name = basic[0] != username
            bad_password = basic[1] != password
            if bad_user_name or bad_password:
                error_message = ['Refuse request']
                if item_uuid:
                    error_message.append('for item se_uuid : "%s"' % item_uuid)
                error_message.append('with bad credentials')
                if bad_user_name:
                    error_message.append('( username used was "%s")' % basic[0])
                error_message.append(self._get_request_information())
                self.logger_authentication.error(' '.join(error_message))
                raise HTTPResponse(self._('listener.error_auth_denied'), 403)
    
    
    def _return_in_400(self, err, logger):
        logger.error('Bad request object received %s : %s' % (self._get_request_information(), err))
        raise HTTPResponse(err, 400)
    
    
    @staticmethod
    def _disallow_duplicate_keys(ordered_pairs: 'List[Tuple[str, Any]]') -> 'ListenerRawItem':
        d = {}
        duplicate = []
        for k, v in ordered_pairs:
            if k in d:
                duplicate.append(k)
            else:
                d[k] = v
        if duplicate:
            raise ValueError('Duplicate keys:%s' % ', '.join(duplicate))
        return d
    
    
    def _get_data(self, logger: 'PartLogger') -> 'ListenerRawItem':
        # Getting lists of information for the commands
        data = request.body.readline()
        if not data:
            self._return_in_400(self._('listener.error_no_data'), logger)
        try:
            try:
                decoded_data = json.loads(data, object_pairs_hook=self._disallow_duplicate_keys)
            except:
                # In python 2.6 object_pairs_hook does not exist
                decoded_data = json.loads(data)
            if '' in decoded_data:
                del decoded_data['']
            for key, value in list(decoded_data.items()):
                if isinstance(value, str):
                    continue
                
                if isinstance(value, Iterable):
                    decoded_data[key] = ','.join(value)
                else:
                    decoded_data[key] = str(value)
            return decoded_data
        except Exception as exp:
            if 'Duplicate keys:' in str(exp):
                duplicate_keys = str(exp).split(':')[1]
                if len(duplicate_keys.split(',')) == 1:
                    return self._return_in_400(self._('listener.error_duplicate_key') % duplicate_keys, logger)
                else:
                    return self._return_in_400(self._('listener.error_duplicate_keys') % duplicate_keys, logger)
            self._return_in_400(self._('listener.error_bad_json') % exp, logger)
    
    
    def delete_item(self, item_uuid: str, item_type: str, import_needed: bool = True):
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400('invalid type %s' % item_type, self.logger_creation)
        
        item_type = ITEM_TYPE.HOSTS
        
        item_found = self.get_dataprovider().find_items(item_type, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name(), where={'_SE_UUID': item_uuid})
        if any(item_found):
            item_found = item_found[0]
        else:
            self.logger_delete.error('item_type : %s, se_uuid : "%s" => There is no %s with this se_uuid %s' % (item_type, item_uuid, item_type.capitalize(), self._get_request_information()))
            raise HTTPResponse(self._('listener.error_id') % (self._('type.host').lower(), item_uuid), 404)
        
        item_name = get_name_from_type(item_type, item_found)
        
        self.logger_delete.info('item_type : %s, name "%s", se_uuid : "%s" => %s' % (item_type, item_name, item_uuid, self._get_request_information()))
        with self.items_lock:
            self.get_dataprovider().delete_item(item_found, item_type, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name())
        response.status = 200
        self.callback_synchronizer_about_delete_elements(items_type=item_type, data={'_id': item_uuid}, import_needed=import_needed)
        self.logger_delete.info('item_type : %s, name "%s", se_uuid : "%s" => %s is DELETED' % (item_type, item_name, item_uuid, item_type.capitalize()))
        return 'done'
    
    
    def update_item(self, item_uuid: str, item_type: str):
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400('invalid type %s' % item_type, self.logger_creation)
        
        hosts = self.get_dataprovider().find_items(ITEM_TYPE.HOSTS, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name(), where={'_SE_UUID': item_uuid})
        new_data = self._get_data(self.logger_update)
        item_type = get_type_item_from_class(ITEM_TYPE.HOSTS, new_data)
        
        self.logger_update.info('item_type : %s, se_uuid : "%s" => %s' % (item_type, item_uuid, self._get_request_information()))
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400(self._('listener.error_only_host_allowed') % item_type, self.logger_update)
        
        host = None  # type: Optional[Dict[str, Any]]
        if any(hosts):
            raw_source_host = hosts[0]
            name_key = DEF_ITEMS[item_type]['key_name']
            host = {
                name_key  : raw_source_host[name_key],
                '_SE_UUID': 'core-%s-%s' % (item_type, raw_source_host['_id']),
                '_id'     : raw_source_host['_id']
            }
        else:
            # if no host in listeners, try to fund it in work_area or staging
            where = {'_id': item_uuid}
            if item_uuid.startswith('core'):
                where = {'_SE_UUID': item_uuid}
            # try to find an object with the same id in staging or work_area
            staging_hosts = self.get_dataprovider().find_merge_state_items(item_type, item_states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING], where=where)
            if any(staging_hosts):
                # now re-use the object _SE_UUID or one from the staging object
                staging_host = staging_hosts[0]
                name_key = DEF_ITEMS[item_type]['key_name']
                host = {
                    name_key  : staging_host[name_key],
                    '_SE_UUID': 'core-%s-%s' % (item_type, staging_host['_id']),
                    '_id'     : staging_host['_id']
                }
        
        # make sure the update host have the same id as the old one
        if not host:
            self.logger_update.error('item_type : %s, se_uuid : "%s" => There is no %s with this se_uuid' % (item_type, item_uuid, item_type))
            raise HTTPResponse(self._('listener.error_se_uuid') % (self._('type.host').lower(), item_uuid), 404)
        
        key_name = DEF_ITEMS[item_type]['key_name']
        new_name = new_data.get(key_name, None)
        if new_name:
            self.logger_update.info('item_type : %s, name : "%s", se_uuid : "%s" => the request will change the name to "%s"' % (item_type, get_name_from_type(item_type, host), item_uuid, new_name))
        
        # use the NOT_TO_LOOK as list of forbidden keys
        forbidden_keys = set([key.upper().strip() for key in list(new_data.keys())]).intersection({name.upper().strip() for name in NOT_TO_LOOK})
        if forbidden_keys:
            return self._return_in_400(self._('listener.error_forbidden_keys') % ', '.join(forbidden_keys), self.logger_update)
        
        # add the '_id' and host_name in sent data, so we can keep track of this object (in the sources' history)
        sent_data = new_data.copy()
        sent_data['_id'] = item_uuid
        
        if key_name not in sent_data and key_name in host:
            sent_data[key_name] = host[key_name]
        elif key_name in sent_data and sent_data[key_name] != host[key_name]:
            # host name have been updated
            # we have to update the synk_keys
            sync_keys = list(host.get('_SYNC_KEYS', ()))
            # remove the old host name
            if host[key_name] in sync_keys:
                sync_keys.remove(host[key_name])
            if not sent_data[key_name] in sync_keys:
                sync_keys.append(sent_data[key_name])
            host['_SYNC_KEYS'] = list(map(operator.methodcaller('lower'), sync_keys))
        
        host.update(new_data)
        host['update_date'] = time.time()
        host['source'] = self.get_name()
        host['import_date'] = time.time()
        
        if not host.get('_SYNC_KEYS', None):
            host['_SYNC_KEYS'] = '%s,%s' % (host['_SE_UUID'], host[key_name])
        with self.items_lock:
            database_cipher = ComponentManagerSynchronizer.get_database_cipher_component()
            ciphered_host = database_cipher.cipher(host, item_type=item_type)
            self.get_dataprovider().save_item(ciphered_host, item_type=item_type, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name())
        response.status = 200
        self.callback_synchronizer_about_update_elements(items_type=item_type, data=sent_data)
        self.logger_update.info('item_type : %s, name : "%s", se_uuid : "%s" => %s is valid and UPDATED' % (item_type, get_name_from_type(item_type, host), item_uuid, item_type.capitalize()))
        return 'done'
    
    
    def get_item(self, item_uuid, item_type: str):
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400('invalid type %s' % item_type, self.logger_creation)
        
        item_type = ITEM_TYPE.HOSTS
        ciphered_items = self.get_dataprovider().find_items(item_type, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name(), where={'_SE_UUID': item_uuid})
        if not any(ciphered_items):
            self.logger_get.error('item_type : %s, se_uuid : "%s" => There is no %s with this se_uuid %s' % (item_type, item_uuid, item_type.capitalize(), self._get_request_information()))
            raise HTTPResponse(self._('listener.error_se_uuid') % (self._('type.host').lower(), item_uuid), 404)
        
        item = ciphered_items[0]
        self.logger_get.info('item_type : %s, name : "%s", se_uuid : "%s" => %s' % (item_type, get_name_from_type(item_type, item), item_uuid, self._get_request_information()))
        database_cipher = ComponentManagerSynchronizer.get_database_cipher_component()
        host = database_cipher.uncipher(item, item_type=item_type)
        host.pop(METADATA.PROPERTY_NAME, None)
        return json.dumps(host)
    
    
    def list_items(self, item_type: str):
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400('invalid type %s' % item_type, self.logger_creation)
        
        item_type = ITEM_TYPE.HOSTS
        self.logger_list.info('item_type : %s => %s' % (item_type, self._get_request_information()))
        raw_hosts = self.get_dataprovider().find_items(ITEM_TYPE.HOSTS, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name())
        hosts = []
        for host in raw_hosts:
            database_cipher = ComponentManagerSynchronizer.get_database_cipher_component()
            database_cipher.uncipher(host, item_type=ITEM_TYPE.HOSTS)
            host.pop(METADATA.PROPERTY_NAME, None)
            hosts.append(host)
        
        return json.dumps(hosts)
    
    
    def create_item(self, item_type: str):
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400('invalid type %s' % item_type, self.logger_creation)
        
        in_creation = True
        item: dict[str, str | float] = self._get_data(self.logger_creation)
        sent_data = item.copy()
        item_type = get_type_item_from_class(ITEM_TYPE.HOSTS, sent_data)
        item_name = get_name_from_type(item_type, sent_data)
        key_for_name = DEF_ITEMS[item_type]['key_name']
        database_cipher = ComponentManagerSynchronizer.get_database_cipher_component()
        
        self.logger_creation.info('item_type : %s, name : "%s" => %s' % (item_type, item_name, self._get_request_information()))
        
        if item_type != ITEM_TYPE.HOSTS:
            return self._return_in_400(self._('listener.error_only_host_allowed') % item_type, self.logger_creation)
        
        if not item_name:
            return self._return_in_400(self._('listener.error_missing_name_field') % key_for_name, self.logger_creation)
        
        if not isinstance(item_name, str):
            return self._return_in_400(self._('listener.error_name_not_string') % key_for_name, self.logger_creation)
        
        # use the NOT_TO_LOOK as list of forbidden keys
        forbidden_keys = set([key.upper().strip() for key in list(item.keys())]).intersection(set(([name.upper().strip() for name in NOT_TO_LOOK])))
        if forbidden_keys:
            return self._return_in_400(self._('listener.error_forbidden_keys') % ', '.join(forbidden_keys), self.logger_creation)
        
        duplicate_keys = []
        for try_key in [k[:-7] for k in item if k.endswith('[FORCE]')]:
            if item.get(try_key, ''):
                duplicate_keys.append(try_key)
        
        if duplicate_keys:
            if len(duplicate_keys) == 1:
                return self._return_in_400(self._('listener.error_duplicate_key') % ', '.join(duplicate_keys), self.logger_creation)
            else:
                return self._return_in_400(self._('listener.error_duplicate_keys') % ', '.join(duplicate_keys), self.logger_creation)
        
        # try to find if an already object exists with this name
        if '_id' not in item:
            name_ignore_case = re.compile(r'^%s$' % item_name, re.IGNORECASE)
            ciphered_item_in_listener = self.get_dataprovider().find_items(item_type, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name(), where={key_for_name: name_ignore_case})
            if any(ciphered_item_in_listener):
                in_creation = False
                # an old item have been found, update the item and return the id
                item_in_listener: 'ListenerRawItem' = database_cipher.uncipher(ciphered_item_in_listener[0], item_type=item_type)
                self.logger_creation.info('item_type : %s, name : "%s" => %s already exists in listener with the same name ( se_uuid "%s" ). This request will UPDATE the item instead of creating it' % (
                    item_type, item_name, item_type.capitalize(), item_in_listener['_SE_UUID']))
                # remove the _SYNC_KEYS, in any case they will be overridden
                item_in_listener.pop('_SYNC_KEYS', None)
                key_to_remove = []
                for key in item_in_listener:
                    try_key = key[:-7] if key.endswith('[FORCE]') else '%s[FORCE]' % key
                    if item.get(try_key, ''):
                        key_to_remove.append(key)
                for key in key_to_remove:
                    item_in_listener.pop(key)
                item_in_listener.update(item)
                item = item_in_listener
                # the _id is an Object, and mongo will refuse to save it again, cast it in unicode
                item['_id'] = str(item['_id'])
            # else try to find if an already object exists with this name in staging and working area
            else:
                # try to find an object with the same name in raw_source
                # for them just reuse their _id and/or _SE_UUID if available
                existing_items = self.get_dataprovider().find_merge_state_items(item_type, item_states=[ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING], where={key_for_name: name_ignore_case})
                if any(existing_items):
                    existing_item = existing_items[0]
                    item['_SE_UUID'] = 'core-%s-%s' % (item_type, existing_item['_id'])
                    item['_id'] = existing_item['_id']
                    self.logger_creation.info(
                        'item_type : %(item_type)s, name : "%(item_name)s" => %(pretty_item_type)s already exists in %(item_state)s with the same name ( se_uuid "%(item_se_uuid)s" ). This request will CREATE item in listener with same SE_UUID' %
                        {
                            'item_type'       : item_type,
                            'item_name'       : item_name,
                            'pretty_item_type': item_type.capitalize(),
                            'item_state'      : METADATA.get_metadata(existing_item, METADATA.STATE),
                            'item_se_uuid'    : item['_SE_UUID']
                        })
        
        if '_SE_UUID' in item:
            item_se_uuid = item['_SE_UUID']
        else:
            _id = uuid.uuid4().hex
            item['_id'] = _id
            item_se_uuid = 'core-%s-%s' % (item_type, _id)
            item['_SE_UUID'] = item_se_uuid
        
        for (k, v) in item.items():
            if not k.startswith('_'):
                continue
            if not isinstance(k, str):
                return self._return_in_400(self._('listener.error_data_keys_not_strings') % (type(k), k), self.logger_creation)
            if k != '_id' and k != k.upper():
                return self._return_in_400(self._('listener.error_data_keys_not_uppercase'), self.logger_creation)
            if not isinstance(v, str):
                return self._return_in_400(self._('listener.error_data_values_not_strings') % (type(v), k, v), self.logger_creation)
        
        address = item.get('address', '')
        if address and not isinstance(address, str):
            return self._return_in_400(self._('listener.error_address_not_string'), self.logger_creation)
        
        item['_SYNC_KEYS'] = '%s,%s' % (item_se_uuid.lower(), item_name.lower())
        item['update_date'] = time.time()
        # request.environ.get('HTTP_USER_AGENT', '') # if needed
        item['imported_from'] = '%s %s' % (self.get_my_source().get_name(), self._('listener.sent_from') % request.environ.get('REMOTE_ADDR', 'UNSET'))
        
        with self.items_lock:
            ciphered_host = database_cipher.cipher(item, item_type=item_type)
            self.get_dataprovider().save_item(ciphered_host, item_type=item_type, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name())
        
        self.callback_synchronizer_about_new_elements(items_type=item_type, data=sent_data)
        
        self.logger_creation.info('item_type : %s, name : "%s" => %s is valid and %s' % (item_type, item_name, item_type.capitalize(), 'CREATED' if in_creation else 'UPDATED'))
        # HTTP code 201 is OK CREATED => REST COMPLIANCE
        response.status = 201
        return json.dumps(item_se_uuid)
    
    
    def get_all_discovery_elements(self):
        database_cipher = ComponentManagerSynchronizer.get_database_cipher_component()
        raw_objects = {'host': []}
        # Get a copy of the values as we don't know how much time they will be keep outside this code, and so outside the lock
        with self.items_lock:
            my_hosts = self.get_dataprovider().find_items(ITEM_TYPE.HOSTS, item_state=ITEM_STATE.INTERNAL_LISTENER, item_source=self.get_name())
        
        # remove unnecessary property for object after merge and set them as metadata
        for host in my_hosts:
            update_date = host.pop('update_date', None)
            host = database_cipher.uncipher(host, item_type=ITEM_TYPE.HOSTS)
            if update_date:
                METADATA.update_metadata(host, METADATA.UPDATE_DATE, int(update_date))
        
        output = self.syncdaemon.t('listener.output_successful') % len(my_hosts)
        raw_objects['host'] = my_hosts
        res = {'state': 'OK', 'output': output, 'objects': raw_objects, 'errors': [], 'warnings': []}
        return res
    
    
    def remove_source_item(self, item_type, source_item):
        with self.items_lock:
            self.delete_item(source_item.get('_SE_UUID', source_item['_id']), item_type, import_needed=False)
    
    
    @staticmethod
    def _get_request_information():
        # type: () -> str
        if getattr(request, 'environ', None) is None:
            return '( request from Configuration UI )'
        else:
            return '( HTTP %s request from %s )' % (request.method, request.remote_addr)
