#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.
import base64
import http.client
import socket
import ssl
import time
import urllib.request, urllib.parse, urllib.error

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.api.synchronizer import ITEM_TYPE, DEF_ITEMS, SourcePropertyDefinition, SourcePropertyDefinitionBind, PYTHON_TYPE, PROPERTY_TYPE, HTML_CATEGORY, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE, PUT_IN, ComponentManagerSyncui, \
    DaemonIsRequestedToStopException
from shinkensolutions.api.synchronizer.source.callback.callback_interface import SourceCallbackOnDelete
from shinkensolutions.api.synchronizer.source.route.route import AbstractRoute

if TYPE_CHECKING:
    from shinken.misc.type_hint import Union, Dict, List
    from shinken.log import PartLogger
    from shinkensolutions.api.synchronizer import SourceConfiguration, SourceTranslatePart
    from shinkensolutions.api.synchronizer.source.abstract_module.source_module import SourceModule

DOWNTIME_PROPERTIES_DEFINITIONS = [
    SourcePropertyDefinition('downtime_on_delete_element_activated', default_value='0', python_type=PYTHON_TYPE.BOOLEAN, save_in_last_execution=True, html_type=PROPERTY_TYPE.CHECKBOX, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=1,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source'], inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED], show_help=['1', '0']),
                             tabulated=1),
    SourcePropertyDefinition('downtime_on_delete_element_duration', default_value='1440', python_type=PYTHON_TYPE.INTEGER, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=2,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source', 'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, '0'], show_help=['1', '0', '0']), tabulated=2),
    SourcePropertyDefinition('downtime_on_delete_element_comment', default_value='', python_type=PYTHON_TYPE.STRING, html_type=PROPERTY_TYPE.TEXTAREA, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=3,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source', 'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, '0'], show_help=['1', '0', '0']), tabulated=2),
    SourcePropertyDefinition('downtime_on_delete_element_author', default_value='', python_type=PYTHON_TYPE.USER, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=4,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source', 'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, '0'], show_help=['1', '0', '0']), tabulated=2),
    SourcePropertyDefinition('downtime_on_delete_element_api_url', default_value='127.0.0.1', python_type=PYTHON_TYPE.STRING, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=5,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source', 'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, '0'], show_help=['1', '0', '0']), tabulated=2),
    SourcePropertyDefinition('downtime_on_delete_element_api_port', default_value='7760', python_type=PYTHON_TYPE.STRING, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=6,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source', 'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, '0'], show_help=['1', '0', '0']), tabulated=2),
    SourcePropertyDefinition('downtime_on_delete_element_api_use_ssl', default_value='0', python_type=PYTHON_TYPE.BOOLEAN, html_type=PROPERTY_TYPE.CHECKBOX, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=7,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source', 'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, '0'], show_help=['1', '0', '0']), tabulated=2),
    SourcePropertyDefinition('downtime_on_delete_element_api_user', default_value='admin', python_type=PYTHON_TYPE.STRING, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=8,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source', 'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, '0'], show_help=['1', '0', '0']), tabulated=2),
    SourcePropertyDefinition('downtime_on_delete_element_api_password', default_value='admin', python_type=PYTHON_TYPE.STRING, html_type=PROPERTY_TYPE.INPUT_PASSWORD, html_category=HTML_CATEGORY.DOWNTIME_ON_DELETE, order=9,
                             linked_definition=SourcePropertyDefinitionBind(['put_in', 'compute_deleted_element_not_in_source', 'downtime_on_delete_element_activated'],
                                                                            inactive_value=[PUT_IN.SOURCE_SPACE, COMPUTE_DELETED_ELEMENT_NOT_IN_SOURCE.DISABLED, '0'], show_help=['1', '0', '0']), tabulated=2)
]  # type: List[SourcePropertyDefinition]


class CallbackDowntime(SourceCallbackOnDelete):
    
    def __init__(self):
        super(CallbackDowntime, self).__init__()
        self.downtime_on_delete_element_activated = False
        self.downtime_on_delete_element_duration = ''
        self.downtime_on_delete_element_comment = ''
        self.downtime_on_delete_element_author = ''
        self.downtime_on_delete_element_api_url = ''
        self.downtime_on_delete_element_api_port = ''
        self.downtime_on_delete_element_api_use_ssl = False
        
        self.downtime_on_delete_element_api_user = ''
        self.downtime_on_delete_element_api_password = ''
        self._auth_string = ''
        self.PROPERTIES_DEFINITIONS = DOWNTIME_PROPERTIES_DEFINITIONS
    
    
    def init_callback(self, conf, logger, source_name, translator):
        # type: (SourceConfiguration, PartLogger, str, SourceTranslatePart) -> None
        super(CallbackDowntime, self).init_callback(conf, logger, source_name, translator)
        self.logger = logger.get_sub_part('CallbackDowntime')
        
        self.downtime_on_delete_element_activated = getattr(conf, 'downtime_on_delete_element_activated')
        self.downtime_on_delete_element_duration = getattr(conf, 'downtime_on_delete_element_duration')
        self.downtime_on_delete_element_comment = getattr(conf, 'downtime_on_delete_element_comment')
        self.downtime_on_delete_element_author = getattr(conf, 'downtime_on_delete_element_author')
        self.downtime_on_delete_element_api_url = getattr(conf, 'downtime_on_delete_element_api_url')
        self.downtime_on_delete_element_api_port = getattr(conf, 'downtime_on_delete_element_api_port')
        self.downtime_on_delete_element_api_use_ssl = getattr(conf, 'downtime_on_delete_element_api_use_ssl')
        
        self.downtime_on_delete_element_api_user = getattr(conf, 'downtime_on_delete_element_api_user')
        self.downtime_on_delete_element_api_password = getattr(conf, 'downtime_on_delete_element_api_password')
        self._auth_string = base64.b64encode(b'%s:%s' % (self.downtime_on_delete_element_api_user.encode('utf-8'), self.downtime_on_delete_element_api_password.encode('utf-8'))).replace(b'\n', b'').decode('utf-8')
    
    
    def callback_on_delete(self, item, item_type):
        # type: (Union[SourceModule,CallbackDowntime], Dict, str) -> None
        
        if not self.downtime_on_delete_element_activated:
            return
        
        if item_type not in ITEM_TYPE.ALL_HOST_CLASS:
            self.logger.debug('Canceling CallbackDowntime, the item type of %s is not HOST !' % item[DEF_ITEMS[item_type].get('key_name')])
            return
        
        # Be sure that the item_name is in str (maybe the source give us bytes)
        item_name = item['host_name']
        if isinstance(item_name, bytes):
            item_name = item_name.decode('utf-8', 'ignore')
        
        receiver_con = self.get_receiver_syncui_connection(('%s:%s' % (self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port)), self.downtime_on_delete_element_api_use_ssl)
        
        downtime_start = '%s' % str(int(time.time()))
        downtime_end = '%s' % str(int(time.time() + int(self.downtime_on_delete_element_duration) * 60))
        
        self.logger.info(
            '[DOWNTIME] Creating a downtime on host %s, by author %s, ending in %sm to receiver api %s:%s.' % (
                item_name, self.downtime_on_delete_element_author, self.downtime_on_delete_element_duration, self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port))
        
        params = urllib.parse.urlencode({
            'host_name' : item_name.encode('utf8'),
            'start_time': downtime_start,
            'end_time'  : downtime_end,
            'author'    : self.downtime_on_delete_element_author.encode('utf8'),  # NOTE: url encode MUST be str if with utf8
            'comment'   : self.downtime_on_delete_element_comment.encode('utf8'),
        })
        
        headers = {
            'Content-type' : 'application/x-www-form-urlencoded',
            'Accept'       : 'text/plain',
            'Authorization': 'Basic %s' % self._auth_string
        }
        
        try:
            receiver_con.request('POST', '/downtime', params, headers)
            resp = receiver_con.getresponse()
            if resp.status != 200 and resp.reason != 'OK':
                msg = 'Callback error: Status=%d, reason=%s' % (resp.status, resp.reason)
                self.logger.warning(msg)
                raise Exception(msg)
        except socket.error:
            msg = 'No connection could be established to %s:%s because the target computer expressly refused it' % (self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port)
            self.logger.warning(msg)
            raise Exception(msg)
    
    
    @staticmethod
    def get_receiver_syncui_connection(url, http_use_ssl=False):
        if http_use_ssl:
            # If we are in SSL mode, do not look at certificate too much
            # NOTE: ssl.SSLContext is only available on last python 2.7 versions
            if hasattr(ssl, 'SSLContext'):
                ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                ssl_context.check_hostname = False
                ssl_context.verify_mode = ssl.CERT_NONE
            else:
                ssl_context = None
            
            args = {}
            if ssl_context:
                args['context'] = ssl_context
            conn = http.client.HTTPSConnection(url, **args)
        else:
            conn = http.client.HTTPConnection(url)
        return conn
    
    
    def test_connection(self, import_in_progress):
        # type: (bool) -> Dict[str,str]
        
        if import_in_progress:
            _return = {
                'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                'output': self.translator.translate('callback_downtime.test_connection.source_is_running'),
            }
        
        if not self.downtime_on_delete_element_activated:
            return {
                'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                'output': self.translator.translate('callback_downtime.test_connection.callback_is_disable'),
            }
        
        receiver_url = ('%s:%s' % (self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port))
        receiver_con = self.get_receiver_syncui_connection(receiver_url, self.downtime_on_delete_element_api_use_ssl)
        
        params = ''
        headers = {
            'Content-type' : 'application/x-www-form-urlencoded',
            'Accept'       : 'text/plain',
            'Authorization': 'Basic %s' % self._auth_string
        }
        
        try:
            receiver_con.request('GET', '/ping', params, headers)
            resp = receiver_con.getresponse()
            if resp.status == 401:
                return {
                    'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                    'output': self.translator.translate('callback_downtime.test_connection.connection_refused_missing_password') % receiver_url
                }
            elif resp.status == 403:
                return {
                    'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                    'output': self.translator.translate('callback_downtime.test_connection.connection_refused_incorrect_password') % receiver_url
                }
            elif resp.status != 200:
                return {
                    'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                    'output': self.translator.translate('callback_downtime.test_connection.unexpected_return') % receiver_url
                }
            else:
                _read = resp.read()
                if _read != 'pong:ws arbiter':
                    return {
                        'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                        'output': self.translator.translate('callback_downtime.test_connection.unexpected_return') % receiver_url
                    }
        except socket.error as e:
            return {
                'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                'output': self.translator.translate('callback_downtime.test_connection.connection_refused') % (receiver_url, e.strerror)
            }
        except Exception as e:
            return {
                'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                'output': self.translator.translate('callback_downtime.test_connection.connection_refused') % (receiver_url, str(e))
            }
        
        _return = {
            'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.OK,
            'output': self.translator.translate('callback_downtime.test_connection.test_is_ok') % ('%s:%s' % (self.downtime_on_delete_element_api_url, self.downtime_on_delete_element_api_port))
        }
        return _return


class RouteTestCallbackDowntimeConnection(AbstractRoute):
    class CONNECTION_STATE:
        OK = 'ok'
        KO = 'ko'
    
    def __init__(self, logger, source_name, callback_downtime):
        # type: (PartLogger, str, CallbackDowntime) -> None
        super(RouteTestCallbackDowntimeConnection, self).__init__(logger, source_name, 'callback_downtime/test_connection')
        self.callback_downtime = callback_downtime
    
    
    def controller(self):
        # type: () -> Dict[str,str]
        _source_info = ComponentManagerSyncui.get_source_info_component().get_source_info(self.source_name)
        
        try:
            _return = ComponentManagerSyncui.get_internal_api_synchronizer_component().test_callback_downtime_connection(self.source_name, _source_info.is_import_in_progress())
        except DaemonIsRequestedToStopException:
            daemon_is_requested_to_stop = ComponentManagerSyncui.get_translate_component().translator().translate('common.backend.exceptions.daemon_is_requested_to_stop')
            _return = {
                'state' : RouteTestCallbackDowntimeConnection.CONNECTION_STATE.KO,
                'output': daemon_is_requested_to_stop
            }
        return _return
