#!/usr/bin/env python3-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.


import json
import os

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.graphite_backend import GraphiteBackendConsts, GraphiteBackend
from shinken.toolbox.url_helper import BaseUrl
from shinkensolutions.cfg_formatter.config_writer import FORMATTER_PATHS

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Any, List, Union, Set, Callable
    
    ConfigUpdateInfo = Dict[str, Any]
    ConfigUpdate = Dict[str, Union[str, int, ConfigUpdateInfo]]
    Config = Dict[str, List[Any]]
    ConfigElement = Dict[str, Any]


def _get_file_path(config_element):
    # type: (ConfigElement) -> str
    if isinstance(config_element['imported_from'], list):
        file_path = config_element['imported_from'][0]
    else:
        file_path = config_element['imported_from']
    return file_path


class ShinkenCfgUpdaterException(Exception):
    pass


class ConfigHooks:
    
    @staticmethod
    def broker_module_livedata_add_broker_module_livedata__api_version(config_element_to_update, update_info, callback):
        # type: (ConfigElement, ConfigUpdateInfo,Callable([str, str, str], None)) -> None
        _input = update_info.get('input')
        if not _input[0] in config_element_to_update and _input[1] not in config_element_to_update:
            config_element_to_update['broker_module_livedata__api_version'] = ['1']
    
    
    @staticmethod
    def broker_module_webui_format_graphite_backends(config_element_to_update, update_info, callback):
        # type: (ConfigElement, ConfigUpdateInfo, Callable([str, str, str], None)) -> None
        configuration_file_path = _get_file_path(config_element_to_update)
        _input = update_info.get('input')
        graphite_backends_raw = config_element_to_update.get(_input[0], [''])[0]
        if graphite_backends_raw:
            graphite_backends = graphite_backends_raw.split(GraphiteBackendConsts.SEPARATOR__BACKEND_LIST)
            new_backends = []
            for backend in graphite_backends:
                if GraphiteBackendConsts.SEPARATOR__REALM in backend:
                    # Already at newer format, or malformed
                    continue
                realm, url = backend.split(GraphiteBackendConsts.SEPARATOR__HOST_PORT, 1)
                url = BaseUrl.from_url(url, strict=False)
                new_format = GraphiteBackend.get_graphite_backend_string_formatted(realm, 'http', url.get_host(), url.get_port())
                new_backends.append(new_format)
            if new_backends:
                callback(configuration_file_path, config_element_to_update[_input[0]][0].replace('*:', '\\*:'), ', '.join(new_backends))
                config_element_to_update[_input[0]] = [', '.join(new_backends)]


class ConfigConverter:
    
    STARTING_VERSION = 0
    CONFIG_FORMAT_ID_TO_CONFIG_ELEMENT_TYPE = {
        'broker_module_livedata'      : 'module',
        'webui'                       : 'module',
        'synchronizer-import'         : 'module',
        'webui_module_service_weather': 'module',
        'livedata_module_sla_provider': 'module',
        
        'realm'                       : 'realm',
        
        'arbiter'                     : 'arbiter',
        'scheduler'                   : 'scheduler',
        'reactionner'                 : 'reactionner',
        'poller'                      : 'poller',
        'synchronizer'                : 'synchronizer',
        'receiver'                    : 'receiver',
        'broker'                      : 'broker',
    }
    
    configuration_updates_to_apply = {}
    
    
    def _execute_update_for_all_elements_and_overrides(self, config, config_element_type, update_info):
        # type: (Config, str, ConfigUpdateInfo) -> None
        if self.CONFIG_FORMAT_ID_TO_CONFIG_ELEMENT_TYPE.get(config_element_type) == 'module':
            config_elements_to_update = [m for m in config['module'] if m.get('module_type', None) == [config_element_type]]
        else:
            config_elements_to_update = [m for m in config[config_element_type]]
        
        for config_element_to_update in config_elements_to_update:
            self._apply_update(config_element_to_update, update_info)
            
            if config_element_to_update.get('overrides'):
                for overrides in config_element_to_update.get('overrides'):
                    self._apply_update(overrides, update_info)
    
    
    def _update_configuration_updates_to_apply(self, configuration_file_path, old_value, new_value):
        # type: (str,str, str) -> None
        if configuration_file_path in self.configuration_updates_to_apply and old_value in self.configuration_updates_to_apply[configuration_file_path]:
            self.configuration_updates_to_apply[configuration_file_path][new_value] = self.configuration_updates_to_apply[configuration_file_path][old_value]
            del self.configuration_updates_to_apply[configuration_file_path][old_value]
        else:
            self.configuration_updates_to_apply[configuration_file_path] = {}
            self.configuration_updates_to_apply[configuration_file_path][new_value] = old_value
    
    
    def _apply_rename_parameter(self, config_element, config_update_info):
        # type: (ConfigElement, ConfigUpdateInfo) -> None
        
        previous_key = config_update_info.get('previous_key')
        new_key = config_update_info.get('new_key')
        configuration_file_path = _get_file_path(config_element)
        if config_element.get(previous_key, None) and new_key not in config_element:
            config_element[new_key] = config_element.get(previous_key)
            del config_element[previous_key]
            self._update_configuration_updates_to_apply(configuration_file_path, previous_key, new_key)
    
    
    def _apply_hook(self, config_element, config_update_info):
        # type: (ConfigElement, ConfigUpdateInfo) -> None
        if hasattr(ConfigHooks, config_update_info.get('name')) and callable(getattr(ConfigHooks, config_update_info.get('name'))):
            method = getattr(ConfigHooks, config_update_info.get('name'))
            method(config_element, config_update_info, self._update_configuration_updates_to_apply)
    
    
    def _apply_update(self, config_element, config_update_info):
        # type: (ConfigElement, ConfigUpdateInfo) -> None
        
        if config_update_info.get('type') == 'rename_parameter':
            self._apply_rename_parameter(config_element, config_update_info)
        
        elif config_update_info.get('type') == 'hook':
            self._apply_hook(config_element, config_update_info)
    
    
    @staticmethod
    def _check_config_update_validity(deprecated_parameters, config_element_type, config_update_info):
        # type: (Set[str], str, ConfigUpdateInfo) -> None
        if config_update_info.get('type') == 'rename_parameter':
            if '%s_%s' % (config_element_type, config_update_info.get('new_key')) in deprecated_parameters:
                raise ShinkenCfgUpdaterException('The key %s could not be used in the %s\'s configuration file because it has already been used in the past.' % (config_update_info.get('new_key'), config_element_type))
            else:
                deprecated_parameters.add('%s_%s' % (config_element_type, config_update_info.get('previous_key')))
    
    
    @staticmethod
    def _sort_config_updates(config_updates):
        # type: (List[ConfigUpdate]) -> List[ConfigUpdate]
        return sorted(config_updates, key=lambda x: x['date'])
    
    
    def _read_config_updates(self):
        # type: () -> List[ConfigUpdate]
        updates_list = []
        folder_path = FORMATTER_PATHS.CONFIG_UPDATE_FOLDER
        for filename in os.listdir(folder_path):
            if not filename.endswith('.json'):
                continue
            
            file_path = os.path.join(folder_path, filename)
            
            with open(file_path, 'r') as _file:
                try:
                    json_data = json.load(_file)
                    if json_data.get('version', 0) >= self.STARTING_VERSION:
                        updates_list.extend(json_data.get('config_updates_infos', []))
                
                except Exception as e:
                    raise ShinkenCfgUpdaterException('Error retrieving %s : %s' % (file_path, e))
        return updates_list
    
    
    def _execute_updates(self, config_updates_infos, config):
        # type: (List[ConfigUpdate], Config) -> None
        deprecated_parameters = set()
        for update_info in config_updates_infos:
            config_element_type, config_update_info = update_info.get('config_element_type'), update_info.get('config_update_info')
            
            self._check_config_update_validity(deprecated_parameters, config_element_type, config_update_info)
            self._execute_update_for_all_elements_and_overrides(config, config_element_type, config_update_info)
    
    
    def convert_config_to_new_format(self, config):
        #  type: (Config) -> None
        
        config_updates = self._read_config_updates()
        config_updates = self._sort_config_updates(config_updates)
        self._execute_updates(config_updates, config)
