#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from .constant import SHINKEN_GRID_CONSTS
from .square import Square, Point
from ..object.messages import ValidatorMessages
from ..object.shinken_ui_composite.shinken_ui_containers import ShinkenUIObject, ShinkenUIList
from ..object.shinken_ui_composite.shinken_ui_property import ShinkenUIIntegerProperty, ShinkenUIFloatProperty, ShinkenUIIntegerOrEnumProperty, ShinkenUIEnumProperty

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List, Type, Dict, Any, Tuple

GRID_TYPE_POSSIBLE_VALUES = ['grid']


class ShinkenGrid(ShinkenUIObject):
    
    def __init__(self, associated_key, validator_message_type=None, mandatory=False, grid_elements_type=ShinkenUIList):
        # type: (str, Type[ValidatorMessages], bool, Type[ShinkenUIList]) -> None
        # Object properties
        super(ShinkenGrid, self).__init__(associated_key, validator_message_type=validator_message_type, mandatory=mandatory)
        self.index = 0
        self.type = ShinkenUIEnumProperty(SHINKEN_GRID_CONSTS.KEYS_TYPE, enum=SHINKEN_GRID_CONSTS.GRID_TYPE_ENUM, validator_message_type=validator_message_type, mandatory=True)
        self.nb_tiles_in_width = ShinkenUIIntegerProperty(SHINKEN_GRID_CONSTS.KEYS_NB_TILES_WIDTH, validator_message_type=validator_message_type, allow_default=True, must_be_strict_positive=True)
        self.nb_tiles_in_height = ShinkenUIIntegerProperty(SHINKEN_GRID_CONSTS.KEYS_NB_TILES_HEIGHT, validator_message_type=validator_message_type, allow_default=True, must_be_strict_positive=True)
        self.ordering = None
        self.separation_grid_element_min = ShinkenUIIntegerProperty(SHINKEN_GRID_CONSTS.KEYS_SEPARATION_ELEMENT_MIN, validator_message_type=validator_message_type, allow_default=True, must_be_positive=True)
        self.separation_grid_element_max = ShinkenUIIntegerOrEnumProperty(SHINKEN_GRID_CONSTS.KEYS_SEPARATION_ELEMENT_MAX, validator_message_type=validator_message_type, enum=('unlimited',), allow_default=True, must_be_positive=True)
        self.separation_grid_element_percent = ShinkenUIFloatProperty(SHINKEN_GRID_CONSTS.KEYS_SEPARATION_ELEMENT_PERCENT, validator_message_type=validator_message_type, range_allowed=[0, 100], allow_default=True)
        
        self.grid_elements = grid_elements_type(SHINKEN_GRID_CONSTS.KEYS_GRID_ELEMENTS, validator_message_type=validator_message_type, mandatory=True)
        
        self.square = None  # type: Optional[Square]
        
        # # Filled by validator when overlapping elements are caught
        # self.overlapping_elements = []  # type: List[Tuple[int, int]]
    
    
    def compute_square(self):
        self.square = Square(Point(0, self.nb_tiles_in_height.get_computed_value()), Point(self.nb_tiles_in_width.get_computed_value(), 0))
    
    
    def _get_stored_values_to_dict(self, computed_value, source_value, config_value, include_validation_messages):
        # type: (bool, bool, bool, bool) -> Dict[str, Any]
        _dict_result = super(ShinkenGrid, self)._get_stored_values_to_dict(computed_value=computed_value, source_value=source_value, config_value=config_value, include_validation_messages=include_validation_messages)
        
        # if include_validation_messages and self.overlapping_elements:
        #     _dict_result[SHINKEN_GRID_CONSTS.KEYS_OVERLAPPING_ELEMENTS] = [
        #         {
        #             SHINKEN_GRID_CONSTS.KEYS_OVERLAPPING_ELEMENTS__INDEX_1: index_1,
        #             SHINKEN_GRID_CONSTS.KEYS_OVERLAPPING_ELEMENTS__INDEX_2: index_2
        #         }
        #         for index_1, index_2 in self.overlapping_elements
        #     ]
        
        return _dict_result
