#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict, Any, Union
    from .object.messages import ValidatorMessages

SUCCESS_RESPONSE_CODE = 200
WARNING_RESPONSE_CODE = 202
ERROR_RESPONSE_CODE = 202
CRITICAL_ERROR_RESPONSE_CODE = 400
UNAUTHORIZED_ERROR_RESPONSE_CODE = 401
FORBIDDEN_ERROR_RESPONSE_CODE = 403
ELEMENT_NOT_FOUND_RESPONSE_CODE = 404


class FrontEndResponse:
    def __init__(self, validation_message, return_code=SUCCESS_RESPONSE_CODE, has_changes=False, message_has_changes=False, delete=None, has_item_in_staging=False, default_text='', was_reject=False):
        # type: (ValidatorMessages, Optional[int], Optional[bool], Optional[bool], Optional[str], Optional[bool], Optional[str], Optional[bool]) -> None
        self.return_code = return_code
        self.has_changes = has_changes
        self.message_has_changes = message_has_changes
        self.validation = validation_message
        self.delete = delete
        self.has_item_in_staging = has_item_in_staging
        self.default_text = default_text
        self.was_reject = was_reject
    
    
    def get_response(self, output=''):
        # type: (Union[str, Dict[str, Any], list]) -> Dict[str, Any]
        # take the error code if not already customized
        if self.return_code == SUCCESS_RESPONSE_CODE:
            if self.validation.has_critical():
                self.return_code = CRITICAL_ERROR_RESPONSE_CODE
            elif self.validation.has_error():
                self.return_code = CRITICAL_ERROR_RESPONSE_CODE
        
        return {
            'rc'          : self.return_code,
            'output'      : output if output else self.validation.get_html(default_text=self.default_text),
            'has_critical': self.validation.has_critical(),
            'has_error'   : self.validation.has_error(),
            'has_warning' : self.validation.has_warning(),
            'critical'    : self.validation.extract_critical_messages().get_html(),
            'error'       : self.validation.extract_error_messages().get_html(),
            'warning'     : self.validation.extract_warning_messages().get_html()
        }
