#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.property import BoolProp
from shinken.util import to_bool

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Union, Number, List, Any
    from shinken.log import PartLogger
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from shinken.objects.config import Config
    from synchronizer.synchronizerlink import SynchronizerLink

_bool_prop = BoolProp()


def read_bool_in_configuration(conf, key_name, default_value, conf_error_stats=None, log_fct=None):
    # type: (Union[ShinkenModuleDefinition|Config, Config, SynchronizerLink], str, Union[str, bool], Optional[dict], Optional[PartLogger]) -> bool
    val = getattr(conf, key_name, default_value)
    try:
        if isinstance(val, str):
            val = to_bool(val, with_strict_check=True)
    except:
        val = handle_error(key_name, val, default_value, log_fct, conf_error_stats)
    return val


def read_int_in_configuration(conf, key_name, default_value, conf_error_stats=None, log_fct=None):
    # type: (Union[ShinkenModuleDefinition|Config, Config, SynchronizerLink], str, Union[str, Number], Optional[dict], Optional[PartLogger]) -> int
    val = getattr(conf, key_name, default_value)
    try:
        val = int(val)
    except:
        val = handle_error(key_name, val, default_value, log_fct, conf_error_stats)
    return val


def read_float_in_configuration(conf, key_name, default_value, conf_error_stats=None, log_fct=None):
    # type: (Union[ShinkenModuleDefinition|Config, Config], str, Union[str, float], Optional[dict], PartLogger) -> float
    val = getattr(conf, key_name, default_value)
    try:
        val = float(val)
    except:
        val = handle_error(key_name, val, default_value, log_fct, conf_error_stats)
    return val


def read_string_in_configuration(conf, key_name, default_value, conf_error_stats=None, log_fct=None):
    # type: (Union[ShinkenModuleDefinition|Config, Config, SynchronizerLink], str, str, Optional[dict], PartLogger) -> str
    val = getattr(conf, key_name, default_value)
    if not isinstance(val, str):
        val = handle_error(key_name, val, default_value, log_fct, conf_error_stats)
    return val


def read_list_in_configuration(conf, key_name, default_value, conf_error_stats=None, log_fct=None):
    # type: (Union[ShinkenModuleDefinition|Config, Config], str, List, Optional[dict], PartLogger) -> List[str]
    val = getattr(conf, key_name, default_value)
    try:
        if isinstance(val, str):
            raw_list = val.split(',')
        else:
            raw_list = val
        val = [i.strip() if isinstance(i, str) else i for i in raw_list]
    except:
        val = handle_error(key_name, val, default_value, log_fct, conf_error_stats)
    return val


def handle_error(key_name, val, default_value, log_fct, conf_error_stats):
    # type: (str, Any, Any, PartLogger, Optional[dict]) -> Any
    if log_fct:
        if isinstance(val, list) and not isinstance(default_value, list):
            log_fct.error('%s parameter is duplicated, resetting to default value [%s]' % (key_name, default_value))
        else:
            log_fct.error('%s has incorrect value[%s], resetting to default value [%s]' % (key_name, val, default_value))
    if conf_error_stats is not None:
        conf_error_stats.update({key_name: {'conf_value': val, 'default_value': default_value}})
    val = default_value
    return val
