#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
import json
import os
from collections import OrderedDict, namedtuple

from shinken.exceptions.business import ShinkenExceptionKeyError
from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import WeakMethod

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List, Any, Union, Dict, Iterable, Tuple
    from shinken.log import PartLogger
    from shinken.objects.module import Module as ShinkenModuleDefinition

HIDDEN_LOG_CONTENT = '-*- hidden -*-'

ConfigurationFormat = namedtuple('ConfigurationFormat', ['configuration_keys', 'default_value', 'key_type', 'prop_name'])


# USAGE of ConfigurationFormat :
#
# configuration_keys is List[str]
#  * list of str  Ex: ['old_parameter_name', 'very_old_parameter_name', 'parameter_name']
#  ==> list entries are looked for in conf file in list order, last value found replace previous ones,
#  ==> log will use last list entry as the name for this parameter
#
# key_type from TypeConfiguration class. Special type HIDDEN_XXX will not print clear data in log file
#
# prop_name is Optional[str], if not empty, will setattr self.prop_name with value from conf, else value will only be printed in log


class TypeConfiguration:
    BOOL = 'bool'
    INT = 'int'
    FLOAT = 'float'
    STRING = 'string'
    HIDDEN_STRING = 'hidden_string'
    LIST = 'list'
    MODULES_NAMES = 'modules_names'
    SEPARATOR = '$separator'


class SeparatorFormat:
    def __init__(self, value):
        self.prop_name = value
        self.default_value = value
        self.configuration_keys = list(value)
        self.key_type = TypeConfiguration.SEPARATOR
    
    
    def __iter__(self):
        return iter((self.configuration_keys, self.default_value, self.key_type, self.prop_name))


class ConfigurationReaderMixin:
    def __init__(self, configuration_format, configuration, logger=None, non_editable_parameters=None):
        # type:(List[Union[ConfigurationFormat,SeparatorFormat]], ShinkenModuleDefinition, Optional[PartLogger], Optional[List[str]]) -> None
        self._configuration_format = []
        self._non_editable_parameters = [] if non_editable_parameters is None else non_editable_parameters
        self._configuration = configuration
        self._logger = logger
        self._configuration_error = {}
        self._configuration_load = OrderedDict()
        
        for element in configuration_format:
            if isinstance(element, SeparatorFormat):
                self._configuration_format.append(element)
            elif not isinstance(element, ConfigurationFormat):
                if logger:
                    logger.error('wrong configuration parameter format ignored ( %s )' % element)
                continue
            # Compatibility : configuration_key was str, now we want a list of str
            if logger and not isinstance(element.configuration_keys, list):
                logger.warning('Wrong format for configuration_keys parameter, backward compatibility enabled for key:[%s]' % element.configuration_keys)
            self._configuration_format.append(ConfigurationFormat(configuration_keys=element.configuration_keys if isinstance(element.configuration_keys, list) else [element.configuration_keys],
                                                                  default_value=element.default_value,
                                                                  key_type=element.key_type,
                                                                  prop_name=element.prop_name))
        
        self._readers = {
            TypeConfiguration.BOOL         : WeakMethod(self.read_bool_in_configuration),
            TypeConfiguration.INT          : WeakMethod(self.read_int_in_configuration),
            TypeConfiguration.FLOAT        : WeakMethod(self.read_float_in_configuration),
            TypeConfiguration.STRING       : WeakMethod(self.read_string_in_configuration),
            TypeConfiguration.HIDDEN_STRING: WeakMethod(self.read_string_in_configuration),
            TypeConfiguration.LIST         : WeakMethod(self.read_list_in_configuration),
            TypeConfiguration.MODULES_NAMES: WeakMethod(self.read_modules_names_in_configuration),
        }
    
    
    def get_configuration_hash(self):
        return self._configuration.hash
    
    
    def set_property(self, prop_name, value):
        setattr(self, prop_name, value)
    
    
    def read_configuration(self):
        # type: () -> OrderedDict
        for configuration_keys, default_value, key_type, prop_name in self._configuration_format:
            if key_type == TypeConfiguration.SEPARATOR:
                self._configuration_load[''.join((TypeConfiguration.SEPARATOR, default_value))] = ('', default_value)
            else:
                key_name, val = self._parse_keys(configuration_keys, default_value, key_type, prop_name)
                if prop_name:
                    self.set_property(prop_name, val)
                if key_type in [TypeConfiguration.HIDDEN_STRING]:
                    val = HIDDEN_LOG_CONTENT
                self._configuration_load[key_name] = (prop_name, val)
        
        return self._configuration_load
    
    
    def _force_default_values_for_non_editable_parameters(self):
        # type: () -> None
        for non_editable_parameter in self._non_editable_parameters:
            if non_editable_parameter in self._configuration_format:
                setattr(self, non_editable_parameter, self._configuration_format[non_editable_parameter].default_value)
    
    
    def _parse_keys(self, configuration_keys, default_value, key_type, prop_name):
        # type: (Union[List, str], Any, str, Optional[str]) -> (str, Any)
        conf_has_been_computed_in_arbiter = hasattr(self._configuration, 'module_info_json')
        if conf_has_been_computed_in_arbiter:
            val = getattr(self._configuration, prop_name, None)
            key_name = configuration_keys[-1]
            configuration_reading_errors_from_arbiter = getattr(self._configuration, 'configuration_reading_errors', None)
            if configuration_reading_errors_from_arbiter and key_name in configuration_reading_errors_from_arbiter:
                # Generate log, and store error data
                self.handle_error(key_name, value=configuration_reading_errors_from_arbiter[key_name]['conf_value'], default_value=configuration_reading_errors_from_arbiter[key_name]['default_value'])
            return key_name, val
        
        val = default_value
        key_name = ''
        for conf_key in configuration_keys:
            def_value = val
            key_name = conf_key
            val = self._readers[key_type](conf_key, def_value)
        if not key_name:
            raise ShinkenExceptionKeyError(text='Configuration (keys[ %s ] default:[ %s ] type:[ %s ] property_name:[ %s ]) has wrong key name' % (configuration_keys, default_value, key_type, prop_name))
        return key_name, val
    
    
    def add_separator(self, prop_name):
        # type: (str) -> None
        self._configuration_load[''.join((TypeConfiguration.SEPARATOR, prop_name))] = ('', prop_name)
    
    
    def read_prop(self, prop_name):
        # type: (str) -> None
        conf = next((i for i in self._configuration_format if i.prop_name == prop_name), None)
        if not conf:
            self._logger.error('The prop 〖 %s 〗 is not known by the module configuration' % prop_name)
            return None
        configuration_keys, default_value, key_type, prop_name = conf
        if key_type == TypeConfiguration.SEPARATOR:
            self._configuration_load[''.join((TypeConfiguration.SEPARATOR, default_value))] = ('', default_value)
        else:
            key_name, val = self._parse_keys(configuration_keys, default_value, key_type, prop_name)
            self._configuration_load[key_name] = (prop_name, val)
            return val
    
    
    def reset_prop_with_wrong_value(self, prop_name):
        # type: (str) -> None
        conf_data = next((configuration for configuration in self._configuration_format if getattr(configuration, 'prop_name', '') == prop_name), None)
        if not conf_data:
            return
        
        key_name = conf_data.configuration_keys
        if isinstance(key_name, list):
            key_name = key_name[-1]
        cur_value = getattr(self, prop_name, None)
        default_value = conf_data.default_value
        
        if conf_data.key_type == TypeConfiguration.BOOL and isinstance(default_value, str):
            default_value = default_value == '1'
        elif conf_data.key_type == TypeConfiguration.INT:
            default_value = int(default_value)
        elif conf_data.key_type == TypeConfiguration.FLOAT:
            default_value = float(default_value)
        elif conf_data.key_type in [TypeConfiguration.LIST, TypeConfiguration.MODULES_NAMES]:
            if isinstance(default_value, str):
                default_value = default_value.split(',')
        
        new_value = self.handle_error(key_name, cur_value, default_value)
        setattr(self, prop_name, new_value)
    
    
    def log_configuration(self, log_properties=False, show_values_as_in_conf_file=False):
        # type: (bool,bool) -> None
        line_length = max([len(kn) for kn in self._configuration_load.keys()]) + 2
        for key_name, value in self._configuration_load.items():
            if key_name.startswith(TypeConfiguration.SEPARATOR):
                self._logger.info(value[1])
            else:
                if log_properties and value[0] and value[1] != HIDDEN_LOG_CONTENT:
                    if not hasattr(self, value[0]):
                        raise ShinkenExceptionKeyError(text='〖 %s 〗 parameter not found in prop 〖 %s 〗' % (key_name, value[0]))
                    val = getattr(self, value[0])
                else:
                    val = value[1]
                if show_values_as_in_conf_file:
                    if isinstance(val, bool):
                        val = 1 if val else 0
                    if isinstance(val, list):
                        val = ', '.join([str(i) for i in val])
                self._logger.info('   - %s :〖 %s 〗' % (('%s ' % key_name).ljust(line_length, '—'), val))
    
    
    def read_bool_in_configuration(self, key_name, default_value):
        # type: (str, Any)->bool
        val = getattr(self._configuration, key_name, default_value)
        try:
            if isinstance(val, str):
                if val in ('1', '0'):
                    val = val == '1'
                else:
                    val = self.handle_error(key_name, val, default_value)
                    if isinstance(val, str):
                        val = val == '1'
        except:
            val = self.handle_error(key_name, val, default_value)
            if isinstance(val, str):
                val = val == '1'
        return val
    
    
    def read_int_in_configuration(self, key_name, default_value):
        # type: (str, Any) -> int
        val = getattr(self._configuration, key_name, default_value)
        try:
            val = int(val)
        except:
            val = int(self.handle_error(key_name, val, default_value))
        return val
    
    
    def read_float_in_configuration(self, key_name, default_value):
        # type: (str, Any) -> float
        val = getattr(self._configuration, key_name, default_value)
        try:
            val = float(val)
        except:
            val = float(self.handle_error(key_name, val, default_value))
        return val
    
    
    def read_string_in_configuration(self, key_name, default_value):
        # type: (str, Any) -> str
        val = getattr(self._configuration, key_name, default_value)
        if not isinstance(val, str):
            val = self.handle_error(key_name, val, default_value)
        return val
    
    
    def read_list_in_configuration(self, key_name, default_value):
        # type: (str, Any) -> List
        val = getattr(self._configuration, key_name, default_value)
        try:
            if isinstance(val, str):
                raw_list = val.split(',')
            else:
                raw_list = val
            val = [i.strip() if isinstance(i, str) else i for i in raw_list]
        except:
            val = self.handle_error(key_name, val, default_value)
            if isinstance(val, str):
                raw_list = val.split(',')
                val = [i.strip() if isinstance(i, str) else i for i in raw_list]
        return val
    
    
    def read_modules_names_in_configuration(self, key_name, default_value):
        # type: (str, List[str]) -> List[Optional[str]]
        modules = getattr(self._configuration, key_name)
        try:
            val = [m.module_name for m in modules]
        except:
            val = self.handle_error(key_name, modules, default_value)
        return val
    
    
    def handle_error(self, key_name, value, default_value):
        # type: (str, Any, Any) -> Any
        if self._logger:
            if isinstance(value, list) and not isinstance(default_value, list):
                self._logger.error('〖 %s 〗 parameter is duplicated. We will use the default value 〖 %s 〗' % (key_name, default_value))
            else:
                self._logger.error('〖 %s 〗 has incorrect value 〖 %s 〗. We will use the default value 〖 %s 〗' % (key_name, value, default_value))
        self._configuration_error[key_name] = {'conf_value': value, 'default_value': default_value}
        return default_value
    
    
    def get_logger(self):
        # type: () -> PartLogger
        return self._logger
    
    
    def after_fork_cleanup(self):
        # type: () -> None
        self._logger = None
        self._configuration = None
        self._configuration_format = None
        self._configuration_error = {}
        self._configuration_load = {}
        self._readers = None
    
    
    def get_properties_from_loaded_configuration(self):
        # type: () -> List[Optional[Tuple[str,Any,None]]]
        properties_list = []
        for prop_name, _ in self._configuration_load.values():
            # LoggerFactory.get_logger().get_sub_part('CONFIGURATION').get_sub_part('MODULE INFO').debug(u'%s PROP:%s VAL:%s' % (getattr(self._configuration, 'module_name', 'NO NAME MODULE'), prop_name, getattr(self, prop_name) if prop_name else u'--'))
            
            if not prop_name:
                continue
            properties_list.append((prop_name, getattr(self, prop_name), None))
        return properties_list
    
    
    @staticmethod
    def configuration_format_jsonify(configuration_format):
        # type: (List) -> str
        class ConfigurationFormatEnc(json.JSONEncoder):
            def default(self, obj):
                if obj.key_type == TypeConfiguration.SEPARATOR:
                    return [obj.configuration_key, obj.key_type]
                else:
                    return [obj.configuration_key, obj.default_value, obj.key_type, obj.prop_name]
        
        return json.dumps(configuration_format, cls=ConfigurationFormatEnc)
    
    
    @staticmethod
    def get_broker_name_from_arbiter_configuration(configuration):
        # type: (ShinkenModuleDefinition) -> str
        broker_name = 'broker'
        if hasattr(configuration, 'father_config'):
            broker_name = configuration.father_config.get('broker_name', broker_name)
        if not broker_name.startswith('shinken'):
            broker_name = 'shinken-%s' % broker_name
        return broker_name


class ConfigurationReader(ConfigurationReaderMixin):
    
    @staticmethod
    def configuration_format_from_loaded_json(configuration_list, logger=None):
        # type: (List, Optional[PartLogger]) -> List
        configuration_format = []
        if not configuration_list:
            return configuration_format
        for element in configuration_list:
            if len(element) == 2:
                if element[1] == TypeConfiguration.SEPARATOR:
                    configuration_format.append(SeparatorFormat(element[0]))
                elif logger:
                    logger.error('wrong configuration parameter format ignored ( %s )' % element)
                continue
            if len(element) != 4:
                if logger:
                    logger.error('wrong configuration parameter format ignored ( %s )' % element)
                continue
            configuration_format.append(ConfigurationFormat(configuration_keys=element[0] if isinstance(element[0], list) else [element[0]],
                                                            default_value=element[1],
                                                            key_type=element[2],
                                                            prop_name=element[3]))
        
        return configuration_format
    
    
    @staticmethod
    def load_all_modules_infos(modules_dir, modules_logger):
        # type: (str, PartLogger) -> Dict[str,Dict]
        if not os.path.isdir(modules_dir):
            modules_logger.warning('[ %s ] is not a directory' % modules_dir)
            return {}
        modules_list = [module_dir_name for module_dir_name in os.listdir(modules_dir) if os.path.isdir(os.path.join(modules_dir, module_dir_name))]
        all_modules_info = {}
        already_load_module_info = {}
        for module_dir in modules_list:
            module_info_path = os.path.join(modules_dir, module_dir, 'module_info.json')
            if os.path.isfile(module_info_path):
                try:
                    with open(module_info_path) as json_file:
                        module_info = json.load(json_file)
                    
                    module_types = set()
                    
                    if 'type' in module_info:
                        module_types.add(module_info['type'])
                    elif 'module_types_by_daemons' in module_info:
                        for module_info_by_daemon in module_info['module_types_by_daemons']:
                            module_type_by_daemon = module_info_by_daemon.get('module_type', None)
                            if module_type_by_daemon:
                                module_types.add(module_type_by_daemon)
                    
                    if not module_types:
                        modules_logger.error('Missing "type" data in module_info.json in directory %s' % module_dir)
                        continue
                    for module_type in module_types:
                        if module_type in all_modules_info:
                            modules_logger.error('Ignoring duplicate module_info for module type %s found in directory %s and keeping data from directory %s' % (module_type, module_dir, already_load_module_info[module_type]))
                            continue
                        all_modules_info.update({module_type: module_info})
                        already_load_module_info.update({module_type: module_dir})
                except:
                    modules_logger.print_stack()
        return all_modules_info
    
    
    @staticmethod
    def update_all_modules_with_module_info(module_dir, modules):
        # type: (str, Iterable[ShinkenModuleDefinition]) -> None
        modules_logger = LoggerFactory.get_logger().get_sub_part('CONFIGURATION').get_sub_part('MODULE INFO')
        
        all_module_info = ConfigurationReader.load_all_modules_infos(module_dir, modules_logger)
        if not all_module_info:
            return
        
        for module in modules:
            module_type = getattr(module, 'module_type', None)
            if module_type not in all_module_info:
                continue
            
            module_info = all_module_info[module_type]
            configuration_errors = {}
            updated_properties = []
            
            standard_parameter_format = ConfigurationReader.configuration_format_from_loaded_json(module_info.get('standard_configuration_parameters', None), logger=modules_logger)
            if standard_parameter_format:
                module_conf = ConfigurationReader(standard_parameter_format, module, logger=modules_logger)
                module_conf.read_configuration()
                updated_properties.extend(module_conf.get_properties_from_loaded_configuration())
                configuration_errors.update(module_conf._configuration_error)
            
            mongo_parameter_format = module_info.get('mongodb_configuration_parameters_prefix', None)
            if mongo_parameter_format:
                # Import here to avoid cyclic import
                from shinkensolutions.ssh_mongodb.mongo_conf import MongoConf
                
                module_conf = MongoConf(module, prefix_module_property=mongo_parameter_format, logger=modules_logger)
                updated_properties.extend(module_conf.get_properties_from_loaded_configuration())
                configuration_errors.update(module_conf._configuration_error)
            
            if updated_properties:
                updated_properties.append(('module_info_json', module_info, None))
                if configuration_errors:
                    updated_properties.append(('configuration_reading_errors', configuration_errors, None))
                module.set_some_properties(updated_properties)
