#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.ssh_mongodb.mongo_error import ShinkenMongoException

if TYPE_CHECKING:
    from shinkensolutions.ssh_mongodb.mongo_client import MongoClient
    from shinkensolutions.ssh_mongodb.mongo_collection import MongoCollection
    from shinkensolutions.ssh_mongodb.mongo_bulk import AbstractBulk


# is_connecting : When we auto_reconnect on connecting to the database function,
# we don't check if it was close (close connection is not an error when it tries to connect) and
# on error we don't need to recall _retry_connection (we already repeat the connecting function in try part)
def retry_on_auto_reconnect(is_connecting: bool = False):
    def decorator(func):
        def wrapper(self: 'MongoClient|MongoCollection|AbstractBulk', *args, **kwargs):
            left_round = self._auto_reconnect_max_try
            collection_name = getattr(self, '_collection_name', '')
            if collection_name:
                operation_name = '%s on %s' % (func.__name__, collection_name)
            else:
                operation_name = '%s' % func.__name__
            
            while True:
                
                try:
                    if not is_connecting and self.is_connection_closed():
                        raise Exception('Connection to mongo was closed.')
                    return func(self, *args, **kwargs)
                except (ShinkenMongoException, StopIteration):
                    # White-listed exceptions, just re-raise them
                    raise
                except Exception as e:
                    if left_round <= 1:
                        error_msg = 'Mongo raised %s( %s ) on the operation %s. Operation failed : %s/%s. We tried %s times but it kept failing.' % (
                            type(e).__name__, e, operation_name, self._auto_reconnect_max_try - left_round + 1, self._auto_reconnect_max_try, self._auto_reconnect_max_try)
                        self.logger.error(error_msg)
                        raise ShinkenMongoException(error_msg)
                    else:
                        self.logger.info('Mongo raised %s( %s ) on the operation %s. Operation failed : %s/%s' % (type(e).__name__, e, operation_name, self._auto_reconnect_max_try - left_round + 1, self._auto_reconnect_max_try))
                    time.sleep(self._auto_reconnect_sleep_between_try)
                    if not is_connecting:
                        self._retry_connection()
                    left_round -= 1
        
        
        return wrapper
    
    
    return decorator
