#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2024:
# This file is part of Shinken Enterprise, all rights reserved.

import os

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import malloc_trim_lock

if TYPE_CHECKING:
    import select
    import selectors

    # noinspection PyPackageRequirements, PyProtectedMember
    from cheroot.connections import _ThreadsafeSelector as cherootSelector

cheroot_selector_list: 'list[cherootSelector]' = []

if os.name == 'nt':
    def fork_setup() -> None:
        pass
else:
    import psutil
    
    from shinken.log import LoggerFactory, logger
    from shinkensolutions.ssh_mongodb.sshtunnelmongomgr import mongo_by_ssh_mgr
    
    FORK_SETUP_IS_REGISTERED = False
    
    
    def before_fork_setup() -> None:
        if logger.is_debug():
            _logger = LoggerFactory.get_logger('FORK_CLEANUP').get_sub_part('BEFORE', register=False).get_sub_part(f'{os.getpid()}')
            mongo_by_ssh_mgr.set_fork_logger(_logger)
        
        mongo_by_ssh_mgr.before_fork_cleanup()
        malloc_trim_lock.acquire()
        logger.pre_fork_lock_all_handlers()
    
    
    def after_fork_in_parent_setup() -> None:
        logger.post_fork_unlock_all_handlers()
        
        if logger.is_debug():
            _logger = LoggerFactory.get_logger('FORK_CLEANUP').get_sub_part('AFTER', register=False).get_sub_part(f'{os.getpid()}')
            mongo_by_ssh_mgr.set_fork_logger(_logger)
        
        mongo_by_ssh_mgr.after_fork_cleanup_in_parent()
        malloc_trim_lock.release()
        mongo_by_ssh_mgr.reset_fork_logger()
    
    
    def after_fork_in_child_setup() -> None:
        logger.post_fork_unlock_all_handlers()
        _logger = None
        
        if logger.is_debug():
            _logger = LoggerFactory.get_logger('FORK_CLEANUP').get_sub_part('AFTER', register=False).get_sub_part(f'{os.getppid()}>>{os.getpid()}')
            mongo_by_ssh_mgr.set_fork_logger(_logger)
        
        for cheroot_selector in cheroot_selector_list:
            try:
                cheroot_selector: 'cherootSelector'
                # noinspection PyProtectedMember
                cheroot_selector._selector: 'selectors.EpollSelector'
                # noinspection PyProtectedMember
                cheroot_selector._selector._selector: 'select.epoll'
                
                if _logger:
                    # noinspection PyProtectedMember
                    _logger.debug(f'cheroot releasing epoll context fd [{cheroot_selector._selector.fileno()}] on {cheroot_selector._selector}/{cheroot_selector._selector._selector}')

                # noinspection PyProtectedMember
                cheroot_selector._selector.close()
                # noinspection PyProtectedMember
                del cheroot_selector._selector
                # noinspection PyProtectedMember
                cheroot_selector._lock.release()
                if _logger:
                    _logger.debug(f'released cheroot_selector lock {cheroot_selector}')
            except Exception as e:
                if _logger:
                    _logger.debug(f'Released cheroot_selector lock {cheroot_selector} failed with error {e}')
                continue
        
        if cheroot_selector_list:
            cheroot_selector_list.clear()
        
        # Close TCP sockets
        p = psutil.Process()
        for connection in p.net_connections(kind='inet') if hasattr(p, 'net_connections') else p.connections(kind='inet'):
            if _logger:
                _logger.debug(f'Closing {connection}')
            try:
                os.close(connection.fd)
            except Exception as e:
                if _logger:
                    _logger.debug(f'Closing {connection} failed with error {str(e)}')
        
        mongo_by_ssh_mgr.after_fork_cleanup_in_child()
        malloc_trim_lock.release()
        mongo_by_ssh_mgr.reset_fork_logger()
    
    
    def fork_setup() -> None:
        global FORK_SETUP_IS_REGISTERED
        
        if not FORK_SETUP_IS_REGISTERED:
            FORK_SETUP_IS_REGISTERED = True
            os.register_at_fork(before=before_fork_setup, after_in_parent=after_fork_in_parent_setup, after_in_child=after_fork_in_child_setup)
