#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import time
import uuid

from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.objects.schedulingitem import SchedulingItem


class CONTEXT_ACKNOWLEDGE_AUTO_DELETION:
    CFG_VALUE_ON_OK = 'on_OK'
    CFG_VALUE_ON_ANY_CHANGE = 'on_status_change'
    ON_OK = 2
    ON_ANY_CHANGE = 1


class Acknowledge:
    # Allows you to acknowledge the current problem for the specified host/service.
    # By acknowledging the current issue, future notifications (for the same host/service state) are disabled.
    
    # List the properties we will send as pickle to others daemons. Put all properties but no the one this reference.
    properties = {
        'id'                              : None,
        'sticky'                          : None,
        'notify'                          : None,
        'end_time'                        : None,
        'start_time'                      : None,
        'author'                          : None,
        'comment'                         : None,
        'automatic'                       : None,
        'state_when_has_been_acknowledged': None,
    }
    
    
    # If the "sticky" option is set to one (1), the acknowledgement
    # will remain until the service returns to an OK state. Otherwise,
    # the acknowledgement will automatically be removed when the
    # service changes state. In this case Web interfaces set a value
    # of (2).
    #
    # If the "notify" option is set to one (1), a notification will be
    # sent out to contacts indicating that the current service problem
    # has been acknowledged.
    #
    # <WTF??>
    # If the "persistent" option is set to one (1), the comment
    # associated with the acknowledgement will survive across restarts
    # of the Shinken process. If not, the comment will be deleted the
    # next time Shinken restarts. "persistent" not only means "survive
    # restarts", but also
    #
    # => End of comment Missing!!
    # </WTF??>
    
    # SEF-10454 : when switching from old style python class to new style python class, Scheduler retention load will crash if __init__ parameters do not have a default value
    # noinspection PyUnusedLocal
    def __init__(
            self,
            ref: 'SchedulingItem|None' = None,
            sticky: 'int|None' = None,
            notify: 'int|None' = None,
            persistent: 'int|None' = None,
            author: 'str|None' = None,
            comment: 'str|None' = None,
            end_time: int = 0,
            automatic: bool = False
    ) -> None:
        # get the current uuid part
        self.id = uuid.uuid1().hex
        self.ref = ref  # pointer to srv or host we are applied
        self.sticky = sticky
        self.notify = notify
        self.end_time = end_time
        self.author = author
        self.comment = comment
        self.start_time = int(time.time())
        self.automatic = automatic
        self.state_when_has_been_acknowledged = ref.state
    
    
    # Call by pickle to serialize the acknowledgement
    # because we DO NOT WANT REF in this pickled data !
    def __getstate__(self):
        cls = self.__class__
        # id is not in *_properties
        res = {'id': self.id}
        for prop in cls.properties:
            if hasattr(self, prop):
                res[prop] = getattr(self, prop)
        return res
    
    
    # Inverted function of getstate
    def __setstate__(self, state):
        cls = self.__class__
        self.id = state['id']
        for prop in cls.properties:
            if prop in state:
                setattr(self, prop, state[prop])
        # If loading an old ack, set the end_time to 0 which refers to infinite
        if not hasattr(self, 'end_time'):
            self.end_time = 0
        if not hasattr(self, 'start_time'):
            self.start_time = 0
        if not hasattr(self, 'automatic'):
            self.automatic = False
        if not hasattr(self, 'state_when_has_been_acknowledged'):
            self.state_when_has_been_acknowledged = 'PENDING'
    
    
    def __str__(self):
        return 'Acknowledge([%s] on [%s] automatic:[%s] author:[%s] comment:[%s]' % (
            self.id,
            self.ref.get_full_name() if hasattr(self, 'ref') and hasattr(self.ref, 'get_full_name') else str(self.ref) if hasattr(self, 'ref') else 'NO-REF',
            self.automatic,
            self.author,
            self.comment,
        )
