#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#     Gabes Jean, naparuba@gmail.com
#     Gerhard Lausser, Gerhard.Lausser@consol.de
#     Gregory Starck, g.starck@gmail.com
#     Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import uuid


class InvalidTokenError(Exception):
    pass


class BROKS_EXCHANGE_PROTOCOLS:
    NO_LIMIT_AND_NO_ACK_PROTOCOL = 1
    BY_BROK_SIZE_ESTIMATION_LIMITED_DATA_SIZE_PROTOCOL = 2
    SPLIT_FULL_DATA_PROTOCOL = 3


class NetworkExchangeSequencer:
    """Holds data together with an acknowledgment token, to ensure that data has been received."""
    
    
    def __init__(self):
        self._sequence_number = None
        self.data = None
        self.protocol = 0
        self.brok_count = 0
        
    
    def get_sequence_number(self):
        return self._sequence_number
    
    
    def get_unacknowledged_data(self):
        return self.protocol, self.data, self.brok_count
    
    
    def _reset(self):
        self._sequence_number = uuid.uuid4().hex
    
    
    def ack(self, token):
        if token == self._sequence_number:
            data = self.data
            protocol = self.protocol
            brok_count = self.brok_count
            self._sequence_number = None
            self.data = None
            self.protocol = 0
            self.brok_count = 0
            return protocol, data, brok_count
        else:
            raise InvalidTokenError
    
    
    def push(self, data, protocol, brok_count):
        self._reset()
        self.data = data
        self.protocol = protocol
        self.brok_count = brok_count
        return self._sequence_number
    
    
    def have_data(self):
        return self.data is not None
    
    
    def is_valid_sequence_number(self, sequence_number):
        return sequence_number != '' and self._sequence_number == sequence_number
