#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

import json

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict, Union, Collection
    from shinken.acknowledge import Acknowledge
    from shinken.downtime import Downtime
    from shinken.objects.service import Service
    from shinken.objects.host import Host


def peek_in_set(s):
    e = None
    for e in s:
        break
    return e


def get_elt_dict(elt, skip_properties=()):
    # type: (List|Acknowledge|Downtime, Collection) -> Union[List, Dict]
    if isinstance(elt, list):
        list_tmp = []
        for e in elt:
            _d = get_elt_dict(e, skip_properties)
            if _d:
                list_tmp.append(_d)
        return list_tmp
    cls = elt.__class__
    couples = [(k, v) for (k, v) in getattr(cls, 'properties', {}).items()] + [(k, v) for (k, v) in getattr(cls, 'running_properties', {}).items()]
    d = {}
    for (k, v) in couples:
        try:
            if k in skip_properties:
                continue
            else:
                value = getattr(elt, k)
            json.dumps(value)
            d[k] = value
        except Exception as exp:  # if cannot json, pass
            if k in ('output', 'long_output'):
                d[k] = 'The output is in a invalid format. It cannot be displayed.'
            # We don't print the value because if this no json-able it may break the logger.
            logger.debug('Invalid properties in [%s]-[%s] [%s] exp:[%s]' % (cls, str(elt), k, exp))
    
    if hasattr(elt, 'id') and isinstance(elt.id, int):
        d['id'] = elt.id
    
    return d


def get_downtime_on_item(item, elt_dict):
    # type: (Union[Host, Service], Dict) -> None
    elt_dict['in_scheduled_downtime'] = item.is_in_downtime()
    elt_dict['in_inherited_downtime'] = item.is_in_inherited_downtime()
    elt_dict['downtimes'] = get_elt_dict(item.downtimes)
    
    if item.__class__.my_type == 'service':
        host_downtimes = get_elt_dict(item.host.downtimes)
        elt_dict['downtimes_inherited'] = host_downtimes


def get_ack_on_item(item, elt_dict):
    # type: (Union[Host, Service], Dict) -> None
    elt_dict['acknowledgement'] = {}
    elt_dict['partial_acknowledge'] = {}
    elt_dict['acknowledgement_inherited'] = {}
    elt_dict['problem_has_been_acknowledged'] = False
    elt_dict['problem_has_been_inherited_acknowledged'] = 0
    
    if item.__class__.my_type == 'service':
        if item.acknowledgement and not item.acknowledgement.automatic:
            elt_dict['acknowledgement'] = get_elt_dict(item.acknowledgement)
            elt_dict['problem_has_been_acknowledged'] = True
        if item.partial_acknowledge:
            elt_dict['partial_acknowledge'] = get_elt_dict(item.partial_acknowledge)
        if item.host.got_business_rule:
            if item.host.acknowledgement and not item.host.acknowledgement.automatic:
                # ACK inherited from cluster if cluster has been acknowledged
                elt_dict['acknowledgement_inherited'] = get_elt_dict(item.host.acknowledgement)
                elt_dict['problem_has_been_inherited_acknowledged'] = 1
                if item.host.partial_acknowledge:
                    elt_dict['partial_acknowledge'] = get_elt_dict(item.host.partial_acknowledge)
        elif item.host.acknowledgement:
            # ACK inherited from host
            elt_dict['acknowledgement_inherited'] = get_elt_dict(item.host.acknowledgement)
            elt_dict['problem_has_been_inherited_acknowledged'] = 1
            if item.host.partial_acknowledge:
                elt_dict['partial_acknowledge'] = get_elt_dict(item.host.partial_acknowledge)
    
    else:
        if item.acknowledgement and not item.acknowledgement.automatic:
            elt_dict['acknowledgement'] = get_elt_dict(item.acknowledgement)
            elt_dict['problem_has_been_acknowledged'] = True
        elif item.acknowledgement and item.acknowledgement.automatic:
            elt_dict['acknowledgement_inherited'] = get_elt_dict(item.acknowledgement)
            elt_dict['problem_has_been_inherited_acknowledged'] = 2
        if item.partial_acknowledge:
            elt_dict['partial_acknowledge'] = get_elt_dict(item.partial_acknowledge)
