#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import time
import uuid
from collections import defaultdict, Counter

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.component.abstract_component import AbstractComponent
from shinkensolutions.mongo_utils import append_to_filter

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Iterable, TypeVar, Sequence
    from shinken.log import PartLogger
    from shinkensolutions.ssh_mongodb.mongo_client import MongoClient
    from shinkensolutions.ssh_mongodb.mongo_collection import MongoCollection
    from shinken.objects.contact import Contact
    from shinkensolutions.webui.objects.view import View
    
    ViewVar = TypeVar('ViewVar', bound=View)


class ShareManagerComponent(AbstractComponent):
    _share_collection: 'MongoCollection'
    
    
    def __init__(self, logger, webui_mongo_client):
        # type: (PartLogger, MongoClient) -> None
        self._logger = logger.get_sub_part('SHARE MANAGER')
        self._webui_mongo_client = webui_mongo_client
    
    
    def init(self):
        self._share_collection = self._webui_mongo_client.get_collection('share')
    
    
    def is_viewable_by_user(self, user, view):
        # type: (Contact, View) -> bool
        find_filter = self._make_collection_filter_to_get_shares_for_a_given_user(user, include_personal=False)
        
        find_filter = append_to_filter(find_filter, {
            'screen.uuid': view.get_uuid(),
            'screen.type': view.get_view_type(),
        })
        
        return self._share_collection.find(find_filter, {'screen': 1}, only_count=True) > 0
    
    
    def filter_views_by_visibility(self, user: 'Contact', views: 'Sequence[ViewVar]') -> list['ViewVar']:
        if len(views) == 0:
            return []
        
        find_filter = self._make_collection_filter_to_get_shares_for_a_given_user(user, include_personal=False)
        
        find_filter = append_to_filter(find_filter, {
            '$or': [
                {
                    '$and': [
                        {
                            'screen.uuid': view.get_uuid(),
                        },
                        {
                            'screen.type': view.get_view_type(),
                        }
                    ]
                }
                for view in views
            ]
        })
        
        nb_shares_per_view: 'defaultdict[str, Counter[str]]' = self._count_number_of_shares_per_view_from_result(
            self._share_collection.find(find_filter, {'screen': 1})
        )
        
        return [
            view for view in views
            if nb_shares_per_view[view.get_view_type()][view.get_uuid()] > 0  # Do not use .get() ; defaultdict and Counter are designed not to fail on missing keys
        ]
    
    
    @classmethod
    def _make_collection_filter_to_get_shares_for_a_given_user(cls, user: 'Contact', *, include_personal: bool) -> dict[str, 'Any']:
        expected_types: list[dict[str, 'Any']] = [
            {'type': 'global'},
        ]
        
        user_groups = user.get_contactgroups()
        if user_groups:
            expected_types.append({
                '$and': [
                    {'type': 'group'},
                    {
                        'group.uuid': {'$in': [group.uuid for group in user_groups]},
                    },
                ]
            })
        if include_personal:
            expected_types.append({
                '$and': [
                    {'type': 'personal'},
                    {'owner.uuid': user.get_uuid()},
                ]
            })
        return {
            '$and': [
                {'state.removed': False},
                {'$or': expected_types},
            ]
        }
    
    
    @classmethod
    def _count_number_of_shares_per_view_from_result(cls, result: 'list[dict[str, Any]]') -> 'defaultdict[str, Counter[str]]':
        nb_shares: 'defaultdict[str, Counter[str]]' = defaultdict(Counter)
        
        for share in result:
            screen = share.get('screen', {})
            screen_uuid = screen.get('uuid', '')
            screen_type = screen.get('type', '')
            if screen_uuid and screen_type:
                # Do not use .get() ; defaultdict and Counter are designed not to fail on missing keys
                nb_shares[screen_type][screen_uuid] += 1
        return nb_shares
    
    
    def get_number_of_shares_by_view(self, view: 'View') -> int:
        return self._share_collection.find(
            {
                'state.removed': False,
                'screen.uuid'  : view.get_uuid(),
                'screen.type'  : view.get_view_type()
            },
            {'screen': 1},
            only_count=True
        )
    
    
    def get_number_of_shares_for_several_views(self, views: 'Sequence[View]') -> 'Counter[str]':
        if len(views) == 0:
            return Counter()
        
        find_filter = {
            '$and': [
                {
                    'state.removed': False,
                },
                {
                    '$or': [
                        {
                            '$and': [
                                {
                                    'screen.uuid': view.get_uuid(),
                                },
                                {
                                    'screen.type': view.get_view_type(),
                                }
                            ]
                        }
                        for view in views
                    ]
                }
            ]
        }
        result = self._share_collection.find(find_filter, {'screen': 1})
        nb_shares_per_view = self._count_number_of_shares_per_view_from_result(result)
        
        # Do not use .get() ; defaultdict and Counter are designed not to fail on missing keys
        return Counter({view.get_uuid(): nb_shares_per_view[view.get_view_type()][view.get_uuid()] for view in views})
    
    
    def get_personal_share_id(self, view_uuid):
        # type: (str) -> str
        share = self._share_collection.find({'state.removed': False, 'type': 'personal', 'screen.uuid': view_uuid})
        return share
    
    
    def get_personal_share_from_user(self, user: 'Contact') -> list:
        shares = self._share_collection.find({'state.removed': False, 'type': 'personal', 'user.uuid': user.uuid})
        return shares
    
    
    def remove_shares_for_views(self, views_uuid):
        # type: (Iterable[str]) -> None
        for view_uuid in views_uuid:
            self._share_collection.remove({'screen.uuid': view_uuid})
    
    
    def get_shares_in_groups_for_view(self, view_uuid: str) -> list:
        return self._share_collection.find({'state.removed': False, 'type': 'group', 'screen.uuid': view_uuid})
    
    
    def get_shares_for_view(self, view_uuid):
        # type: (str) -> list
        return self._share_collection.find({'state.removed': False, 'type': 'group', 'screen.uuid': view_uuid})
    
    
    def change_owner_of_personal_share(self, view_uuid: str, user: 'Contact') -> None:
        self._share_collection.update({'type': 'personal', 'screen.uuid': view_uuid}, update={'$set': {'owner.uuid': user.uuid, 'owner.name': user.contact_name}}, upsert=True)
    
    
    def create_view_share(self, share_type: str, user: 'Contact', view: 'View', is_reference: bool) -> None:
        share_uuid = uuid.uuid4().hex
        created_on = time.time()
        self._share_collection.insert_many([{
            '_id'              : share_uuid,
            'uuid'             : share_uuid,
            'name'             : view.get_name(),
            'type'             : share_type,
            'index'            : len(self.get_personal_share_from_user(user)) + 1,
            'kind'             : 1,
            'screen'           : {
                'uuid': view.get_uuid(),
                'type': view.get_view_type()
            },
            'owner'            : {
                'uuid': user.uuid,
                'name': user.contact_name,
            },
            'screenIsReference': is_reference,
            'created_on'       : created_on,
            'last_modified_on' : created_on,
            'state'            : {
                'removed': False,
            }
        }])
