#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import time

from shinken.misc.type_hint import TYPE_CHECKING, cast
from shinken.thread_helper import Thread
from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriver, AbstractDataHubDriverConfig
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Any, Tuple, ContextManager
    from shinken.log import PartLogger
    from shinkensolutions.data_hub.data_hub import DataHubConfig


class DataHubMetaDriverConfigCleanupDirectory(AbstractDataHubDriverConfig):
    # Set it to True in order not to spawn the periodic cleanup thread
    disable_periodic_cleanup_thread = False  # type: bool
    
    
    def __init__(self, config_main_driver, cleanup_interval):
        # type: (AbstractDataHubDriverConfig, int) -> None
        super(DataHubMetaDriverConfigCleanupDirectory, self).__init__('CLEANUP')
        self.config_main_driver = config_main_driver
        self.cleanup_interval = cleanup_interval


def data_hub_meta_driver_cleanup_directory_factory(logger, driver_config, data_hub_config):
    # type: (PartLogger, AbstractDataHubDriverConfig, DataHubConfig) -> DataHubMetaDriverCleanupDirectory
    
    driver_config = cast(DataHubMetaDriverConfigCleanupDirectory, driver_config)
    
    main_driver = DataHubFactory.build_driver(logger, driver_config.config_main_driver, data_hub_config)
    return DataHubMetaDriverCleanupDirectory(logger, driver_config, main_driver)


DataHubFactory.register_driver_factory(DataHubMetaDriverConfigCleanupDirectory, data_hub_meta_driver_cleanup_directory_factory)


class DataHubMetaDriverCleanupDirectory(AbstractDataHubDriver, Thread):
    def __init__(self, logger, driver_config, main_driver):
        # type: (PartLogger, DataHubMetaDriverConfigCleanupDirectory, AbstractDataHubDriver) -> None
        Thread.__init__(self)
        AbstractDataHubDriver.__init__(self, logger, driver_config)
        
        self.logger = self._logger
        self.main_driver = main_driver
        self.cleanup_interval = driver_config.cleanup_interval
        
        self._start_thread_at_initialization = not driver_config.disable_periodic_cleanup_thread
    
    
    def lock_context(self, data_id):
        # type: (str) -> ContextManager[Any]
        return self.main_driver.lock_context(data_id)
    
    
    def loop_turn(self):
        # type: () -> None
        nb_data_cleaned = 0
        start_time = time.time()
        for data_id in self.main_driver.get_all_data_id():
            try:
                with self.lock_context(data_id):
                    if (time.time() - self.main_driver.get_last_modification_date(data_id)) >= self.cleanup_interval:
                        self.main_driver.remove(data_id)
                        nb_data_cleaned += 1
            except:
                continue  # Item not found, can't delete it
        if nb_data_cleaned or time.time() - start_time >= 1:
            self._logger.log_perf(start_time, 'cleanup_directory', 'Cleanup of file older than %s done for %s %s' % (self.cleanup_interval, nb_data_cleaned, self._data_type), min_time=0,
                                  warn_time=1)
    
    
    def get_thread_name(self):
        # type: () -> str
        return 'DHub clean %s' % self._name
    
    
    def init(self):
        # type: () -> None
        self.main_driver.init()
        if self._start_thread_at_initialization:
            self.start_thread()
    
    
    def stop(self):
        # type: () -> None
        Thread.stop(self)
    
    
    def get_all_data_id(self):
        # type: () -> List[str]
        return self.main_driver.get_all_data_id()
    
    
    def get_all_data(self, log_error=True):
        # type: (bool) -> List[Any]
        return self.main_driver.get_all_data(log_error=log_error)
    
    
    def get_all_data_with_id(self, log_error=True):
        # type: (bool) -> List[Tuple[str, Any]]
        return self.main_driver.get_all_data_with_id(log_error=log_error)
    
    
    def find_data_id(self, filters):
        # type: (Any) -> List[str]
        return self.main_driver.find_data_id(filters)
    
    
    def find_data(self, filters, log_error=True):
        # type: (Any, bool) -> List[Any]
        return self.main_driver.find_data(filters, log_error=log_error)
    
    
    def find_data_with_id(self, filters, log_error=True):
        # type: (Any, bool) -> List[Tuple[str, Any]]
        return self.main_driver.find_data_with_id(filters, log_error=log_error)
    
    
    def is_data_correct(self, data_id):
        # type: (str) -> bool
        return self.main_driver.is_data_correct(data_id)
    
    
    def write(self, data_id, data):
        # type: (str, Any) -> None
        self.main_driver.write(data_id, data)
    
    
    def write_raw(self, data_id, data):
        # type: (str, Any) -> None
        return self.main_driver.write_raw(data_id, data)
    
    
    def read(self, data_id, log_error=True):
        # type: (str, bool) -> Any
        return self.main_driver.read(data_id, log_error=log_error)
    
    
    def read_all(self, data_id_list, log_error=True):
        # type: (List[str], bool) -> List[Any]
        return self.main_driver.read_all(data_id_list, log_error=log_error)
    
    
    def read_all_with_id(self, data_id_list, log_error=True):
        # type: (List[str], bool) -> List[Tuple[str, Any]]
        return self.main_driver.read_all_with_id(data_id_list, log_error=log_error)
    
    
    def read_raw(self, data_id, log_error=True):
        # type: (str, bool) -> Any
        return self.main_driver.read_raw(data_id, log_error=log_error)
    
    
    def remove(self, data_id):
        # type: (str) -> None
        self.main_driver.remove(data_id)
    
    
    def get_last_modification_date(self, data_id):
        # type: (str) -> int
        return self.main_driver.get_last_modification_date(data_id)
    
    
    def restore(self, data_id, data):
        # type: (str, Any) -> None
        self.main_driver.write(data_id, data)
    
    
    def sanitize(self, data_id, data):
        # type: (str, Any) -> None
        self.main_driver.write(data_id, data)
    
    
    def destroy(self):
        # type: () -> None
        self.main_driver.destroy()
    
    
    def check_right_access_issues_then_fix_them(self):
        # type: () -> None
        self.main_driver.check_right_access_issues_then_fix_them()
    
    
    def get_total_size(self):
        # type: () -> int
        return self.main_driver.get_total_size()
    
    
    def get_size_of(self, data_id):
        # type: (str) -> int
        return self.main_driver.get_size_of(data_id)
    
    
    def get_number_of_stored_data(self):
        # type: () -> int
        return self.main_driver.get_number_of_stored_data()
