#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.string_helper import split_and_strip_list

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional, Any, Set, Callable


class Index:
    
    def __init__(self, property_name, getter, on_list):
        # type: (str, Optional[Callable[[Dict], Any]], bool) -> None
        self.property_name = property_name
        self.getter = None  # type: Optional[Callable[[Dict], Any]]
        if callable(getter):
            self.getter = getter
        self.index = {}  # type: Dict[Any, Set[int]]
        self.reverse_index = {}  # type: Dict[int, Set[Any]]
        self.on_list = on_list
    
    
    def add_index(self, indexed_collection):
        # type: (Dict[int, Dict]) -> None
        for key_index, item in indexed_collection.items():
            self.index_item(item, key_index)
    
    
    def index_item(self, item, key_index):
        # type: (Dict, int) -> None
        value_to_index = self._get_value(item, self.property_name) if self.getter is None else self.getter(item)
        
        if self.on_list:
            for v in value_to_index:
                self.index.setdefault(v, set()).add(key_index)
                self.reverse_index.setdefault(key_index, set()).add(v)
        else:
            self.index.setdefault(value_to_index, set()).add(key_index)
            self.reverse_index.setdefault(key_index, set()).add(value_to_index)
    
    
    def unindex_item(self, key_index):
        # type: (int) -> None
        all_values = self.reverse_index.pop(key_index, set())
        for value in all_values:
            self.index[value].discard(key_index)
    
    
    def find(self, value_search):
        # type: (Any) -> Set[int]
        return self.index.get(value_search, set())
    
    
    def _get_value(self, item, property_name):
        # type: (Dict, str) -> Any
        p, dot, rest = property_name.partition('.')
        if dot:
            return self._get_value(item.get(p, {}), rest)
        
        value = item.get(p, '')
        
        if self.on_list:
            value = split_and_strip_list(value)
        return value
