# !/usr/bin/env python3-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

# Parser Algorithm used https://en.wikipedia.org/wiki/Shunting_yard_algorithm

import re

from shinken.log import logger
from shinken.util import strip_and_uniq


class NODE_TYPE:
    TEMPLATE_OP = 'template'
    AND_OP = '&'
    OR_OP = '|'


class ArbiterComplexExpressionNode:
    def __init__(self, operand: None | str = None, sons: list | None = None, configuration_errors: list | None = None, not_value: bool = False, is_leaf: bool = False, content: None | str = None, items: list[str] | None = None):
        self.operand = operand
        if sons is None:
            self.sons = []
        else:
            self.sons = sons
        
        if items is None:
            self.items = []
        else:
            self.items = items
        
        if configuration_errors is None:
            self.configuration_errors = []
        else:
            self.configuration_errors = configuration_errors
        
        if content is None:
            self.content = ''
        else:
            self.content = content
        
        self.not_value = not_value
        self.is_leaf = is_leaf
    
    
    def __str__(self):
        if not self.is_leaf:
            return "Op:'%s' Leaf:%s Sons:'[%s] IsNot:%s'" % (self.operand, self.is_leaf, ','.join([str(s) for s in self.sons]), self.not_value)
        else:
            return 'IS LEAF %s IsNot:%s' % (self.items, self.not_value)
    
    
    def resolve_elements(self, is_root: bool = False) -> (set[str], set[str]):
        
        if self.is_leaf:
            if self.not_value:
                return set(), set(self.items)
            else:
                return set(self.items), set()
        
        not_items = None
        items = None
        for son in self.sons:
            son_items, son_not_items = son.resolve_elements()
            if self.operand == '|':
                if items is None:
                    items = son_items
                else:
                    items = items.union(son_items)
                
                if not_items is None:
                    not_items = son_not_items
                else:
                    not_items = not_items.intersection(son_not_items)
            elif self.operand == '&':
                
                if not son.not_value:
                    if items is None:
                        items = son_items
                    else:
                        items = items.intersection(son_items)
                
                if not_items is None:
                    not_items = son_not_items
                else:
                    not_items = not_items.union(son_not_items)
        
        if is_root:
            return items - not_items, set()
        
        return items, not_items


class COMPLEX_EXPRESSION_CONTEXTS:
    HOSTGROUPS: str = 'hostgroups'
    TEMPLATES: str = 'templates'
    ONLY_TEMPLATES: str = 'only_templates'


_ALLOWED_CONTEXTS = (
    COMPLEX_EXPRESSION_CONTEXTS.HOSTGROUPS,
    COMPLEX_EXPRESSION_CONTEXTS.TEMPLATES,
    COMPLEX_EXPRESSION_CONTEXTS.ONLY_TEMPLATES
)


class ComplexExpressionParser:
    
    OPERAND = '&+|,!'
    OPERAND_PRECEDENCE = {'|': 0, ',': 0, '&': 1, '+': 1, '!': 2}
    # 0 left, 1 : right
    OPERAND_ASSOCIATIVITY = {'|': 0, ',': 0, '&': 0, '+': 0, '!': 1}
    SUBEXPRESSION_DELIMITER = '()'
    
    
    @staticmethod
    def tokenize_expression(expression: str) -> list[str]:
        pattern = fr'([{ComplexExpressionParser.OPERAND}{ComplexExpressionParser.SUBEXPRESSION_DELIMITER}])'
        return [token.strip() for token in re.split(pattern, expression) if token.strip() != '']
    
    
    @staticmethod
    def _validate_token(previous_token: str | None, current_token: str | None, errors: list[str], opening_parenthesis: [int], closing_parenthesis: [int]):
        if current_token == '(':
            opening_parenthesis[0] += 1
        if current_token == ')':
            closing_parenthesis[0] += 1
        if previous_token is None:
            if current_token in '&+|,)':
                errors.append('validator.servicehosttpls_complex_expression_starts_with_operator')
        
        elif current_token is None:
            if previous_token in '&+|,!(':
                errors.append('validator.servicehosttpls_complex_expression_ends_with_operator')
            if opening_parenthesis[0] > closing_parenthesis[0]:
                errors.append('validator.servicehosttpls_complex_expression_too_many_opening_parenthesis')
            elif opening_parenthesis[0] < closing_parenthesis[0]:
                errors.append('validator.servicehosttpls_complex_expression_too_many_closing_parenthesis')
        
        else:
            if previous_token in '&+|,(' and current_token in '&+|,)':
                errors.append(f'validator.servicehosttpls_complex_expression_two_successive_operators:{previous_token}:{current_token}')
            if previous_token == '!' and current_token in '&+|,)':
                errors.append(f'validator.servicehosttpls_complex_expression_two_successive_operators:{previous_token}:{current_token}')
            if previous_token not in '&+|,(!' and current_token == '!':
                errors.append('validator.servicehosttpls_complex_expression_missing_operator_before_exclamation_point')
            if previous_token not in '&+|,!(' and current_token == '(':
                errors.append('validator.servicehosttpls_complex_expression_missing_operator_before_opening_parenthesis')
            if previous_token == ')' and current_token not in '&+|,)':
                errors.append('validator.servicehosttpls_complex_expression_missing_operator_after_closing_parenthesis')
    
    
    @staticmethod
    def _is_greater_precedence(operand_1: str, operand_2: str) -> bool:
        return ComplexExpressionParser.OPERAND_PRECEDENCE[operand_1] >= ComplexExpressionParser.OPERAND_PRECEDENCE[operand_2]
    
    
    @staticmethod
    def _associativity(operand: str) -> str:
        return ComplexExpressionParser.OPERAND_ASSOCIATIVITY[operand]
    
    
    def _update_stacks(self, _operator_stack: list[str], _value_stack: list[dict[str:str | list | None]]) -> None:
        raise NotImplementedError
    
    
    @staticmethod
    def _make_node(operand: None | str | NODE_TYPE = None, sons: None | list = None, not_value: bool = False, content: str | None = None, items: list[str] | None = None) -> dict[str, str | bool | list[dict] | None] | ArbiterComplexExpressionNode:
        raise NotImplementedError
    
    
    def _make_leaf_node(self, content: str = None) -> dict[str, str | bool | list[dict] | None] | ArbiterComplexExpressionNode:
        raise NotImplementedError
    
    
    def parse_complex_exp(self, expression: str) -> dict[str, str | bool | list[dict] | None]:
        raise NotImplementedError
    
    
    def _check_invalid_tree_with_only_negated_templates(self, node: dict[str, str | bool | list[dict] | None] | ArbiterComplexExpressionNode) -> list[bool]:
        raise NotImplementedError
    
    
    def _optimize_tree(self, node: dict[str, str | bool | list[dict] | None] | ArbiterComplexExpressionNode) -> None:
        raise NotImplementedError
    
    
    def _parse_tokenized_value(self, tokens: list[str], errors: list[str]) -> dict | ArbiterComplexExpressionNode:
        # Parser Algorithm used https://en.wikipedia.org/wiki/Shunting_yard_algorithm
        value_stack = []
        operator_stack = []
        previous_token = None
        opening_parenthesis = [0]
        closing_parenthesis = [0]
        for token in tokens:
            self._validate_token(previous_token, token, errors, opening_parenthesis, closing_parenthesis)
            
            if token in ComplexExpressionParser.OPERAND:
                if not operator_stack or (operator_stack and operator_stack[-1] == '('):
                    operator_stack.append(token)
                
                elif operator_stack and self._is_greater_precedence(token, operator_stack[-1]) and self._associativity(token) == 1:
                    operator_stack.append(token)
                
                else:
                    while operator_stack and not operator_stack[-1] in ComplexExpressionParser.SUBEXPRESSION_DELIMITER and self._is_greater_precedence(operator_stack[-1], token) and self._associativity(token) == 0:
                        self._update_stacks(operator_stack, value_stack)
                    operator_stack.append(token)
            
            elif token == '(':
                operator_stack.append(token)
            
            elif token == ')':
                while operator_stack and operator_stack[-1] != '(':
                    self._update_stacks(operator_stack, value_stack)
                if operator_stack:
                    operator_stack.pop()
            else:
                value_stack.append(self._make_leaf_node(content=token))
            previous_token = token
        self._validate_token(previous_token, None, errors, opening_parenthesis, closing_parenthesis)
        while operator_stack:
            self._update_stacks(operator_stack, value_stack)
        return value_stack.pop()


class ArbiterComplexExpressionParser(ComplexExpressionParser):
    
    context = COMPLEX_EXPRESSION_CONTEXTS.HOSTGROUPS
    all_groups = None
    all_elements = None
    
    
    def __init__(self, context=COMPLEX_EXPRESSION_CONTEXTS.HOSTGROUPS, all_groups=None, all_elements=None):
        if context not in _ALLOWED_CONTEXTS:
            raise Exception('The context %s is not a valid one' % context)
        
        self.context = context
        self.all_groups = all_groups
        self.all_elements = all_elements
    
    
    def _update_stacks(self, _operator_stack: list[str], _value_stack: list[dict[str:str | list | None]]) -> None:
        popped_operator = _operator_stack.pop()
        if popped_operator == '!':
            node = _value_stack.pop()
            node.not_value = not node.not_value
        else:
            if _value_stack[-2].operand == popped_operator and not _value_stack[-2].not_value:
                node_son = _value_stack.pop()
                node = _value_stack.pop()
                node.sons.append(node_son)
            else:
                node = self._make_node(popped_operator)
                node.sons = [_value_stack.pop(), _value_stack.pop()]
                node.sons.reverse()
        _value_stack.append(node)
    
    
    @staticmethod
    def _make_node(operand: None | str = None, sons: None | list = None, not_value: bool = False, content: str = None, configuration_errors: None | str = None, leaf: bool = False, items: list[str] | None = None) -> ArbiterComplexExpressionNode:
        return ArbiterComplexExpressionNode(operand=operand, sons=sons, configuration_errors=configuration_errors, not_value=not_value, is_leaf=leaf, content=content, items=items)
    
    
    def _make_leaf_node(self, content: None | str = None) -> ArbiterComplexExpressionNode:
        item_names, error = self.find_objects(content)
        if isinstance(error, str):
            error = [error]
        return self._make_node(operand=self.context, items=item_names, content=content, leaf=True, configuration_errors=error)
    
    
    def _optimize_tree(self, node: ArbiterComplexExpressionNode) -> None:
        
        if node.is_leaf:
            return
        
        if node.not_value:
            
            node.not_value = False
            node.operand = NODE_TYPE.OR_OP if node.operand == NODE_TYPE.AND_OP else NODE_TYPE.AND_OP
            for son in node.sons:
                son.not_value = not son.not_value
        
        for son in node.sons:
            self._optimize_tree(son)
        
        can_reduce_tree = True
        new_sons = []
        uniq_new_sons = []
        uniq_hash = set()
        for son in node.sons:
            son: ArbiterComplexExpressionNode
            if son.operand == node.operand:
                new_sons.extend(son.sons)
            elif son.operand in _ALLOWED_CONTEXTS:
                new_sons.append(son)
            else:
                can_reduce_tree = False
                break
        
        if can_reduce_tree:
            for son in new_sons:
                
                if son.operand in _ALLOWED_CONTEXTS:
                    node_hash = f'{son.not_value}-{son.content}'
                    if node_hash not in uniq_hash:
                        uniq_hash.add(node_hash)
                        uniq_new_sons.append(son)
                else:
                    uniq_new_sons.append(son)
            
            node.sons = uniq_new_sons
        
        return
    
    
    def _check_invalid_tree_with_only_negated_templates(self, node: ArbiterComplexExpressionNode) -> bool:
        if node.is_leaf:
            return node.not_value
        
        else:
            result = [self._check_invalid_tree_with_only_negated_templates(son) for son in node.sons]
            if node.operand == '|':
                if node.not_value:
                    return True not in result
                return True if True in result else False
            elif node.operand == '&':
                if node.not_value:
                    return True if False in result else False
                return False not in result
    
    
    def parse_complex_exp(self, expression: str, optimize_tree: bool = True) -> ArbiterComplexExpressionNode:
        
        tokens = self.tokenize_expression(expression)
        errors = []
        try:
            root_node = self._parse_tokenized_value(tokens, errors)
            if optimize_tree:
                self._optimize_tree(root_node)
            if self._check_invalid_tree_with_only_negated_templates(root_node):
                logger.error(f'Cannot parse the pattern "{expression}" because [validator.servicehosttpls_complex_expression_no_template_defined]')
        
        except Exception:
            
            if not errors:
                logger.error(f'Cannot parse the pattern "{expression}" because [validator.servicehosttpls_complex_expression_parsing_failed]')
                return ArbiterComplexExpressionNode()
            
            logger.error(f'Cannot parse the pattern "{expression}" because [{", ".join(errors)}]')
            return ArbiterComplexExpressionNode()
        
        if errors:
            logger.error(f'Cannot parse the pattern "{expression}" because [{", ".join(errors)}]')
            return ArbiterComplexExpressionNode()
        
        return root_node
    
    
    # We've got an object, like super-grp, so we should link th group here
    
    def find_objects(self, pattern: str) -> (list[str], None | str):
        error = None
        pattern = pattern.strip()
        
        if pattern == '*':
            item_names = [h.host_name for h in self.all_elements.items.values() if getattr(h, 'host_name', '') != '' and not h.is_tpl()]
            return item_names, error
        
        if self.context == COMPLEX_EXPRESSION_CONTEXTS.HOSTGROUPS:
            hg = self.all_groups.find_by_name(pattern)
            if not hg:
                error = "Error : cannot find the %s of the expression '%s'" % (self.context, pattern)
                return hg, error
            elts = hg.get_hosts().split(',')
            elts = strip_and_uniq(elts)
            
            if '*' in elts:
                elts.extend([h.host_name for h in self.all_elements.items.values() if getattr(h, 'host_name', '') != '' and not h.is_tpl()])
                elts.remove('*')
            return elts, error
        
        if self.context == COMPLEX_EXPRESSION_CONTEXTS.TEMPLATES:
            item_names = self.all_elements.find_hosts_that_use_template(pattern)
            return item_names, error
        
        if self.context == COMPLEX_EXPRESSION_CONTEXTS.ONLY_TEMPLATES:
            item_names = [pattern]
            return item_names, error
