#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import pickle
import multiprocessing
import time
from queue import Empty

from shinken.compat import SHINKEN_PICKLE_PROTOCOL
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Number, Optional
    from shinken.basesubprocess import ToModuleCommandRequest


class ShinkenQueue:
    def __init__(self, queue_name=None, father_name=None, child_name=None):
        self.queue = multiprocessing.Queue()
        # We do not want to lock for this queue, we don't care about it
        self.queue.cancel_join_thread()
        self.name = queue_name
        self.father_name = father_name
        self.child_name = child_name
    
    
    # SEF-10415: Cannot be used under Windows, will make multiprocessing fork crash
    # def __getattr__(self, item):
    #     return getattr(self.queue, item)
    
    def put(self, item, block=True, timeout=None):
        # type: (ToModuleCommandRequest, bool, Optional[int]) -> None
        self.queue.put(item, block=block, timeout=timeout)
    
    
    def get(self, block=True, timeout=None):
        # type: (bool, Optional[int]) -> None
        return self.queue.get(block=block, timeout=timeout)
    
    
    def close(self):
        self.queue.close()
        self.queue.join_thread()
    
    
    def send(self, message):
        try:
            self.queue.put(message)
        except IOError:
            # logger.warning('The windows queue seems to be close, will be detected in the next loop')
            raise IOError(11, 'queue do not seems to be available')
    
    
    # Has element?
    def poll(self):
        r = not self.queue.empty()
        return r
    
    
    @staticmethod
    def write_poll(timeout):
        # type: (Optional[Number]) -> bool
        if timeout:
            time.sleep(timeout)
        return True
    
    
    # Get message
    # WARNING: we did check the poll, but the python doc say maybe this will be void...
    def recv(self):
        try:
            raw_message = self.queue.get_nowait()
        except Empty:  # Fake object then
            raw_message = pickle.dumps(None, SHINKEN_PICKLE_PROTOCOL)
        return raw_message
    
    
    # For linux queue compat
    def get_queues_size(self):
        return self.queue.qsize()
