#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

""" This python module contains the class BaseModule
that shinken modules will subclass
"""

import datetime
import inspect
import os
import signal
import threading
import time
from datetime import datetime
from multiprocessing import Process

from shinken.basesubprocess import BaseSubProcess, CommandQueueHandler, UnknownCommandException
from shinken.ipc.shinken_queue.shinken_queue import build_shinken_queue, ShinkenQueue
from shinken.log import logger as _logger, PART_INITIALISATION, LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.module_info_reader import ModuleInfoReaderMixin
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.util import mem_wait_for_fork_possible

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.inter_daemon_message import InterDaemonMessage
    from shinken.misc.type_hint import List, Optional

# TODO: use a class for defining the module "properties" instead of plain dict??  Like:
'''
class ModuleProperties:
    def __init__(self, type, phases, external=False)
        self.type = type
        self.phases = phases
        self.external = external
'''
# and  have the new modules instantiate this like follow:
'''
properties = ModuleProperties('the_module_type', the_module_phases, is_mod_ext)
'''

# The properties' dict defines what the module can do and if it's an external module or not.
properties = {
    'type'    : None,  # name of the module type ; to distinguish between them:
    'external': True,  # is the module "external" (external means here a daemon module)?
    'phases'  : ['configuration', 'late_configuration', 'running', 'retention'],  # Possible configuration phases where the module is involved:
}


class ModulePhases:
    CONFIGURATION = 1
    LATE_CONFIGURATION = 2
    RUNNING = 4
    RETENTION = 8


class ModuleState:
    OK = 'OK'
    WARNING = 'WARNING'
    CRITICAL = 'CRITICAL'
    FATAL = 'FATAL'


MAX_MODULE_INIT_TRY_INTERVAL = 60


class INSTANCE_INIT_STATE:
    INIT_READY = 1
    INIT_SKIPPED = 2
    INIT_FAILED = 3
    INIT_SUCCESSFUL = 4


class SOURCE_STATE:
    PENDING = 'PENDING'
    RUNNING = 'RUNNING'
    OK = 'OK'
    CRITICAL = 'CRITICAL'
    WARNING = 'WARNING'
    UNKNOWN = 'UNKNOWN'
    CONFIGURATION_ERROR = 'CONFIGURATION_ERROR'
    NOT_CONFIGURED = 'not-configured'  # This state will block the source import because the import will change anything
    NOT_CONFIGURED_BUT_CAN_LAUNCH_IMPORT = 'not-configured-before-import'  # This state will launch the source import because the import can change the state
    READY_FOR_IMPORT = 'ready-for-import'
    NEED_CONFIRMATION = 'need-confirmation'
    NEVER_IMPORT = 'never-import'
    DIFFERENCE_COMPUTING = 'difference-computing'
    
    STATES_ORDER = (
        CONFIGURATION_ERROR,
        NEED_CONFIRMATION,
        NOT_CONFIGURED_BUT_CAN_LAUNCH_IMPORT,
        NOT_CONFIGURED,
        CRITICAL,
        WARNING,
        UNKNOWN,
        OK,
        PENDING,
        RUNNING,
        READY_FOR_IMPORT,
        NEVER_IMPORT,
        DIFFERENCE_COMPUTING,
    )
    
    
    @staticmethod
    def get_best_of_two_states(state1, state2):
        for s in SOURCE_STATE.STATES_ORDER:
            if s in (state1, state2):
                return s
        raise Exception('state [%s] [%s] not in STATES_ORDER' % (state1, state2))


class BaseModule(BaseSubProcess, ModuleInfoReaderMixin):
    """This is the base class for the shinken modules.
    Modules can be used by the different shinken daemons/services
    for different tasks.
    Example of task that a shinken module can do:
     - load additional configuration objects.
     - recurrently save hosts/services status/perfdata
       information in different format.
     - ...
     """
    
    is_worker_based = False
    
    
    def __init__(self, mod_conf):
        # type: (ShinkenModuleDefinition) -> None
        """Instantiate a new module.
        There can be many instance of the same type. 'mod_conf' is module configuration object for this new module instance.
        """
        self.myconf = mod_conf
        BaseSubProcess.__init__(self, mod_conf.get_name())
        ModuleInfoReaderMixin.__init__(self, mod_conf)
        
        self.logger = LoggerFactory.get_logger(mod_conf.get_name())
        
        # We can have submodules
        self.modules = getattr(mod_conf, 'modules', [])
        self.module_type = mod_conf.properties['type']
        self.interrupted = False
        
        # the queue the module will receive data to manage
        self.to_q = None  # type: Optional[ShinkenQueue]
        # the queue the module will put its result data
        self.from_module_to_main_daemon_queue = None  # type: Optional[ShinkenQueue]
        self.commands_from_q = None  # type: Optional[ShinkenQueue]
        self.commands_to_q = None  # type: Optional[ShinkenQueue]
        self.command_queue_handler = None  # type: Optional[CommandQueueHandler]
        self.process = None
        
        self.init_try = 1
        
        self._init_try_count = 0
        self._second_interval_init_try = 20
        self._last_init_try = int(time.time())
        
        # We want to know where we are load from? (broker, scheduler, etc)
        self.loaded_into = 'unknown'
        self.daemon_display_name = 'UNSET'
        
        # Will be created when called by create queues
        self._queue_factory = None
        self.manager = None
        
        # For sub process clean, we will have to keep a trace of all our queues
        self._all_queues = []  # type: List[Optional[ShinkenQueue]]
        
        self.fatal_error_has_been_managed = False
        
        self.messages_to_send_at_arbiter_lock = threading.RLock()
        self.messages_to_send_at_arbiter = []  # type: List[InterDaemonMessage]
    
    
    def get_internal_state(self):
        return ModuleState.OK
    
    
    def init(self):
        """Handle this module "post" init ; just before it'll be started.
        Like just open necessaries file(s), database(s),
        or whatever the module will need.
        """
        pass
    
    
    def has_never_been_init(self):
        return self._init_try_count == 0
    
    
    def stop_all(self, update_stop_logger=None):
        if not update_stop_logger:
            update_stop_logger = self.logger
        if self.has_never_been_init():
            return
        
        try:
            update_stop_logger.info('Trying to stop module')
            try:
                if hasattr(self, 'quit') and callable(self.quit):
                    self.quit()
            except Exception as e:
                update_stop_logger.error('Stopping module failed. The quit() function has failed with error:  %s.' % e)
                update_stop_logger.print_stack()
            
            self.stop_process()
            
            # If external, clean its queues
            if self.is_external or self.is_worker_based:
                self.clear_queues()
                self.kill_manager()
            
            update_stop_logger.info('Stopped module successfully.')
        except Exception as exp:
            update_stop_logger.info('Stopping module has failed raising exception: %s.' % exp)
            update_stop_logger.print_stack()
    
    
    def get_process_name(self):
        _daemon_display_name = self.father_name
        if not _daemon_display_name or _daemon_display_name == 'NO-DISPLAY-NAME':
            _daemon_display_name = 'shinken-%s' % self.loaded_into
        return "%s [ - Module: %s ]%s" % (_daemon_display_name, self.name, self._get_worker_display_name())
    
    
    def set_loaded_into(self, daemon_name):
        self.loaded_into = daemon_name
    
    
    def kill_manager(self):
        if self.manager is None:
            return
        process = getattr(self.manager, '_process', None)
        if process:
            pid = process.pid
            try:
                self.logger.debug('kill_manager:: %s -5' % pid)
                process.terminate()
                process.join(0.1)
                
                start_wait_is_dead = time.time()
                while process.is_alive() and (time.time() - start_wait_is_dead) < 3:
                    process.join(0.1)
                
                if process.is_alive():
                    self.logger.debug('kill_manager:: %s -9' % pid)
                    os.kill(process.pid, signal.SIGKILL)
                    self.logger.debug('kill_manager:: %s wait' % pid)
            except Exception as e:
                self.logger.warning('kill_manager:: fail to stop manager with error e:[%s]' % e)
            
            try:
                self.logger.debug('kill_manager:: %s wait 2' % pid)
                process.join()
            except Exception as e:
                self.logger.warning('kill_manager:: fail to stop manager with error e:[%s]' % e)
        else:
            self.logger.debug('kill_manager:: classic shutdown')
            self.manager.shutdown()
        self.logger.info('The queue manager process is now stopped.')
        self.manager = None
        self.logger.info('The queue manager process is now clean.')
    
    
    def queue_factory(self, name, main_name=None, other_side_name=None):
        # type: (str, Optional[str], Optional[str]) -> ShinkenQueue
        queue = ShinkenQueue(name, main_name, other_side_name)
        queue.name = name
        self._all_queues.append(queue)
        self.logger.debug(PART_INITIALISATION, 'Creating a exchange queue [%s] with the module. Queue address (for debug only): %s ' % (name, queue))
        return queue
    
    
    def create_queues(self, manager_factory=None):
        self.clear_queues()
        self.logger.get_sub_part(PART_INITIALISATION).debug('Creating module[%s] queues.' % self.get_name())
        module_name = self.get_name()
        self.from_module_to_main_daemon_queue, self.to_q = build_shinken_queue('BRK', 'Main', '%s' % module_name)
        self.commands_from_q, self.commands_to_q = build_shinken_queue('CMD', 'Main', '%s' % module_name)
        self.command_queue_handler = CommandQueueHandler(self.get_name(), self.logger, self.commands_to_q, self.commands_from_q, self)
        self._all_queues.append(self.from_module_to_main_daemon_queue)
        self._all_queues.append(self.commands_from_q)
    
    
    # Release the resources associated to the queues of this instance
    def clear_queues(self):
        for queue in self._all_queues:
            queue.close()
        
        self.to_q = self.from_module_to_main_daemon_queue = self.commands_from_q = self.commands_to_q = None
        self._all_queues = []
    
    
    # Basic modules do not handle workers
    def start_workers(self, daemon_display_name='UNSET'):
        return
    
    
    # Start this module process if it's external. if not -> do nothing
    def start(self, http_daemon=None, daemon_display_name='UNSET'):
        self.daemon_display_name = daemon_display_name
        self.update_father(daemon_display_name)
        if not self.is_external:
            return
        self.stop_process()
        self.logger.info(PART_INITIALISATION, "Starting external process")
        is_fork_possible = mem_wait_for_fork_possible('module %s' % self.name, fast_error=True)
        if not is_fork_possible:
            raise Exception('[PERFORMANCE][MEMORY] Cannot start the module process as there is not enough memory for this')

        p = Process(target=self._main, args=())
        p.start()
        
        # We save the process data AFTER the fork()
        self.process = p
        self.logger.info(PART_INITIALISATION, "module process is started as pid=%d" % p.pid)
    
    
    # Sometime terminate() is not enough, we must "help" external modules to die...
    @staticmethod
    def __kill(process):
        if os.name == 'nt':
            process.terminate()
        else:
            # Ok, let him 1 second before really KILL IT
            os.kill(process.pid, signal.SIGTERM)
            time.sleep(1)
            # You do not let me another choice guy...
            try:
                if process.is_alive():
                    os.kill(process.pid, signal.SIGKILL)
            except AssertionError:  # zombie process
                try:
                    os.kill(process.pid, signal.SIGKILL)
                except OSError:
                    pass
    
    
    def is_manager_alive(self):
        if not self.manager:
            return True
        process = getattr(self.manager, '_process', None)
        if not process:
            return False
        return process.is_alive()
    
    
    def is_alive(self):
        if self.get_internal_state() == ModuleState.FATAL:
            return False
        
        if not self.is_external:
            return True
        
        try:
            return self.process.is_alive() and self.is_manager_alive()
        except AssertionError:  # maybe the process became daemon, so we are no more its parent. Kill it anyway
            self.logger.warning('Seems that the manager of the module is in a bad state')
            return False
    
    
    def is_ready_to_init(self, module_manager_logger, late_start):
        # type: (PartLogger, bool) -> int
        if not late_start and self._init_try_count != 0:
            retry_interval = min(self._second_interval_init_try, MAX_MODULE_INIT_TRY_INTERVAL)
            # Check if current time is inferior of time last init try + 1hour. The time of the machine can change.
            if int(time.time()) > self._last_init_try + 3600:
                self._last_init_try = int(time.time())
            if int(time.time()) < self._last_init_try + retry_interval:
                module_logger = module_manager_logger.get_sub_part(self.get_name()).get_sub_part('module-type=%s' % self._module_type, register=False)
                module_logger.debug('The module is not ready to start')
                return INSTANCE_INIT_STATE.INIT_SKIPPED
        return INSTANCE_INIT_STATE.INIT_READY
    
    
    def do_init(self, module_manager_logger):
        module_logger = module_manager_logger.get_sub_part(self.get_name(), register=False).get_sub_part('module-type=%s' % self._module_type, register=False)
        self._second_interval_init_try += 10
        self._init_try_count += 1
        self._last_init_try = int(time.time())
        try:
            module_logger.info('Trying to init module.')
            self.init()
            self._second_interval_init_try = 20
            self._init_try_count = 1
        except Exception as exp:
            module_logger.error('The module has failed to init raising exception: %s.' % (str(exp)))
            module_logger.print_stack()
            if self.get_internal_state() != ModuleState.FATAL:
                module_logger.warning('Tried to start the module %s times.' % self._init_try_count)
                module_logger.warning('Will retry to start the module at %s.' % (datetime.fromtimestamp(self._last_init_try + self._second_interval_init_try).strftime('%H:%M:%S')))
            return True, str(exp)
        module_logger.info('The module is started.')
        return False, ''
    
    
    def _do_stop_process(self, process):
        self.logger.info("Stopping module process pid=%s " % process.pid)
        try:
            process.terminate()
        except OSError:  # Maybe the process was already gone, like kill by the system with an OOM or an angry admin
            pass
        try:
            process.join(timeout=1)
        except OSError:  # Maybe the process was already gone too, but I prefer to be sure to try a join()
            pass
        if process.is_alive():
            self.logger.info("The sub-process [%s] is still alive, I help it to die" % process.pid)
            self.__kill(process)
    
    
    # Request the module process to stop and release it
    def stop_process(self):
        if self.process:
            self._do_stop_process(self.process)
            self.process = None
    
    
    def get_module_info(self):
        pass
    
    
    def get_submodule_states(self, module_wanted=''):
        module_wanted = module_wanted.split('.')
        manager = getattr(self, 'modules_manager', None)
        if manager:
            return manager.get_modules_states(module_wanted)
    
    
    def send_command(self, command_name, args=None, raw_args=False):
        if not (self.is_external and self.commands_to_q):
            raise Exception('[MODULES      ] [ %s ] Module is not an external module.' % self.name)
        if self.process is None:
            raise Exception('[MODULES      ] [ %s ] Module process is currently starting.' % self.name)
        elif not self.process.is_alive():
            raise Exception(' Module %s process is currently down and is scheduled for a restart.' % self.name)
        
        command_exist = bool(getattr(self, command_name, None))
        if not command_exist:
            raise UnknownCommandException('[MODULES      ] [ %s ] Command %s was unknown.' % (self.name, command_name))
        
        return self.command_queue_handler.send_command(command_name, args, raw_args=raw_args)
    
    
    def get_state(self, module_wanted=None):
        if not module_wanted:
            module_wanted = []
        
        status = {"status": ModuleState.OK, "output": "OK"}
        
        if self.is_external and self.commands_to_q:
            if self.process is None:
                return {'status': ModuleState.CRITICAL, 'output': 'Module process is currently starting.'}
            elif not self.process.is_alive():
                return {'status': ModuleState.CRITICAL, 'output': 'Module process is currently down and is scheduled for a restart.'}
            
            try:
                module_info = self.send_command('get_module_info')
                if module_info:
                    status.update(module_info)
            except Exception as e:
                return {'status': ModuleState.CRITICAL, 'output': str(e)}
            
            if self.modules:
                try:
                    module_info = self.send_command('get_submodule_states', ['.'.join(module_wanted)])
                    if module_info:
                        status.update(module_info)
                except Exception:
                    status.update({'status': ModuleState.WARNING, 'output': 'Module is running, but does not provide submodule states.'})
        return status
    
    
    def get_module_on_type(self):
        return self._module_on_type
    
    
    def get_module_type(self):
        return self._module_type
    
    
    # TODO: are these 2 methods really needed?
    def get_name(self):
        return self.name
    
    
    def has(self, prop):
        """The classic has: do we have a prop or not?"""
        return hasattr(self, prop)
    
    
    # For in scheduler modules, we will not send all broks to external
    # modules, only what they really want
    def want_brok(self, b):
        return True
    
    
    def manage_brok(self, brok):
        """Request the module to manage the given brok.
        There a lot of different possible broks to manage.
        """
        manage = getattr(self, 'manage_' + brok.type + '_brok', None)
        if manage:
            # Be sure the brok is prepared before call it
            brok.prepare()
            return manage(brok)
    
    
    def do_stop(self):
        """Called just before the module will exit
        Put in this method all you need to cleanly
        release all open resources used by your module
        """
        pass
    
    
    # For external modules only: implement in this method the body of you main loop
    def do_loop_turn(self):
        # type: () -> None
        return None
    
    
    def loop_turn(self):
        # type: () -> None
        return None
    
    
    # Set by worker based modules to display their worker in the process list
    # noinspection -> real methode is not Static
    # noinspection PyMethodMayBeStatic
    def _get_worker_display_name(self):
        return ''
    
    
    def _start_sub_process_threads(self):
        super(BaseModule, self)._start_sub_process_threads()
        
        self.command_queue_handler.start_thread()
        
        # Ensure that all our queues have their threads, so we don't have to pop them at runtime
        for queue in self._all_queues:
            _ = queue.get_queues_size()  # this will start the needed threads for this queue
    
    
    # module 'main' method. Only used by external modules.
    def _main(self):
        _logger.set_name('%s' % self.name)
        self._sub_process_common_warmup()
        self.logger = LoggerFactory.get_logger()
        
        self.logger.info(PART_INITIALISATION, 'Worker now running..')
        try:
            # Will block here!
            self.logger.debug(PART_INITIALISATION, 'go main')
            self.main()
            self.do_stop()
            self.logger.info('exiting now..')
        except Exception as exp:
            self.logger.error('Exit with error : %s' % exp)
            self.logger.print_stack()
    
    
    # TODO: apparently some modules would uses 'work' as the main method??
    work = _main
    
    
    def _internal_get_raw_stats(self, param='', module_wanted=''):
        module_wanted = module_wanted.split('.')
        if len(module_wanted) == 0:
            actual_module = None
        else:
            actual_module = module_wanted[0]
        _internal_get_raw_stats = {}
        modules_manager = getattr(self, 'modules_manager', None)
        if not modules_manager:
            return _internal_get_raw_stats
        
        for module_name, module in modules_manager.get_all_instances_with_name():
            module_get_raw_stats = getattr(module, 'get_raw_stats', None)
            result = {}
            if module_get_raw_stats and (not actual_module or actual_module == module_name):
                arg_spec = inspect.getfullargspec(module_get_raw_stats)
                if len(arg_spec.args) > 2:
                    result = module_get_raw_stats(param, module_wanted[1:])
                elif len(arg_spec.args) > 1:
                    result = module_get_raw_stats(param)
                else:
                    result = module_get_raw_stats()
            
            module_type = module.myconf.module_type
            t = _internal_get_raw_stats.get(module_type, {})
            _internal_get_raw_stats[module_type] = t
            _internal_get_raw_stats[module_type][module_name] = result
        return _internal_get_raw_stats
    
    
    def get_raw_stats(self, param='', module_wanted=None):
        if not module_wanted:
            module_wanted = []
        if self.is_external and self.commands_to_q:
            raw_stats = {
                'modules': {}
            }
            try:
                final_param = []
                final_param.append(param)
                final_param.append('.'.join(module_wanted))
                
                raw_stats['modules'] = self.send_command('_internal_get_raw_stats', final_param)
                return raw_stats
            except Exception as e:
                self.logger.error('Fail in get_raw_stats with error : [%s]' % getattr(e, 'message', str(e)))
    
    
    def handle_messages_received_from_arbiter(self, push_message_by_arbiter: 'InterDaemonMessage') -> None:
        if self.is_external and self.commands_to_q:
            self.send_command('_internal_handle_messages_received_from_arbiter', args=[push_message_by_arbiter], raw_args=True)
        else:
            self._internal_handle_messages_received_from_arbiter(push_message_by_arbiter)
    
    
    def _internal_handle_messages_received_from_arbiter(self, push_message_by_arbiter: 'InterDaemonMessage') -> None:
        modules_manager = getattr(self, 'modules_manager', None)
        if not modules_manager:
            return
        
        message_to = push_message_by_arbiter.message_to
        if message_to.sub_module_name and message_to.sub_module_name != self.name:
            for module in modules_manager.get_all_alive_instances():
                if message_to.sub_module_name == module.get_name():
                    module.handle_messages_received_from_arbiter(push_message_by_arbiter)
    
    
    def send_messages_to_arbiter(self, message_to_arbiter: 'InterDaemonMessage') -> None:
        with self.messages_to_send_at_arbiter_lock:
            self.messages_to_send_at_arbiter.append(message_to_arbiter)
    
    
    def get_messages_to_send_at_arbiter(self) -> 'list[InterDaemonMessage]':
        if self.is_external and self.commands_to_q:
            try:
                return self.send_command('_internal_get_messages_to_send_at_arbiter')
            except Exception as e:
                self.logger.error('Fail in get_messages_to_send_at_arbiter with error : [%s]' % str(e))
                return []
        return self._internal_get_messages_to_send_at_arbiter()
    
    
    def _internal_get_messages_to_send_at_arbiter(self) -> 'list[InterDaemonMessage]':
        with self.messages_to_send_at_arbiter_lock:
            ret = self.messages_to_send_at_arbiter
            self.messages_to_send_at_arbiter = []
        
        modules_manager = getattr(self, 'modules_manager', None)
        if modules_manager:
            for module in modules_manager.get_all_alive_instances():
                ret.extend(module.get_messages_to_send_at_arbiter())
        return ret
