#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import re
import threading
import time
from collections import defaultdict

from . import item
from shinken.log import logger, get_chapter_string, get_section_string

CHAPTER_PROXY = get_chapter_string('PROXY')
SECTION_CLEAN = get_section_string('CLEAN')

PROXY_ITEM_EXPORT_PROPERTIES = [
    'state',
    'in_downtime',
    'in_acknowledge',
    'in_partial_acknowledge',
    'in_partial_flapping',
    'in_partial_downtime',
    'ack_author',
    'ack_comment',
    'partial_ack_authors',
    'partial_ack_comments',
    'inherited_ack_author',
    'inherited_ack_comment',
    'acknowledge_is_auto',
    'in_flapping',
    'inherited_flapping',
    'in_inherited_acknowledged',
    'in_inherited_downtime',
    'last_change',
    'root_problems',
    'business_impact',
]

ALL_PROXY_PROPERTIES = PROXY_ITEM_EXPORT_PROPERTIES + ['id', 'type', 'templates', 'groups', 'uuid', 'name', 'host_name', 'host_uuid']

# Keep the change history for 1min
CHANGE_HISTORY_KEEP_TIME = 60

PROXY_ITEM_STATE_UNINITIALIZED = -1
PROXY_ITEM_STATE_OK = 0
PROXY_ITEM_STATE_WARNING = 1
PROXY_ITEM_STATE_CRITICAL = 2
PROXY_ITEM_STATE_UNKNOWN = 3


class ProxyItem:
    my_type = 'proxy_item'
    __slots__ = set(ALL_PROXY_PROPERTIES)
    
    
    def __init__(self, ref_item):
        # We want to have some data from the original object
        
        self.business_impact = ref_item.business_impact
        if ref_item.my_type == 'service':
            self.type = 'check'
            self.templates = ref_item.use[:]
            self.groups = []
            self.uuid = '%s-%s' % (ref_item.host.uuid, ref_item.uuid)
            self.name = getattr(ref_item, 'service_description', '__NO_SERVICE_DESCRIPTION__')  # configuration will be wrong
            self.host_name = ref_item.host.host_name
            self.host_uuid = ref_item.host.uuid
        elif ref_item.is_cluster:
            self.type = 'cluster'
            self.templates = []
            self.groups = []
            self.uuid = ref_item.uuid
            self.name = ref_item.host_name
        else:
            self.type = 'host'
            self.templates = [t.name for t in ref_item.templates]
            self.groups = [g.get_name() for g in ref_item.hostgroups]
            self.uuid = ref_item.uuid
            self.name = ref_item.host_name
        self.id = self.uuid  # for Items Class sorting
        self.state = PROXY_ITEM_STATE_UNINITIALIZED
        self.in_downtime = False
        
        # Save ack things to allow others elements to manage their own ack
        self.in_acknowledge = False
        self.in_partial_acknowledge = False
        self.in_partial_downtime = False
        self.in_inherited_acknowledged = False
        self.in_inherited_downtime = False
        self.ack_author = ''
        self.ack_comment = ''
        self.acknowledge_is_auto = False
        self.partial_ack_comments = ''
        self.partial_ack_authors = ''
        self.inherited_ack_comment = ''
        self.inherited_ack_author = ''
        
        # Save flapping state
        self.in_flapping = False
        self.inherited_flapping = False
        self.in_partial_flapping = False
        
        # Save root problems
        self.root_problems = set()
        self.last_change = 0
    
    
    def __str__(self):
        display_props = (
            'type', 'name', 'uuid',
            'state', 'in_downtime', 'in_partial_downtime', 'in_inherited_downtime', 'in_acknowledge', 'in_partial_acknowledge', 'in_inherited_acknowledged', 'in_flapping', 'inherited_flapping', 'in_partial_flapping'
        )
        prop_values = {}
        display_format = []
        for display_prop in display_props:
            display_format.append('%s:[%%(%s)s]' % (display_prop, display_prop))
            prop_values[display_prop] = getattr(self, display_prop, '')
        
        prop_values['state'] = self.get_state_display()
        prop_values['name'] = self.get_full_name()
        
        _format = 'ProxyItem:[%s]' % (', '.join(display_format))
        return _format % prop_values
    
    def __repr__(self):
        return self.__str__()
    
    # For debugging purpose only (nice name)
    def get_name(self):
        return self.name
    
    
    def get_full_name(self):
        if self.type == 'check':
            return '%s/%s' % (self.host_name, self.name)
        return self.name
    
    
    def is_ok_up(self):
        return self.state == PROXY_ITEM_STATE_OK
    
    
    def is_acknowledge(self):
        return self.in_acknowledge or self.in_inherited_acknowledged
    
    
    def is_partial_acknowledge(self):
        return self.in_partial_acknowledge
    
    
    def is_flapping(self):
        return self.in_flapping
    
    
    def is_partial_flapping(self):
        return self.in_partial_flapping
    
    
    def get_state_display(self):
        __map = {PROXY_ITEM_STATE_UNINITIALIZED: 'UNINITIALIZED', PROXY_ITEM_STATE_OK: 'OK', PROXY_ITEM_STATE_WARNING: 'WARNING', PROXY_ITEM_STATE_CRITICAL: 'CRITICAL', PROXY_ITEM_STATE_UNKNOWN: 'UNKNOWN'}
        if self.type == 'host':  # host is lying there
            __map[PROXY_ITEM_STATE_WARNING] = 'CRITICAL'
        return __map.get(self.state)
    
    
    def get_export_state(self):
        r = {}
        for prop in PROXY_ITEM_EXPORT_PROPERTIES:
            r[prop] = getattr(self, prop)
        return r
    
    
    # Call by pickle to data-ify the host
    # we do a dict because list are too dangerous for retention save and co :( even if it's more extensive
    # The setstate function do the inverse
    def __getstate__(self):
        res = {}
        for prop in ALL_PROXY_PROPERTIES:
            if hasattr(self, prop):
                # logger.debug('PROXY Saving property: %s=> %s for %s' % (prop, getattr(self, prop), self.name))
                res[prop] = getattr(self, prop)
        return res
    
    
    # Inverted function of __getstate__
    def __setstate__(self, state):
        for (prop, value) in state.items():
            # logger.debug('PROXY Restoring property: %s=> %s for %s' % (prop, value, state['name']))
            setattr(self, prop, value)


# Important: do not want to pickle the lock in the arbiter
#            when serializing the conf, and as a patch must
#            be ok with unpatched arbiter currently
CHANGES_HISTORY_LOCK = threading.RLock()


class ProxyItems(item.Items):
    name_property = "uuid"
    inner_class = ProxyItem
    
    host_flags = "grlt"
    service_flags = "grlt"
    
    
    def __init__(self, items):
        super(ProxyItems, self).__init__(items)
        self.hosts = {}
        self.services = {}
        for p in self:
            if p.type == 'check':
                self.services[p.uuid] = p
            else:
                self.hosts[p.uuid] = p
        self.changes_history = {}
    
    
    # We have the full list from the conf so:
    # * create if missing
    # * update some fields if already have
    # * delete if not in the new whole part (because such elements are removed from the realm)
    def refresh_items(self, new_whole_items):
        super(ProxyItems, self).__init__(new_whole_items)
        prev_hosts = self.hosts
        prev_services = self.services
        self.hosts = {}
        self.services = {}
        
        for p in self:
            if p.type == 'check':
                self.services[p.uuid] = p
                old_proxy = prev_services.get(p.uuid, None)
            else:
                self.hosts[p.uuid] = p
                old_proxy = prev_hosts.get(p.uuid, None)
            # Keep previous value of exported properties if available
            if old_proxy:
                for prop_name in PROXY_ITEM_EXPORT_PROPERTIES:
                    setattr(p, prop_name, getattr(old_proxy, prop_name))
    
    
    # We get states from other scheduler, and we need to update it
    def update_from_other_states(self, states):
        for (i_uuid, state) in states.items():
            if i_uuid not in self:
                continue
            p = self[i_uuid]
            if state['last_change'] <= p.last_change:
                continue
            # Ok we can really update value
            last_change = state['last_change']
            is_updated = False
            for property_name in PROXY_ITEM_EXPORT_PROPERTIES:
                is_updated |= self.update_property(i_uuid, property_name, state[property_name], last_change=last_change, manual_element_did_change=True)
            
            if is_updated:
                self.element_did_change(i_uuid, last_change)
    
    
    # Search items using a list of filter callbacks. Each callback is passed
    # the item instances and should return a boolean value indicating if it
    # matched the filter.
    # Returns a list of items matching all filters.
    def find_hosts_by_expr(self, host_expr, flags):
        all_hosts = list(self.hosts.values())
        # If no flags, can be all or just one
        if not flags:
            # If all, return all hosts
            if host_expr == "*":
                return all_hosts
            
            # ok got a name
            return [p for p in all_hosts if p.name == host_expr]
        
        # Got a special flag
        
        if 't' in flags and 'r' in flags:
            host_template_re = re.compile(host_expr)
            return [p for p in all_hosts if [tpl for tpl in p.templates if host_template_re.match(tpl)]]
        elif 'g' in flags:
            return [p for p in all_hosts if host_expr in p.groups]
        elif 'r' in flags:
            host_re = re.compile(host_expr)
            return [p for p in all_hosts if host_re.match(p.name)]
        elif 't' in flags:
            return [p for p in all_hosts if host_expr in p.templates]
        else:
            return []
    
    
    # Find services based on both host and service expression
    # First: look at hosts, then for all services of theses hosts
    # apply the service expr
    def find_services_by_exprs(self, hst_expr, hst_flags, chk_expr, chk_flags):
        matching_hosts = self.find_hosts_by_expr(hst_expr, hst_flags)
        matched_hst_uuids = {p.uuid for p in matching_hosts}
        
        possible_checks = [p for p in self.services.values() if p.host_uuid in matched_hst_uuids]
        
        # No flags can means all or just a name
        if not chk_flags:
            if chk_expr == "*":
                return possible_checks
            # ok so directly the name
            return [p for p in possible_checks if p.name == chk_expr]
        
        if "t" in chk_flags and "r" in chk_flags:
            host_template_re = re.compile(chk_expr)
            return [p for p in possible_checks if [tpl for tpl in p.templates if host_template_re.match(tpl)]]
        elif "r" in chk_flags:
            name_re = re.compile(chk_expr)
            return [p for p in possible_checks if name_re.match(p.name)]
        elif "t" in chk_flags:
            return [p for p in possible_checks if chk_expr in p.templates]
        else:
            return []
    
    
    def find_check(self, hname, sdesc):
        for p in self.services.values():
            if p.host_name == hname and p.name == sdesc:
                return p
        return None
    
    
    def find_host(self, hname):
        for p in self.hosts.values():
            if p.name == hname:
                return p
        return None
    
    
    def get_all(self):
        return list(self.items.values())
    
    
    # an element did change a property, so index it when someone will ask us for last_changes
    def element_did_change(self, i_uuid, last_change):
        with CHANGES_HISTORY_LOCK:
            if last_change not in self.changes_history:
                self.changes_history[last_change] = set()
            entry = self.changes_history[last_change]
        entry.add(i_uuid)
        
        # Warn the proxy graph that we did change, and so the cluster that rely on us must be recompute
        proxyitemsgraph.trigger_my_clusters_state_compute(i_uuid)
    
    
    def update_property(self, i_uuid, property_name, update_value, last_change=None, manual_element_did_change=False):
        try:
            p = self[i_uuid]
        except KeyError:  # SEF-7928: don't know how to gt into this case, so putting a stop crash here with more infos
            # so we can fix it in the future
            logger.error('The proxy items are missing the element uuid %s. Current elements are: %s (=%d elements). Please report a bug to your support with all this log file.' % (i_uuid, list(self.items.keys()), len(self.items)))
            logger.print_stack()
            return False
        perv_value = getattr(p, property_name)
        # If the same, bail out
        if perv_value == update_value:
            return False
        
        setattr(p, property_name, update_value)
        
        if last_change is None:
            last_change = int(time.time())
        p.last_change = last_change
        
        # and index this modification so we can send only changed elements
        if not manual_element_did_change:
            self.element_did_change(i_uuid, last_change)
        return True
    
    
    def update_state(self, i_uuid, state, last_change=None):
        self.update_property(i_uuid, 'state', state, last_change)
    
    
    def update_flapping(self, i_uuid, is_flapping, last_change=None, inherited_flapping=False):
        update1 = self.update_property(i_uuid, 'in_flapping', is_flapping, last_change, manual_element_did_change=True)
        update2 = self.update_property(i_uuid, 'inherited_flapping', inherited_flapping, last_change, manual_element_did_change=True)
        
        if update1 or update2:
            if last_change is None:
                last_change = int(time.time())
            self.element_did_change(i_uuid, last_change)
    
    
    def update_partial_flapping(self, i_uuid, is_partial_flap, last_change=None):
        self.update_property(i_uuid, 'in_partial_flapping', is_partial_flap, last_change)
    
    
    def update_acknowledge(self, i_uuid, is_ack, last_change=None):
        self.update_property(i_uuid, 'in_acknowledge', is_ack, last_change)
    
    
    def update_partial_acknowledge(self, i_uuid, is_partial_ack, last_change=None):
        self.update_property(i_uuid, 'in_partial_acknowledge', is_partial_ack, last_change)
    
    
    def update_partial_downtime(self, i_uuid, is_partial_dt, last_change=None):
        self.update_property(i_uuid, 'in_partial_downtime', is_partial_dt, last_change)
    
    
    def update_downtime(self, i_uuid, is_downtime, last_change=None):
        self.update_property(i_uuid, 'in_downtime', is_downtime, last_change)
    
    
    def update_in_inherited_downtime(self, i_uuid, in_inherited_downtime, last_change=None):
        self.update_property(i_uuid, 'in_inherited_downtime', in_inherited_downtime, last_change)
    
    
    def update_in_inherited_acknowledged(self, i_uuid, in_inherited_acknowledged, last_change=None, author='', comment=''):
        self.update_property(i_uuid, 'in_inherited_acknowledged', in_inherited_acknowledged, last_change)
        self.update_property(i_uuid, 'inherited_ack_author', author, last_change)
        self.update_property(i_uuid, 'inherited_ack_comment', comment, last_change)
    
    
    def update_acknowledge_author_and_comment_auto(self, i_uuid, author, comment, auto, last_change=None):
        update1 = self.update_property(i_uuid, 'ack_author', author, last_change, manual_element_did_change=True)
        update2 = self.update_property(i_uuid, 'ack_comment', comment, last_change, manual_element_did_change=True)
        update3 = self.update_property(i_uuid, 'acknowledge_is_auto', auto, last_change, manual_element_did_change=True)
        
        if update1 or update2 or update3:
            if last_change is None:
                last_change = int(time.time())
            self.element_did_change(i_uuid, last_change)
    
    
    def update_partial_acknowledge_author_and_comment(self, i_uuid, author, comment, last_change=None):
        update = self.update_property(i_uuid, 'partial_ack_authors', author, last_change, manual_element_did_change=True)
        update |= self.update_property(i_uuid, 'partial_ack_comments', comment, last_change, manual_element_did_change=True)
        
        if update:
            if last_change is None:
                last_change = int(time.time())
            self.element_did_change(i_uuid, last_change)
    
    
    def update_root_problems(self, i_uuid, root_problems, last_change=None):
        self.update_property(i_uuid, 'root_problems', root_problems, last_change)
    
    
    def update_business_impact(self, i_uuid, bi, last_change=None):
        self.update_property(i_uuid, 'business_impact', bi, last_change)
    
    
    def get_export_state_if_need(self, i_uuid, since):
        p = self[i_uuid]
        if p.last_change > since:
            return p.get_export_state()
        return None
    
    
    def clean_history(self):
        begin = time.time()
        now = int(begin)
        too_late_time = now - CHANGE_HISTORY_KEEP_TIME
        with CHANGES_HISTORY_LOCK:
            nb_changes_total = 0
            nb_changes_cleaned = 0
            history_date_availables = list(self.changes_history.keys())  # IMPORTANT: not .keys because we will del entries
            for t in history_date_availables:
                if t < too_late_time:
                    nb_changes_cleaned += len(self.changes_history[t])
                    del self.changes_history[t]
                else:
                    nb_changes_total += len(self.changes_history[t])
        logger.debug('%s %s Cleaning %4d of proxy entries changes (to be exchanged with other schedulers), %4d are remaining. (done in %.3f)' % (CHAPTER_PROXY, SECTION_CLEAN, nb_changes_cleaned, nb_changes_total, time.time() - begin))
    
    
    # get export state from elements, but only from elements that did change since it
    def get_export_state_since(self, since, what_to_look_for):
        r = {}
        # If it ask for a specific time range, look for only changes
        if since != 0:
            with CHANGES_HISTORY_LOCK:
                valid_times = [t for t in self.changes_history.keys() if t >= since]
                if valid_times:
                    logger.debug('EXPORTING ELEMENTS that did changes: time keys: %s' % valid_times)
                all_our_elements = set()
                for t in valid_times:
                    all_our_elements = all_our_elements.union(self.changes_history[t])
            # take only elements that are in both in index and what_to_look_for
            changes_elements = all_our_elements.intersection(what_to_look_for)
            if len(changes_elements):
                logger.debug('EXPORTING: final changes elements: %s=>%s ' % (since, len(changes_elements)))
        else:  # ok take all
            changes_elements = what_to_look_for
        
        for i_uuid in changes_elements:
            p = self[i_uuid]
            st = p.get_export_state()
            if st is not None:
                r[i_uuid] = st
        return r


class ProxyItemsGraph:
    def __init__(self):
        self.father_to_sons = None
        self.son_to_fathers = None
        self.cluster_to_recompute_state = None
        self.cluster_to_recompute_root_problems = None
        self.reset()
    
    
    def reset(self):
        self.father_to_sons = defaultdict(set)
        self.son_to_fathers = defaultdict(set)
        
        self.cluster_to_recompute_state = set()
        self.cluster_to_recompute_root_problems = set()
    
    
    def update_acknowledge_author_and_comment_auto(self, i_uuid, author, comment, auto):
        p = self[i_uuid]
        prev_author = p.ack_author
        prev_comment = p.ack_comment
        prev_auto = p.acknowledge_is_auto
        
        if (prev_author == author) and (prev_comment == comment) and (prev_auto == auto):
            return
        p.ack_author = author
        p.ack_comment = comment
        p.acknowledge_is_auto = auto
        p.last_change = int(time.time())
    
    
    # List of (from, to) elements and all to element event them without from (empty cluster for ex)
    def reset_from_all_links(self, links, son_nodes):
        self.reset()
        
        for son_uuid in son_nodes:
            self.son_to_fathers[son_uuid] = set()
        
        for (father_uuid, son_uuid) in links:
            self.father_to_sons[father_uuid].add(son_uuid)
            self.son_to_fathers[son_uuid].add(father_uuid)
    
    
    def reset_from_other(self, other):
        self.reset()
        self.father_to_sons = other.father_to_sons
        self.son_to_fathers = other.son_to_fathers
        
        # if we did reset, set all as dirty
        for c_uuid in self.son_to_fathers.keys():
            self.cluster_to_recompute_state.add(c_uuid)
            # But also force to recompute possible root problems
            self.cluster_to_recompute_root_problems.add(c_uuid)
    
    
    # We have another graph and this one have a partial view, so update our values with it
    # NOTE: this will make the father to son list INVALID! because we update only a part
    # NOTE: ONLY FOR THE UIs!!!!
    def update_from_son_to_fathers(self, other_son_to_father):
        self.son_to_fathers.update(other_son_to_father)
    
    
    # When an element get a state change, look in which cluster it is present
    # and keep this cluster uuid in a dirty list we will need to recompute:
    # * state : obvious
    # * but also root problems because myabe this element will be its own root problem and so should
    #   be set in the cluster root problem
    def trigger_my_clusters_state_compute(self, father_uuid):
        clusters = self.father_to_sons.get(father_uuid, [])
        for c_uuid in clusters:
            self.cluster_to_recompute_state.add(c_uuid)
            # But also force to recompute possible root problems
            self.cluster_to_recompute_root_problems.add(c_uuid)
    
    
    # When an element get a root problem change, so take them to the cluster list they are
    def trigger_my_clusters_root_problem(self, father_uuid):
        clusters = self.father_to_sons.get(father_uuid, [])
        for c_uuid in clusters:
            self.cluster_to_recompute_root_problems.add(c_uuid)
    
    
    def get_and_reset_clusters_to_recompute_state(self):
        r = self.cluster_to_recompute_state
        self.cluster_to_recompute_state = set()
        return r
    
    
    def get_and_reset_clusters_to_recompute_root_problems(self):
        r = self.cluster_to_recompute_root_problems
        self.cluster_to_recompute_root_problems = set()
        return r


# utility functions for root problems, before they are move entirely into proxygraph
def root_problems_to_uuids(ref, v):
    if ref.is_cluster:
        r = list(proxyitemsmgr[ref.get_instance_uuid()].root_problems)
    else:
        r = [e.get_instance_uuid() for e in v]
    
    return r


# Set this element as singleton
proxyitemsmgr = ProxyItems([])
proxyitemsgraph = ProxyItemsGraph()
