#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import PartLogger
from shinken.misc.type_hint import List, Dict, Optional
from shinkensolutions.api.synchronizer import SourceTranslatePart, ItemType, ValidationState
from shinkensolutions.api.synchronizer.source.check_property import CHECK_PROPERTY
from shinkensolutions.api.synchronizer.source.file_loader import FileLoader
from shinkensolutions.api.synchronizer.source.item.origin_item import OriginItemPropertyName
from shinkensolutions.api.synchronizer.source.list_operations import ListOperations
from shinkensolutions.api.synchronizer.source.source_configuration_value import SERVICE_MODE, ServiceMode


class ApiItemProperties:
    def __init__(self, logger, translator, file_loader, configuration_state):
        # type: (PartLogger, SourceTranslatePart, FileLoader, ServiceMode) -> None
        self.logger = logger
        self._translator = translator
        self.file_loader = file_loader
        self.user_file = file_loader.api_properties.user_file
        self.configuration_state = configuration_state
        self.properties_allows = {}  # type: Dict[ItemType,List[OriginItemPropertyName]]
        self.properties_known = {}  # type: Dict[ItemType,List[OriginItemPropertyName]]
        self.validation_state = None  # type: Optional[ValidationState]
        self.reload()
    
    
    def add_error(self, message=''):
        # type: (str) -> None
        self.validation_state.add_error(message=message)
    
    
    def add_warning(self, message=''):
        # type: (str) -> None
        self.validation_state.add_warning(message=message)
    
    
    def check_property(self, item_type, property_name):
        # type: (ItemType, OriginItemPropertyName) -> str
        if self.configuration_state == SERVICE_MODE.OFF:
            return CHECK_PROPERTY.IS_ALLOW
        
        check_mapping_rule = ListOperations.check_property_mapping_value(item_type, property_name, self.check_one_property)
        
        if check_mapping_rule == 'NO_RULE_FOUND':
            return self.check_one_property(item_type, property_name)
        else:
            return check_mapping_rule
    
    
    def check_one_property(self, item_type, property_name):
        # type: (ItemType, OriginItemPropertyName) -> str
        
        property_name = ListOperations.remove_mapping_rule_from_property_mapping_value(property_name)
        if property_name in self.properties_allows.get(item_type, []):
            return CHECK_PROPERTY.IS_ALLOW
        if property_name in self.properties_known.get(item_type, []):
            return CHECK_PROPERTY.IS_DISABLE
        return CHECK_PROPERTY.IS_UNKNOWN
    
    
    def reload(self):
        # type: () -> None
        self.logger.info('reload known api properties')
        self.properties_allows, self.properties_known, self.validation_state = self._load()
        self.validate()
    
    
    # This method can be override to validate api properties after load
    def validate(self):
        # type: () -> None
        pass
    
    
    def _load(self):
        # type: () -> (Dict[ItemType,List[OriginItemPropertyName]], Dict[ItemType,List[OriginItemPropertyName]], ValidationState)
        
        validation_state = ValidationState(self.logger, self._translator)
        
        if self.configuration_state == SERVICE_MODE.OFF:
            return {}, {}, validation_state
        
        default_file = self.file_loader.api_properties.default_file
        if not default_file.exist():
            validation_state.add_warning(translate_key='warning.missing_description_file', params=(default_file.path,))
            return {}, {}, validation_state
        
        try:
            raw_properties_allows = default_file.load()
        except Exception as e:
            validation_state.add_error(translate_key='error.fail_to_load_file', params=(default_file.path, str(e)))
            return {}, {}, validation_state
        
        if not isinstance(raw_properties_allows, dict):
            validation_state.add_error(translate_key='error.fail_to_load_file', params=(default_file.path, ''))
            return {}, {}, validation_state
        
        self.user_file = self.file_loader.api_properties.user_file
        if self.configuration_state == SERVICE_MODE.ON and not self.user_file.exist():
            validation_state.add_warning(translate_key='warning.missing_user_file', params=(self.user_file.path,))
        if self.configuration_state == SERVICE_MODE.ON and validation_state.is_valid() and self.user_file.exist():
            user_raw_properties_allows = {}
            try:
                user_raw_properties_allows = self.user_file.load()
            except Exception as e:
                validation_state.add_error(translate_key='error.fail_to_load_file', params=(self.user_file.path, str(e)))
            
            if not isinstance(user_raw_properties_allows, dict):
                validation_state.add_error(translate_key='error.fail_to_load_file', params=(self.user_file.path, ''))
                return {}, {}, validation_state
            
            for item_type, user_raw_properties_allows_by_type in user_raw_properties_allows.items():
                if item_type not in raw_properties_allows:
                    raw_properties_allows[item_type] = user_raw_properties_allows_by_type
                    continue
                for property_name, allowed in user_raw_properties_allows_by_type.items():
                    raw_properties_allows[item_type][property_name] = allowed
        
        properties_allows = {}
        properties_known = {}
        for item_type, raw_properties_allows_by_type in raw_properties_allows.items():
            properties_allows[item_type] = {property_name for property_name, allowed in raw_properties_allows_by_type.items() if self.read_allowed_value(allowed, property_name, validation_state)}
            properties_known[item_type] = list(raw_properties_allows_by_type.keys())
        return properties_allows, properties_known, validation_state
    
    
    def read_allowed_value(self, allowed, property_name, validation_state):
        try:
            return allowed.lower() == 'true'
        except AttributeError:
            validation_state.add_warning(translate_key='warning.unknown_value', params=(property_name, self.user_file.path,))
            return False
