#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import inspect
import math
import sys
import traceback

from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import List

CFG_SEPARATOR = ';'
NEW_LINE_IN_TAG = '(br)'
DEFAULT_FORMAT = '───'

IGNORED_PROPERTIES = ('module_type',)

FORMATTER_VERSION = 1


class ShinkenCfgFormatterException(Exception):
    pass


class FORMATTER_KEYS:
    DESCRIPTION_CFG_FORMATTER_VERSION = 'description_cfg_formatter_version'
    
    FILE_FORMAT = 'file_format'
    
    SET_LINE_SIZE = 'set_line_size'
    DESCRIPTION_TITLE = 'description_title'
    DESCRIPTION_TEXT = 'description_text'
    DESCRIPTION_END = 'description_end'
    
    DEFINE_START = 'define_start'
    DEFINE_STOP = 'define_stop'
    
    PARAMETER_TITLE = 'parameter_title'
    PARAMETER_TITLE_MANDATORY = 'parameter_title_mandatory'
    PARAMETER_TEXT = 'parameter_text'
    PARAMETER_TEXT_WARNING = 'parameter_text_warning'
    PARAMETER_TEXT_INDENT = 'parameter_text_indent'
    PARAMETER_TEXT_DEFAULT = 'parameter_text_default'
    PARAMETER_TEXT_RECOMMENDED = 'parameter_text_recommended'
    
    PARAMETER_LIST_EXAMPLE = 'parameter_list_example'
    PARAMETER_LIST_EXAMPLE_2 = 'parameter_list_example_2'
    PARAMETER_LIST_EXAMPLE_3 = 'parameter_list_example_3'
    PARAMETER_LIST_DEFAULT = 'parameter_list_default'
    PARAMETER_LIST = 'parameter_list'
    PARAMETER_LIST_INDENT = 'parameter_list_indent'
    PARAMETER_LIST_INDENT_2 = 'parameter_list_indent_2'
    PARAMETER_LIST_INDENT_3 = 'parameter_list_indent_3'
    
    PARAMETER_VALUE = 'parameter_value'
    DAEMON_PARAMETER_VALUE = 'daemon_parameter_value'
    PARAMETER_COMMENTED = 'parameter_value_comment'
    DAEMON_PARAMETER_COMMENTED = 'daemon_parameter_value_comment'
    
    OVERRIDE_FROM = 'override_from'
    OVERRIDE_TYPE = 'override_type'
    
    OVERRIDE_FILE_FORMAT = 'override'
    DAEMON_FILE_FORMAT = 'daemon'
    MODULE_FILE_FORMAT = 'module'
    
    HEADER_1 = 'h1'
    HEADER_2 = 'h2'
    BREAK_LINE = 'br'
    SEPARATOR_PARAMETER = 'separator_parameter'
    
    SHOULD_NOT_BE_LAST_LINE = (PARAMETER_TEXT_DEFAULT, PARAMETER_TEXT_WARNING, PARAMETER_TEXT_RECOMMENDED, PARAMETER_LIST_EXAMPLE, PARAMETER_LIST, PARAMETER_LIST_INDENT)
    ALL_PARAMETER_KEYS = (PARAMETER_VALUE, PARAMETER_COMMENTED, DAEMON_PARAMETER_VALUE, DAEMON_PARAMETER_COMMENTED)


DEFAULT_SIZE_LINE = 107
ALIGN_VALUES = 50
DESCRIPTION_LENGTH = 80

need_separator = False
last_line_type = False

OVERRIDE_TYPE_KEY = '__OVERRIDE_TYPE__'


def set_line_size(_buffer, data):
    # type: (List[str], str) -> None
    global SIZE_LINE
    SIZE_LINE = int(data)


def _append_to_buffer(_buffer, data, type_line):
    # type: (List[str], str, str) -> None
    global last_line_type
    _buffer.append(data)
    last_line_type = type_line


def _separator_description(_buffer):
    # type: (List[str]) -> None
    text = DESCRIPTION_LENGTH * '='
    _append_to_buffer(_buffer, '#%s' % text, inspect.stack()[0][3])


def _separator_default(_buffer, _=''):
    # type: (List[str], str) -> None
    global last_line_type
    if last_line_type is not False and last_line_type != inspect.stack()[0][3]:
        _append_to_buffer(_buffer, '', inspect.stack()[0][3])


def separator_parameter(_buffer):
    # type: (List[str]) -> None
    global last_line_type
    _type_line = inspect.stack()[0][3]
    if last_line_type != _type_line and last_line_type not in FORMATTER_KEYS.SHOULD_NOT_BE_LAST_LINE:
        parameter_text(_buffer, '', line_type=inspect.stack()[0][3])


def separator_parameter_value(_buffer):
    # type: (List[str]) -> None
    global last_line_type
    _type_line = inspect.stack()[0][3]
    if last_line_type in (
            _type_line, FORMATTER_KEYS.PARAMETER_TEXT_DEFAULT, FORMATTER_KEYS.PARAMETER_TEXT_WARNING, FORMATTER_KEYS.PARAMETER_TEXT_RECOMMENDED, FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE, FORMATTER_KEYS.PARAMETER_LIST,
            FORMATTER_KEYS.PARAMETER_LIST_INDENT):
        _type_line = FORMATTER_KEYS.SEPARATOR_PARAMETER
    parameter_text(_buffer, '', line_type=_type_line)


def description_title(_buffer, data):
    # type: (List[str], str) -> None
    _separator_description(_buffer)
    description_text(_buffer, data)
    _separator_description(_buffer)


def description_text(_buffer, data):
    # type: (List[str], str) -> None
    _append_to_buffer(_buffer, '# %s' % data, inspect.stack()[0][3])


def description_cfg_formatter_version(_buffer, data):
    # type: (List[str], str) -> None
    _append_to_buffer(_buffer, '# CFG_FORMAT_VERSION %s ( SHINKEN : DON\'T TOUCH THIS LINE )\n' % data, inspect.stack()[0][3])


def description_end(_buffer, data):
    # type: (List[str], str) -> None
    description_text(_buffer, data)
    _separator_description(_buffer)


def define_start(_buffer, data):
    # type: (List[str], str) -> None
    _separator_default(_buffer)
    _append_to_buffer(_buffer, 'define %s {' % data, inspect.stack()[0][3])
    _separator_default(_buffer)


def define_stop(_buffer, _):
    # type: (List[str], str) -> None
    _separator_default(_buffer)
    _append_to_buffer(_buffer, '}', inspect.stack()[0][3])
    _separator_default(_buffer)


def parameter_title(_buffer, data):
    # type: (List[str], str) -> None
    _separator_default(_buffer)
    parameter_text(_buffer, data, inspect.stack()[0][3])


def parameter_title_mandatory(_buffer, data):
    # type: (List[str], str) -> None
    _size = SIZE_LINE - len(' [ MANDATORY ] ') - 2 * len(DEFAULT_FORMAT) - len(data) - 3
    _text_to_add = _size * ' '
    parameter_title(_buffer, '%s%s [ MANDATORY ]' % (data, _text_to_add))


def parameter_text(_buffer, data, line_type=''):
    # type: (List[str], str, str) -> None
    prefix_structure = '' if line_type == FORMATTER_KEYS.SEPARATOR_PARAMETER else DEFAULT_FORMAT
    line_type = line_type or inspect.stack()[0][3]
    space_end = (SIZE_LINE - len('# %s  %s' % (prefix_structure, DEFAULT_FORMAT)) - len(data)) * ' '
    _append_to_buffer(_buffer, '    # %s %s%s %s' % (prefix_structure, data, space_end, DEFAULT_FORMAT), line_type)


def override_from(_buffer, data):
    # type: (List[str], str) -> None
    line_type = inspect.stack()[0][3]
    _append_to_buffer(_buffer, '    [OVERRIDE_FROM] %s' % data, line_type)


def override_type(_buffer, data):
    # type: (List[str], str) -> None
    line_type = inspect.stack()[0][3]
    size_line = SIZE_LINE - len('# ── #')
    _separator_default(_buffer)
    _append_to_buffer(_buffer, '    # ─%s─ #' % ('─' * size_line), inspect.stack()[0][3])
    _append_to_buffer(_buffer, '    # This comment is used by Shinken to recognize this file, please do not edit or remove it.', inspect.stack()[0][3])
    _append_to_buffer(_buffer, '    # If done so, several parts of Shinken, like sanitize, may not work properly.', inspect.stack()[0][3])
    _append_to_buffer(_buffer, '    # %s %s' % (OVERRIDE_TYPE_KEY, data.strip()), line_type)
    _append_to_buffer(_buffer, '    # ─%s─ #' % ('─' * size_line), inspect.stack()[0][3])
    _separator_default(_buffer)


def parameter_text_warning(_buffer, data):
    # type: (List[str], str) -> None
    parameter_text_indent(_buffer, '/!\\ %s' % data, line_type=FORMATTER_KEYS.PARAMETER_TEXT_WARNING)


def parameter_text_default(_buffer, data):
    # type: (List[str], str) -> None
    separator_parameter(_buffer)
    split = data.split(CFG_SEPARATOR)
    nb_split = len(split)
    prefix = formatter_prefix(data, is_default=True)
    tag = ''
    if nb_split == 3:
        value = '%s' % split[1]
        tag = split[2]
    elif nb_split == 2:
        value = split[0]
        tag = split[1]
    else:
        value = split[0]
    
    formatter_values(_buffer, prefix=prefix, value=value, tag=tag, _last_line_type=inspect.stack()[0][3])


def parameter_text_recommended(_buffer, data):
    # type: (List[str], str) -> None
    parameter_list_indent(_buffer, 'Recommended : %s' % data, 2)


def parameter_list_example(_buffer, data, nb_indent=1):
    # type: (List[str], str, int) -> None
    parameter_list_indent(_buffer, 'Example : %s' % data, nb_indent=nb_indent, line_type=FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE)


def parameter_list_example_2(_buffer, data):
    # type: (List[str], str) -> None
    parameter_list_example(_buffer, data, 2)


def parameter_list_example_3(_buffer, data):
    # type: (List[str], str) -> None
    parameter_list_example(_buffer, data, 3)


def parameter_list(_buffer, data, suffix=''):
    # type: (List[str], str, str) -> None
    separator_parameter(_buffer)
    split = data.split(CFG_SEPARATOR)
    nb_split = len(split)
    prefix = formatter_prefix(data)
    tag = ''
    if nb_split == 3:
        value = split[1]
        tag = split[2]
    elif nb_split == 2:
        value = split[0]
        tag = split[1]
    else:
        value = split[0]
    formatter_values(_buffer, prefix=prefix, value=value, tag=tag, suffix=suffix, _last_line_type=inspect.stack()[0][3])


def parameter_list_indent(_buffer, data, nb_indent=1, line_type=FORMATTER_KEYS.PARAMETER_LIST_INDENT):
    # type: (List[str], str, int, str) -> None
    space_start = (int(nb_indent) * 5) * ' '
    prefix_indent = ' ->' if line_type == FORMATTER_KEYS.PARAMETER_LIST_INDENT else ''
    parameter_text(_buffer, '%s%s %s' % (space_start, prefix_indent, data), line_type=line_type)


def parameter_text_indent(_buffer, data, nb_indent=1, line_type=FORMATTER_KEYS.PARAMETER_TEXT_INDENT):
    # type: (List[str], str, int, str) -> None
    space_start = (int(nb_indent) * 5) * ' '
    parameter_text(_buffer, '%s %s' % (space_start, data), line_type=line_type)


def parameter_list_indent_2(_buffer, data):
    # type: (List[str], str) -> None
    parameter_list_indent(_buffer, data, 2)


def parameter_list_indent_3(_buffer, data):
    # type: (List[str], str) -> None
    parameter_list_indent(_buffer, data, 3)


def parameter_value(_buffer, data, is_commented=False):
    # type: (List[str], str, bool) -> None
    separator_parameter_value(_buffer)
    prop_name, value = data.split(CFG_SEPARATOR, 1)
    space = max((ALIGN_VALUES + 2 - len(prop_name)), 1) * ' '
    
    _append_to_buffer(_buffer, '    %s%s%s%s' % ('# ' if is_commented else '', prop_name, space, value), inspect.stack()[0][3])
    _separator_default(_buffer)


def daemon_parameter_value(_buffer, data, is_commented=False):
    # type: (List[str], str, bool) -> None
    separator_parameter_value(_buffer)
    prop_name, value = data.split(CFG_SEPARATOR, 1)
    
    _append_to_buffer(_buffer, '    %s%s=%s' % ('# ' if is_commented else '', prop_name, value), inspect.stack()[0][3])
    _separator_default(_buffer)


def parameter_value_comment(_buffer, data):
    # type: (List[str], str) -> None
    parameter_value(_buffer, data, is_commented=True)


def daemon_parameter_value_comment(_buffer, data):
    # type: (List[str], str) -> None
    daemon_parameter_value(_buffer, data, is_commented=True)


def formatter_h1(_buffer, data):
    # type: (List[str], str) -> None
    size_line = SIZE_LINE - len('# ┌┐ #')
    space_before = 1
    space_after = 4
    before = '─' * int(math.floor((size_line - len(data) - (space_after + space_before) * 2) / 2.0))
    after = '─' * int(math.ceil((size_line - len(data) - (space_after + space_before) * 2) / 2.0))
    _separator_default(_buffer)
    _append_to_buffer(_buffer, '    # ┌%s┐ #' % ('─' * size_line), inspect.stack()[0][3])
    _append_to_buffer(_buffer, '    # │%s%s%s%s%s%s%s│ #' % (' ' * space_before, before, ' ' * space_after, data.upper(), ' ' * space_after, after, ' ' * space_before), inspect.stack()[0][3])
    _append_to_buffer(_buffer, '    # └%s┘ #' % ('─' * size_line), inspect.stack()[0][3])
    _separator_default(_buffer)


def formatter_h2(_buffer, data):
    # type: (List[str], str) -> None
    size_line = 58
    space_before = 1
    space_after = 2
    before = '─' * int(math.floor((size_line - len(data) - (space_after + space_before) * 2) / 2.0))
    after = '─' * (int(math.ceil((size_line - len(data) - (space_after + space_before) * 2) / 2.0)) + SIZE_LINE - size_line - 2)
    _separator_default(_buffer)
    _append_to_buffer(_buffer, '    #%s%s%s%s%s%s%s#' % (' ' * space_before, before, ' ' * space_after, data, ' ' * space_after, after, ' ' * space_before), inspect.stack()[0][3])
    _separator_default(_buffer)


def formatter_values(_buffer, prefix='', value='', tag='', suffix='', _last_line_type=''):
    # type: (List[str], str, str, str, str, str) -> None
    _last_line_type = last_line_type or inspect.stack()[0][3]
    
    new_lines = []
    next_line_align = 0
    next_line_align_tag = 0
    next_line_tag = ''
    if NEW_LINE_IN_TAG in tag:
        tag, next_line = tag.split(NEW_LINE_IN_TAG)
        tag = ' ( %s' % tag
        next_line_tag = ' %s )' % next_line
        next_line_align_tag = len(value) + len(prefix)
    elif tag:
        tag = ' ( %s )' % tag
    
    if NEW_LINE_IN_TAG in value:
        new_lines = value.split(NEW_LINE_IN_TAG)
        value = new_lines.pop(0)
        next_line_align = len(prefix)
    
    if new_lines:
        space_end = (SIZE_LINE - len('#  ───') - len(prefix) - len(value) - len(tag)) * ' '
        _append_to_buffer(_buffer, '    # %s%s%s%s ───' % (prefix, value, tag, space_end), _last_line_type)
        for _line in new_lines:
            space_end = (SIZE_LINE - len('#  ───') - next_line_align - len(_line) - len(suffix)) * ' '
            _append_to_buffer(_buffer, '    # %s%s%s%s ───' % (next_line_align * ' ', _line, suffix, space_end), _last_line_type)
    else:
        space_end = (SIZE_LINE - len('#  ───') - len(prefix) - len(value) - len(tag) - len(suffix)) * ' '
        _append_to_buffer(_buffer, '    # %s%s%s%s%s ───' % (prefix, value, tag, suffix, space_end), _last_line_type)
        if next_line_tag:
            space_end = (SIZE_LINE - len('#  ───') - next_line_align_tag - len(next_line_tag) - len(suffix) - 2) * ' '
            _append_to_buffer(_buffer, '    #   %s%s%s%s ───' % (next_line_align_tag * ' ', next_line_tag, suffix, space_end), _last_line_type)


def formatter_prefix(data, is_default=False, _=False):
    # type: (str, bool, bool) -> str
    split = data.split(CFG_SEPARATOR)
    nb_split = len(split)
    prefix = ''
    if nb_split == 3:
        prefix = split[0]
    
    if is_default and nb_split > 2:
        prefix = '%s          Default : %s => ' % ('', prefix)
    elif is_default:
        prefix = '%s          Default : %s' % ('', prefix)
    elif nb_split > 2:
        prefix = '%s          ...     : %s => ' % ('', prefix)
    else:
        prefix = '%s          ...     : %s' % ('', prefix)
    
    return prefix


def format_cfg_def(cfg_def):
    # type: (str) -> str
    
    global SIZE_LINE
    SIZE_LINE = DEFAULT_SIZE_LINE
    
    _buffer = []
    FORMATTER_FUNCTIONS['description_cfg_formatter_version'](_buffer, str(FORMATTER_VERSION))
    index = 0
    for raw_line in cfg_def.splitlines():
        index = index + 1
        if not raw_line:
            continue
        op, _, data = raw_line.partition(CFG_SEPARATOR)
        
        op = op.strip()
        if not op:  # separator used as comment
            continue
        if op == FORMATTER_KEYS.FILE_FORMAT:
            continue
        try:
            formatter_function = FORMATTER_FUNCTIONS[op]
        except KeyError:
            print('Error: Formatter key [ %s ] does not exist :' % op, file=sys.stderr)
            print(raw_line, file=sys.stderr)
            continue
        try:
            formatter_function(_buffer, data)
        except Exception:
            print('Error in line : ', file=sys.stderr)
            print(raw_line, file=sys.stderr)
            traceback.print_exc()
    
    return '\n'.join(_buffer)


FORMATTER_FUNCTIONS = {
    FORMATTER_KEYS.SET_LINE_SIZE                    : set_line_size,
    FORMATTER_KEYS.DESCRIPTION_TITLE                : description_title,
    FORMATTER_KEYS.DESCRIPTION_TEXT                 : description_text,
    FORMATTER_KEYS.DESCRIPTION_CFG_FORMATTER_VERSION: description_cfg_formatter_version,
    FORMATTER_KEYS.DESCRIPTION_END                  : description_end,
    
    FORMATTER_KEYS.DEFINE_START                     : define_start,
    FORMATTER_KEYS.DEFINE_STOP                      : define_stop,
    
    FORMATTER_KEYS.PARAMETER_TITLE                  : parameter_title,
    FORMATTER_KEYS.PARAMETER_TITLE_MANDATORY        : parameter_title_mandatory,
    FORMATTER_KEYS.PARAMETER_TEXT                   : parameter_text,
    FORMATTER_KEYS.PARAMETER_TEXT_WARNING           : parameter_text_warning,
    FORMATTER_KEYS.PARAMETER_TEXT_INDENT            : parameter_text_indent,
    FORMATTER_KEYS.PARAMETER_TEXT_DEFAULT           : parameter_text_default,
    FORMATTER_KEYS.PARAMETER_TEXT_RECOMMENDED       : parameter_text_recommended,
    
    FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE           : parameter_list_example,
    FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE_2         : parameter_list_example_2,
    FORMATTER_KEYS.PARAMETER_LIST_EXAMPLE_3         : parameter_list_example_3,
    FORMATTER_KEYS.PARAMETER_LIST_DEFAULT           : parameter_text_default,
    FORMATTER_KEYS.PARAMETER_LIST                   : parameter_list,
    FORMATTER_KEYS.PARAMETER_LIST_INDENT            : parameter_list_indent,
    FORMATTER_KEYS.PARAMETER_LIST_INDENT_2          : parameter_list_indent_2,
    FORMATTER_KEYS.PARAMETER_LIST_INDENT_3          : parameter_list_indent_3,
    
    FORMATTER_KEYS.PARAMETER_VALUE                  : parameter_value,
    FORMATTER_KEYS.PARAMETER_COMMENTED              : parameter_value_comment,
    
    FORMATTER_KEYS.OVERRIDE_FROM                    : override_from,
    FORMATTER_KEYS.OVERRIDE_TYPE                    : override_type,
    
    FORMATTER_KEYS.DAEMON_PARAMETER_VALUE           : daemon_parameter_value,
    FORMATTER_KEYS.DAEMON_PARAMETER_COMMENTED       : daemon_parameter_value_comment,
    
    FORMATTER_KEYS.HEADER_1                         : formatter_h1,
    FORMATTER_KEYS.HEADER_2                         : formatter_h2,
    FORMATTER_KEYS.BREAK_LINE                       : _separator_default,
}
