#!/usr/bin/env python3-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import datetime
import hashlib
import re
import shutil
from collections import OrderedDict
from copy import deepcopy
from os import listdir, makedirs, getenv
from os.path import abspath, dirname, join, isfile, splitext, basename, exists, commonpath, split

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.cfg_formatter.cfg_formatter import CFG_SEPARATOR, FORMATTER_KEYS, format_cfg_def, OVERRIDE_TYPE_KEY
from shinkensolutions.system_tools import run_command_with_return_code

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Set, List, Optional, Any, Union


class VERBOSE_LEVELS:
    NONE = 0
    ERROR = 1
    INFO = 2
    DEBUG = 3


class ShinkenConfigWriterException(Exception):
    pass


VERBOSE_LEVEL = VERBOSE_LEVELS.NONE
DEFAULT_CONFIGURATION_FILES_PATH = '/etc/shinken/_default/'

IGNORED_PROPERTIES = ('module_type',)

UPDATE_DIR = ''
DO_NOT_DELETE_KEYS = ['imported_from', 'module_name', 'module_type']
BACKUP_FOLDER_NAME = 'configuration_files_updated_by_sanitize'


class FORMATTER_PATHS:
    SCRIPT_FOLDER = dirname(abspath(__file__))
    GENERATED_CFG_FOLDER = join(SCRIPT_FOLDER, 'cfg')
    CFG_DEF_FOLDER = join(SCRIPT_FOLDER, 'cfg_def')
    PREVIOUS_CFG_DEF_FOLDER = join(SCRIPT_FOLDER, 'previous_cfg_def')
    CONFIG_UPDATE_FOLDER = join(SCRIPT_FOLDER, 'updates')


class SHINKEN_OBJECTS:
    SYNCHRONIZER = 'synchronizer'
    REALM = 'realm'
    SCHEDULER = 'scheduler'
    REACTIONNER = 'reactionner'
    ARBITER = 'arbiter'
    RECEIVER = 'receiver'
    POLLER = 'poller'
    BROKER = 'broker'
    MODULE = 'module'


class FORMATTER_FILES:
    ARCHITECTURE_EXPORT = 'architecture-export'
    SYNCHRONIZER_IMPORT = 'synchronizer-import'
    BROKER_MODULE_LIVEDATA = 'broker-module-livedata'
    SOURCE_DISCOVERY = 'discovery'
    EVENT_MANAGER_READER = 'event_manager_reader'
    EVENT_MANAGER_WRITER = 'event_manager_writer'
    GRAPHITE = 'graphite'
    LISTENER_REST = 'listener-rest'
    LIVEDATA_MODULE_SLA_PROVIDER = 'livedata-module-sla-provider'
    MODULE_MONGO = 'mongodb'
    MONGOS_SOCKET_KEEP_ALIVE = 'mongos_socket_keep_alive'
    MODULE_RETENTION_MONGO = 'retention_mongodb'
    SLA = 'sla'
    SYNCUI = 'syncui'
    BASE_CONFIG_FROM_SYNCHRONIZER = 'base_config_from_synchronizer'
    BASE_CONFIG_FROM_SYNCHRONIZER_OVERRIDE = 'base_config_from_synchronizer_cfg_overload'
    WEBUI = 'webui'
    WEBUI_DEFAULT = 'webui_default'
    WEBUI_OVERRIDE = 'webui_cfg_overload'
    WEBUI_MODULE_SERVICE_WEATHER = 'webui_module_service_weather'
    WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_GRID_LAYOUT = 'webui_module_service_weather__default_grid_layout'
    WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_INFO_BAR_LAYOUT = 'webui_module_service_weather__default_info_bar_layout'
    WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_WIDGETS_LAYOUT = 'webui_module_service_weather__default_widgets_layout'
    WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_NOTIFICATIONS_SETTINGS = 'webui_module_service_weather__default_notifications_settings'
    SYNCHRONIZER_COLLECTOR_VMWARE = 'synchronizer-collector-vmware'
    
    INPUT_FILE_NAMES = (
        ARCHITECTURE_EXPORT,
        SYNCHRONIZER_IMPORT,
        BROKER_MODULE_LIVEDATA,
        SOURCE_DISCOVERY,
        EVENT_MANAGER_READER,
        EVENT_MANAGER_WRITER,
        GRAPHITE,
        LIVEDATA_MODULE_SLA_PROVIDER,
        MODULE_MONGO,
        MONGOS_SOCKET_KEEP_ALIVE,
        MODULE_RETENTION_MONGO,
        WEBUI_DEFAULT,
        SLA,
        SYNCUI,
        BASE_CONFIG_FROM_SYNCHRONIZER,
        BASE_CONFIG_FROM_SYNCHRONIZER_OVERRIDE,
        WEBUI,
        WEBUI_OVERRIDE,
        WEBUI_MODULE_SERVICE_WEATHER,
        WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_GRID_LAYOUT,
        WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_INFO_BAR_LAYOUT,
        WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_WIDGETS_LAYOUT,
        WEBUI_MODULE_SERVICE_WEATHER__DEFAULT_NOTIFICATIONS_SETTINGS,
        SYNCHRONIZER_COLLECTOR_VMWARE,
        LISTENER_REST,
    )
    
    CFG = 'cfg'
    CFG_DEF = 'cfg_def'


def set_verbose_level(verbose_level):
    # type: (int) -> None
    global VERBOSE_LEVEL
    VERBOSE_LEVEL = verbose_level


def _get_file_md5(file_name):
    # type: (str) -> str
    with open(file_name, 'rb') as f:
        data = f.read()
        return hashlib.md5(data).hexdigest()


def _compute_md5(content):
    # type: (str) -> str
    return hashlib.md5(content.replace('\r\n', '\n').encode('utf-8')).hexdigest()


def _get_cfg_def_file_path_if_exists(cfg_def_without_ext, is_old_cfg=False):
    # type: (str, bool) -> Optional[str]
    folder_path = FORMATTER_PATHS.PREVIOUS_CFG_DEF_FOLDER if is_old_cfg else FORMATTER_PATHS.CFG_DEF_FOLDER
    file_path = join(folder_path, '%s.%s' % (cfg_def_without_ext, FORMATTER_FILES.CFG_DEF))
    return file_path if isfile(file_path) else None


def _does_cfg_def_file_exists(file_name_without_ext, is_old_cfg=False):
    # type: (str, bool) -> bool
    return bool(_get_cfg_def_file_path_if_exists(file_name_without_ext, is_old_cfg))


def _get_cfg_def_to_use(config, config_object, config_type):
    # type: (Dict[str, Any], Dict[str, Any], str) -> Optional[str]
    if config_type == SHINKEN_OBJECTS.MODULE:
        module_type = config_object.get('module_type')[0]
        if module_type == 'event_container':
            if config_object.get('module_name')[0] in set(module_name.strip() for m in config[SHINKEN_OBJECTS.MODULE] if m.get('module_type', None) == ['webui'] for module_name in m.get('modules', [''])[0].split(',')):
                module_type = 'event_manager_reader'
            
            elif config_object.get('module_name')[0] in set(module_name.strip() for m in config[SHINKEN_OBJECTS.BROKER] for module_name in m.get('modules', [''])[0].split(',')):
                module_type = 'event_manager_writer'
        
        if _does_cfg_def_file_exists(module_type):
            return module_type
        if _does_cfg_def_file_exists(module_type.replace('_', '-')):
            return module_type.replace('_', '-')
    
    else:
        if _does_cfg_def_file_exists(config_type):
            return config_type
    
    return None


def _read_file_cfg_def(file_name, is_old_cfg=False):
    # type: (str, bool) -> str
    if is_old_cfg:
        folder_path = FORMATTER_PATHS.PREVIOUS_CFG_DEF_FOLDER
    else:
        folder_path = FORMATTER_PATHS.CFG_DEF_FOLDER
    file_path = join(folder_path, '%s.%s' % (file_name, FORMATTER_FILES.CFG_DEF))
    if not isfile(file_path):
        raise ShinkenConfigWriterException('This provided file name %s does not have a cfg_def file associated' % file_path)
    with open(file_path, 'r', encoding='utf-8') as fd:
        ret = fd.read()
    return ret


def _get_parameters_managed_by_cfg_def(cfg_def_to_use):
    # type: (str) -> Set[str]
    cfg_def = _read_file_cfg_def(cfg_def_to_use, is_old_cfg=False)
    managed_keys = set()
    file_lines = cfg_def.splitlines()
    for line in file_lines:
        line_data = line.split(CFG_SEPARATOR)
        if len(line_data) == 1:
            continue
        formatter_file_key = line_data[0].strip()
        if not formatter_file_key:
            continue
        parameter_name = line_data[1].strip()
        if formatter_file_key in FORMATTER_KEYS.ALL_PARAMETER_KEYS:
            managed_keys.add(parameter_name)
    return managed_keys


def _extract_file_path_from_imported_from(file_path):
    # type: (str) -> str
    if isinstance(file_path, list):
        file_path = file_path[0]
    file_path = file_path.strip().rsplit(':', 1)[0]
    return file_path


def _get_parameters_formats_from_cfg_def_file_format_key(file_format_key):
    # type: (str) -> (str, str)
    
    if file_format_key == FORMATTER_KEYS.DAEMON_FILE_FORMAT:
        return FORMATTER_KEYS.DAEMON_PARAMETER_VALUE, FORMATTER_KEYS.DAEMON_PARAMETER_COMMENTED
    elif file_format_key in (FORMATTER_KEYS.MODULE_FILE_FORMAT, FORMATTER_KEYS.OVERRIDE_FILE_FORMAT):
        return FORMATTER_KEYS.PARAMETER_VALUE, FORMATTER_KEYS.PARAMETER_COMMENTED
    else:
        raise ValueError


def _replace_def_with_conf_values(cfg_def, conf, is_base_config_file=False):
    # type: (str, Dict[str, Any], bool) -> str
    # is_base_config_file -> is /etc/shinken/shinken.cfg & /etc/shinken/synchronizer.cfg file format.
    file_lines = cfg_def.splitlines()
    computed_conf = []
    try:
        parameter_value_format, commented_value_format = _get_parameters_formats_from_cfg_def_file_format_key(file_lines[0].split(CFG_SEPARATOR)[1].strip())
    except (ValueError, IndexError):
        raise ValueError('Unknown cfg file format. cfg_def file(s) is(are) corrupted')
    # This flag here is set to True when encountering a "define stop" in cfg_def file, in this case we need to append this key to the result
    # In overrides types of cfg_def, we do not have this key
    need_to_stop_definition = False
    unmanaged_keys = deepcopy(conf)
    for default_managed_key in DO_NOT_DELETE_KEYS:
        unmanaged_keys.pop(default_managed_key, None)
    
    for line in file_lines:
        if line.strip() == FORMATTER_KEYS.DEFINE_STOP:
            need_to_stop_definition = True
            continue
        
        line_data = line.split(CFG_SEPARATOR)
        
        if len(line_data) == 1:
            computed_conf.append(line)
            continue
        
        formatter_file_key = line_data[0].strip()
        if not formatter_file_key:
            computed_conf.append('')
            continue
        
        parameter_name = line_data[1].strip()
        if formatter_file_key in FORMATTER_KEYS.ALL_PARAMETER_KEYS:
            remove_parameter_from_conf = True
            
            if formatter_file_key in IGNORED_PROPERTIES:
                computed_conf.append(line)
            
            conf_value = conf.get(parameter_name, None)
            if conf_value is None:
                line_data[0] = commented_value_format
                computed_conf.append(CFG_SEPARATOR.join(line_data))
            else:
                if isinstance(conf_value, list):
                    if parameter_name in ('cfg_file', 'cfg_dir') and is_base_config_file:
                        cfg_def_value = line_data[2]
                        if cfg_def_value in conf_value:
                            unmanaged_keys[parameter_name].remove(cfg_def_value)
                        else:
                            # if we don't find the cfg_file or cfg_dir with cfg_def value in current configuration we add it as commented value
                            formatter_file_key = ''
                            line_data[0] = FORMATTER_KEYS.DAEMON_PARAMETER_COMMENTED
                        
                        if len(unmanaged_keys[parameter_name]) > 0:
                            remove_parameter_from_conf = False
                    
                    else:
                        line_data[2] = ','.join(conf_value)
                else:
                    line_data[2] = conf_value
                
                if remove_parameter_from_conf:
                    unmanaged_keys.pop(parameter_name, None)
                
                if formatter_file_key == commented_value_format:
                    line_data[0] = parameter_value_format
                
                computed_conf.append(CFG_SEPARATOR.join(line_data))
        elif formatter_file_key == FORMATTER_KEYS.OVERRIDE_FROM:
            # We want to ignore the override_from keys because we will treat them separately at the end
            continue
        else:
            computed_conf.append(line)
    
    if 'overrides' in conf:
        unmanaged_keys.pop('overrides', None)
        if not is_base_config_file:
            for override_conf in conf.get('overrides'):
                imported_from = override_conf.get('imported_from', '')
                computed_conf.append('%s%s%s' % (FORMATTER_KEYS.OVERRIDE_FROM, CFG_SEPARATOR, imported_from))
    if unmanaged_keys:
        computed_conf.append('h1; Key(s) not reformatted')
    if 'cfg_file' in unmanaged_keys:
        for cfg_file in unmanaged_keys.get('cfg_file'):
            computed_conf.append(CFG_SEPARATOR.join([parameter_value_format, 'cfg_file', cfg_file]))
        unmanaged_keys.pop('cfg_file')
    
    if 'cfg_dir' in unmanaged_keys:
        for cfg_dir in unmanaged_keys.get('cfg_dir'):
            computed_conf.append(CFG_SEPARATOR.join([parameter_value_format, 'cfg_dir', cfg_dir]))
        unmanaged_keys.pop('cfg_dir')
    
    for (unmanaged_key, conf_value) in unmanaged_keys.items():
        if isinstance(conf_value, list):
            conf_value = ','.join(conf_value)
        computed_conf.append(CFG_SEPARATOR.join([parameter_value_format, unmanaged_key, conf_value]))
    
    if need_to_stop_definition:
        computed_conf.append(FORMATTER_KEYS.DEFINE_STOP)
    
    return '\n'.join(computed_conf)


def compute_cfg_from_conf(file_name, conf, is_base_config_file=False, is_old_cfg=False):
    # type: (str, Dict , bool, bool) -> str
    cfg_def = _read_file_cfg_def(file_name, is_old_cfg)
    computed_cfg_from_conf = _replace_def_with_conf_values(cfg_def, conf, is_base_config_file)
    return computed_cfg_from_conf


def write_file(file_name, content, append_file=False):
    # type: (str, str, bool) -> None
    content = content.replace('\r\n', '\n')
    if append_file:
        write_mode = 'a'
    else:
        write_mode = 'w+'
    with open(file_name, write_mode, newline='\n', encoding='utf-8') as fd:
        if append_file:
            fd.write('\n')
        fd.write(content)


def backup_file(file_path):
    # type: (str) -> None
    timestamp = datetime.datetime.now().strftime('%Y%m%d%H%M%S')
    file_directory, file_name = split(file_path)
    file_name = splitext(file_name)[0]
    if commonpath([DEFAULT_CONFIGURATION_FILES_PATH]) == commonpath([DEFAULT_CONFIGURATION_FILES_PATH, file_directory]):
        file_name = f'default_{file_name}'
    # do not inverse the order with the previous if condition
    if file_directory.startswith('/'):
        file_directory = file_directory[1:]
    if UPDATE_DIR:
        configuration_file_backup_dir = join(UPDATE_DIR, BACKUP_FOLDER_NAME, file_directory)
        if not exists(configuration_file_backup_dir):
            makedirs(configuration_file_backup_dir)
        backup_file_path = join(configuration_file_backup_dir, '%s_%s.%s' % (file_name, timestamp, 'bkp'))
        try:
            shutil.copy2(file_path, backup_file_path)
        except Exception as e:
            print('Error during backup: {%s}' % e)


def format_and_check_config_file_modification(config_object, updated_files, already_written_paths, cfg_def_to_use, is_base_config_file=False, need_old_cfg=False):
    # type: (Dict, List[str], Set[str], str,bool, bool) -> None
    module_cfg_file_path = _extract_file_path_from_imported_from(config_object.get('imported_from', ''))
    computed_cfg_from_conf = compute_cfg_from_conf(cfg_def_to_use, config_object, is_base_config_file, need_old_cfg)
    new_cfg_file = format_cfg_def(computed_cfg_from_conf)
    if isfile(module_cfg_file_path):
        file_md5 = _get_file_md5(module_cfg_file_path)
        if file_md5 != _compute_md5(new_cfg_file):
            # If we detect that the file we currently want to use has already been written, we want to append the content and not override it.
            # This is because the file contains multiple module definition.
            if module_cfg_file_path in already_written_paths:
                write_file(module_cfg_file_path, new_cfg_file, append_file=True)
            else:
                # Will override the existing file
                backup_file(module_cfg_file_path)
                write_file(module_cfg_file_path, new_cfg_file)
            updated_files.append(module_cfg_file_path)
    else:
        write_file(module_cfg_file_path, new_cfg_file)
        updated_files.append(module_cfg_file_path)
    already_written_paths.add(module_cfg_file_path)


def _merge_all_overrides(overrides):
    # type: (List[Dict[str, str]]) -> Dict[str, str]
    
    merged_overrides = {}
    for override in overrides:
        merged_overrides.update(override)
    
    merged_overrides.pop('imported_from', None)
    return merged_overrides


def _get_value_from_key(cfg_def, key):
    # type: (str, str) -> Optional[str]
    file_lines = cfg_def.splitlines()
    
    for line in file_lines:
        line_data = line.split(CFG_SEPARATOR)
        if len(line_data) == 3:
            if line_data[1].strip() == key:
                return line_data[2].strip()
    
    return None


def get_parameters_from_formatter_key(cfg_def, formatter_key):
    # type: (str, str) -> List[str]
    file_lines = cfg_def.splitlines()
    parameters = []
    
    for line in file_lines:
        line_data = line.split(CFG_SEPARATOR)
        if line_data[0].strip() == formatter_key:
            parameters.append(line_data[1])
    
    return parameters


def _get_overrides_cfg_def_and_path_in_cfg_def(cfg_def_file_path, is_old_cfg=False):
    # type: (str, bool) -> Dict[str, str]
    overrides_cfg_def_and_path = OrderedDict()
    
    cfg_def = _read_file_cfg_def(cfg_def_file_path, is_old_cfg)
    default_override_paths = get_parameters_from_formatter_key(cfg_def, FORMATTER_KEYS.OVERRIDE_FROM)
    
    for default_override_path in default_override_paths:
        cfg_def_to_use = splitext(basename(default_override_path))[0]
        if cfg_def_to_use == cfg_def_file_path:
            cfg_def_to_use = '_'.join([cfg_def_to_use, 'default'])
        overrides_cfg_def_and_path[default_override_path] = cfg_def_to_use
    
    return overrides_cfg_def_and_path


def _format_override_config_files(config_object, updated_files, already_written_paths, parent_config_object_cfg_def, need_old_cfg):
    # type: (Dict, List, Set[str],str, bool) -> None
    overrides = config_object.get('overrides', None)
    if not overrides:
        return
    
    overrides_cfg_def_and_path = _get_overrides_cfg_def_and_path_in_cfg_def(parent_config_object_cfg_def, need_old_cfg)
    for override in overrides:
        cfg_def_to_use = None
        if _get_cfg_def_name_from_override_file_path(override['imported_from']) in overrides_cfg_def_and_path.values():
            cfg_def_to_use = _get_cfg_def_name_from_override_file_path(override['imported_from'])
        elif override['imported_from'] in overrides_cfg_def_and_path:
            cfg_def_to_use = overrides_cfg_def_and_path[override['imported_from']]
        if cfg_def_to_use:
            format_and_check_config_file_modification(override, updated_files, already_written_paths, cfg_def_to_use)


def _format_override_config_file_synchronizer(config_object, updated_files, already_written_paths):
    # type: (Dict, List, Set[str]) -> None
    
    cfg_def_to_use = 'base_config_from_synchronizer_cfg_overload'
    
    for overload_file in config_object.get('overrides', []):
        format_and_check_config_file_modification(overload_file, updated_files, already_written_paths, cfg_def_to_use, True)


def _get_cfg_def_name_from_override_file_path(file_path):
    # type: (str) -> Optional[str]
    if not isfile(file_path):
        return None
    
    with open(file_path, 'r', encoding='utf-8') as override_file:
        for line in override_file.readlines():
            if OVERRIDE_TYPE_KEY not in line:
                continue
            _, override_type_found = line.split(OVERRIDE_TYPE_KEY, 1)
            return override_type_found.strip()
    
    return None


def update_configuration_files_with_updates_values(result, already_written_paths, configuration_updates_to_apply=None):
    # type: (Set,Set, Union[Dict[str, Dict[str, str]], None]) -> None
    
    if configuration_updates_to_apply is None:
        return
    for file_path in configuration_updates_to_apply:
        _file_path = file_path.split(':')[0]
        if _file_path not in already_written_paths:
            already_written_paths.add(_file_path)
            backup_file(_file_path)
        for new_pattern, old_pattern in configuration_updates_to_apply[file_path].items():
            sed_command = r'''sed -i 's#%(old_pattern)s#%(new_pattern)s#g' %(file_to_edit)s''' % {
                'old_pattern' : old_pattern,
                'new_pattern' : new_pattern,
                'file_to_edit': _file_path,
            }
            _code, _stdout, _stderr = run_command_with_return_code(sed_command)
            if _code == 0:
                result.add(_file_path.split(':')[0])


def format_files_with_conf_value(config, result, already_written_paths):
    # type: (Dict, Dict, Set) -> None
    for config_type in config.keys():
        
        _file_name = ''
        for config_object in config.get(config_type):
            updated_files = []
            cfg_def_to_use = _get_cfg_def_to_use(config, config_object, config_type)
            if not cfg_def_to_use:
                continue
            if config_type == 'base_config_from_synchronizer':
                _format_override_config_file_synchronizer(config_object, updated_files, already_written_paths)
                format_and_check_config_file_modification(config_object, updated_files, already_written_paths, cfg_def_to_use, is_base_config_file=True, need_old_cfg=False)
            
            else:
                # Do not inverse the order !!
                _format_override_config_files(config_object, updated_files, already_written_paths, cfg_def_to_use, need_old_cfg=False)
                format_and_check_config_file_modification(config_object, updated_files, already_written_paths, cfg_def_to_use, False)
            
            if updated_files:
                
                if config_type == SHINKEN_OBJECTS.MODULE:
                    module_type = config_object.get('module_type')[0]
                    if module_type in list(result.keys()):
                        result[module_type] = result[module_type] + updated_files
                    else:
                        result[module_type] = updated_files
                else:
                    if config_type in list(result.keys()):
                        result[config_type] = result[config_type] + updated_files
                    else:
                        result[config_type] = updated_files


def format_all_files_with_conf_value(config, configuration_updates_to_apply=None, reformat_conf_file=False, backup_dir=''):
    # type: (Dict,  Union[Dict[str, Dict[str, str]], None], str, str) -> Dict
    global UPDATE_DIR
    result = {'files_with_key_updated': set(), 'files_formatted': {}}
    already_written_paths = set()
    UPDATE_DIR = backup_dir if backup_dir else getenv('OUTPUT_FOLDER')
    if UPDATE_DIR:
        update_configuration_files_with_updates_values(result['files_with_key_updated'], already_written_paths, configuration_updates_to_apply)
        if reformat_conf_file:
            format_files_with_conf_value(config, result['files_formatted'], already_written_paths)
    
    return result


def does_cfg_def_contains_key(cfg_def_name, key_to_search):
    # type: (str, str) -> bool
    cfg_def_path = _get_cfg_def_file_path_if_exists(cfg_def_name)
    
    if not cfg_def_path:
        return False
    
    # Matching all lines starting with our keyword and followed by spaces and the CFG delimiter and then anything
    # It will also work if the keyword itself has spaces / tabs before it
    # Example : KEYWORD    ;the_value_of_the_keyword
    key_regex = re.compile(r'^\s*%s\s*%s.*$' % (key_to_search, CFG_SEPARATOR), re.MULTILINE)
    with open(cfg_def_path, 'r', encoding='utf-8') as cfg_def_file:
        cfg_def_content = cfg_def_file.read()
    
    return bool(key_regex.search(cfg_def_content))


def pycharm_format_file(file_name):
    # type: (str) -> None
    if VERBOSE_LEVEL == VERBOSE_LEVELS.INFO:
        print('formatting file %s' % file_name)
    cfg_def = _read_file_cfg_def(file_name)
    _buffer = format_cfg_def(cfg_def)
    file_path = join(FORMATTER_PATHS.GENERATED_CFG_FOLDER, '%s.%s' % (file_name, FORMATTER_FILES.CFG))
    write_file(file_path, _buffer)


def pycharm_format_all_files():
    # type: () -> None
    files = listdir(FORMATTER_PATHS.CFG_DEF_FOLDER)
    
    try:
        shutil.rmtree(FORMATTER_PATHS.GENERATED_CFG_FOLDER)
    except Exception:
        pass
    makedirs(FORMATTER_PATHS.GENERATED_CFG_FOLDER)
    
    for _file in files:
        _file_name, _ = splitext(_file)
        if _file_name in FORMATTER_FILES.INPUT_FILE_NAMES:
            pycharm_format_file(_file_name)


def main():
    print('start formatting cfg files.')
    set_verbose_level(VERBOSE_LEVELS.INFO)
    pycharm_format_all_files()
    print('formatting cfg files done.')
