#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import Dict, TYPE_CHECKING, Any
from ..component.input import InputStringComp
from ..object.html.tag import TagHtml
from ..object.shinken_object import ShinkenObject, ShinkenObjectContainer

if TYPE_CHECKING:
    from shinkensolutions.common_ui.factory.ui_factory_component import UiFactoryComponent
    from shinkensolutions.common_ui.component.button.generic_button import GenericButton

ShinkenObjectType = str or bytes  # type of shinken_object_type. For type hint.
ShinkenObjectName = str or bytes  # type of shinken_object_name. For type hint.


class HTML_TYPE:
    BUTTON = 'button'
    FILTER = 'filter'
    PROPERTY = 'property'
    OTHER = 'other'
    INPUT = 'input'
    CHECKBOX = 'check_box'
    TABLE = 'table'
    HTML = 'html'


class HtmlDistributor:
    
    def __init__(self, ui_factory_component):
        # type: (UiFactoryComponent) -> None
        self._ui_factory_component = ui_factory_component
        self._content = {}  # type: Dict[ShinkenObjectType, Dict[ShinkenObjectName, ShinkenObject]]
    
    
    def add(self, type_object, name, shinken_object):
        # type: (ShinkenObjectType, ShinkenObjectName, ShinkenObject) -> None
        if not shinken_object:
            return
        if type_object not in self._content:
            self._content[type_object] = {}
        self._content[type_object][name] = shinken_object
    
    
    def get(self, type_object, name_object):
        # type: (ShinkenObjectType,ShinkenObjectName) -> ShinkenObject
        return self._content[type_object][name_object]
    
    
    def has(self, type_object, name_object):
        # type: (ShinkenObjectType,ShinkenObjectName) -> bool
        if type_object not in self._content:
            return False
        
        if name_object not in self._content[type_object]:
            return False
        
        return True
    
    
    def get_html(self, type_object, name_object):
        # type: (ShinkenObjectType,ShinkenObjectName) -> str
        if self.has(type_object, name_object):
            return self.get(type_object, name_object).get_html()
        else:
            return ''
    
    
    def add_button(self, key, data_name='', label=None):
        # type: (str, str, Any) -> GenericButton
        if not data_name:
            data_name = key
        _to_add = self._ui_factory_component.build_generic_button(data_name)
        _to_add.set_label(label)
        _to_add.set_tag(TagHtml('div', internal_class='shinken-button-container'))
        _to_add.add_data('name', data_name)
        self.add(HTML_TYPE.BUTTON, key, _to_add)
        return _to_add
    
    
    def add_input(self, key, current_value='', name='', placeholder=''):
        # type: (str, str, str, str)-> InputStringComp
        _to_add = InputStringComp(self._ui_factory_component.translate, name=name)
        _to_add.set_value(current_value)
        _to_add.set_place_holder(placeholder)
        self.add(HTML_TYPE.INPUT, key, _to_add)
        return _to_add
    
    
    def add_html(self, key, html):
        # type: (str, str) -> ShinkenObject
        _to_add = ShinkenObjectContainer(html)
        self.add(HTML_TYPE.HTML, key, _to_add)
        return _to_add
