//__lib__ lib_common_V02_01
"use strict";

SHINKEN_GRID.Grid           = function ( data ) {
    this.uuid = SHINKEN.TOOLS.STRING.buildUUID();
    this.init( data );
};
SHINKEN_GRID.Grid.prototype = {
    init             : function ( data ) {
        this.zoom_percent            = 100;
        this.zoom_percent_for_widget = 100;
        this.initPhase();
        this.initCounterCommon();
        this.initInternal();
        this.tile_pixel_size = 0;
        this.class           = "shinken-grid";
        if ( data ) {
            this.updateData( data );
            this.max_nb_tiles = this.grid_elements.getMaxTileInHeight();
        }
    },
    initCounterCommon: function () {
        this.counters = {};
        this.initCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.ERRORS );
        this.initCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.WARNINGS );
        this.initCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.OVERLAPPING );
        this.initCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.SELECTED );
    },
    //********************************************  JSON  *****************************************************//
    _getValidOwnProperty: function () {
        return new SHINKEN_VALIDATION.Parameters( [
            [SHINKEN_GRID.CONST.PARAM.GRID.TILES_MAX_X, SHINKEN_VALIDATION.CONST.NUMBER.INTEGER_POSITIVE],
            [SHINKEN_GRID.CONST.PARAM.GRID.TILES_MAX_Y, SHINKEN_VALIDATION.CONST.NUMBER.INTEGER_POSITIVE],
            [SHINKEN_GRID.CONST.PARAM.GRID.SEPARATION_GRID_ELEMENT_MIN, SHINKEN_VALIDATION.CONST.NUMBER.INTEGER_POSITIVE],
            [SHINKEN_GRID.CONST.PARAM.GRID.SEPARATION_GRID_ELEMENT_MAX, SHINKEN_VALIDATION.CONST.OTHERS.DEFINE_STRING_OR_INTEGER_POSITIVE, [SHINKEN_GRID.CONST.SPECIAL_VALUE.SEPARATION_GRID_ELEMENT_MAX_UNLIMITED, this.getDefaultValueLabel()]],
            [SHINKEN_GRID.CONST.PARAM.GRID.SEPARATION_GRID_ELEMENT_PERCENT, SHINKEN_VALIDATION.CONST.NUMBER.FLOAT_POSITIVE],
            [SHINKEN_GRID.CONST.PARAM.GRID.TYPE, SHINKEN_VALIDATION.CONST.LIST.INCLUDE_IN, ["grid"], true]
        ] );
    },
    _getValidOwnChildren: function () {
        return new SHINKEN.OBJECT.DefaultConfigurationChildren( [
            [SHINKEN_GRID.CONST.PARAM.GRID.GRID_ELEMENTS, "WEATHER.Cells"],
            [SHINKEN_GRID.CONST.PARAM.GRID.LAYOUTS, "SHINKEN.OBJECT.DefaultConfigurationGrid"]
        ] );
    },
    //********************************************  EVENT LISTENER  *****************************************************//
    doActionAfter                      : function ( event_name, param, param_2, param_3 ) {
        switch ( event_name ) {
            case "compute_html_done":
                break;
            case "delete_selected_grid_cell":
                this.deleteGridElements( this.selected_grid_cell.extractContentIfHasCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.DELETED ) );
                SHINKEN.OBJECT.ConfirmationPopup_V3.getInstance().hide();
                break;
            case "start_selection_by_zone":
                this.resetOverlayFrame();
                //if ( this.overlayed_frame ) {
                //    this.resetOverlayFrame();
                //    break;
                //}
                this.overlayed_frame = new SHINKEN.OBJECT.OverlayFrameZoneSelection( this.tile_pixel_size );
                this.overlayed_frame.init( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].clientX, param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].clientY, param[ "dom_relative_position_container_2024_07_18" ] );
                this.overlayed_frame.computeHtml();
                this.addDomElement( this.overlayed_frame.getDomElement() );
                break;
            case "start_space_box":
                this.resetOverlayFrame();
                this.doActionAfter( "unselect_all_other_shinken_grid_cell" );
                this.overlayed_frame = param[ "space_box_type" ] === SHINKEN.OBJECT.SPACEBOX.CONST.TYPE.CREATION ? new SHINKEN.OBJECT.SpaceBoxCreation( this ) : new SHINKEN.OBJECT.SpaceBoxDeletion( this );
                this.overlayed_frame.init( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].clientX, param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].clientY, param[ "dom_relative_position_container_2024_07_18" ] );
                this.overlayed_frame.computeHtml();
                this.dom_element.appendChild( this.overlayed_frame.getDomElement() );
                this.addDomElement( this.overlayed_frame.getDomElement() );
                this.overlayed_frame.doActionAfter( "mouse_down" );
                break;
            case "on_scroll":
                this.grid_elements.doActionAfter( event_name, param, this );
                if ( this.overlayed_frame ) {
                    const dict_param = {
                        [ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ]: param,
                        [ "grid_2024_07_19" ]                   : this,
                        [ "grid_elements_2024_07_19" ]          : this.grid_elements
                    };
                    this.overlayed_frame.doActionAfter( event_name, dict_param, param_2, param_3 );
                    switch ( this.overlayed_frame.getObjectClassName() ) {
                        case "SHINKEN.OBJECT.GridCellResizer":
                        case "SHINKEN.OBJECT.OverlayFrameZoneSelection":
                            break;
                        default:
                            this.updateSelectedCellsWithDelta();
                            break;
                    }
                }
                break;
            case "mouse_leave":
                this.updateHighLightedTile( true );
                this.doActionAfter( "reset_grid_cell_to_add" );
                SHINKEN.TOOLS.MOUSE.unsetLastMouseEvent();
                break;
            case "change_zoom_percent_for_widget":
                this.zoom_percent_for_widget = param[ SHINKEN_GRID.CONST.PARAM.GRID.ZOOM_PERCENT_FOR_WIDGET ];
                this.doActionAfter( "reset_grid_cell_to_add" );
                break;
            case "mouse_wheel":
                this.updateHighLightedTile( true );
                break;
            case "mouse_move":
                if ( this.actionCurrentlyInProgress() ) {
                    this.scrollWhenAtEdgeOfView( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ] );
                }
                if ( this.overlayed_frame ) {
                    param[ "grid_elements_2024_07_19" ] = this.grid_elements;
                    param[ "grid_2024_07_19" ]          = this;
                    this.overlayed_frame.doActionAfter( event_name, param );
                }
                SHINKEN.TOOLS.MOUSE.setLastMouseEvent( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ] );
                break;
            case "update_grid__highlighted_tile":
                this.computeCursorGridPosition( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ] );
                this.updateHighLightedTile();
                break;
            case "unselect_single_shinken_grid_cell":
                this.selected_grid_cell.getContentByUUID( param ).setPhase( SHINKEN.OBJECT.CONST.PHASE.RUNNING );
                this.selected_grid_cell.removeContent( param, true );
                break;
            case "is_an_action_in_progress_in_grid":
                return this.actionCurrentlyInProgress();
            case "remove_hover_grid_cells":
                param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELLS_OBJECT ].forEach( grid_cell => {
                    grid_cell.setPhase( SHINKEN.OBJECT.CONST.PHASE.EDITING );
                } );
                break;
            case "hover_grid_cells":
                const grid_cells = param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELLS_OBJECT ];
                grid_cells.forEach( grid_cell => {
                    grid_cell.setPhase( SHINKEN.OBJECT.CONST.PHASE.HOVER );
                } );
                break;
            case "unselect_all_other_shinken_grid_cell":
                if ( !this.selected_grid_cell ) {
                    return;
                }
                for ( let i = this.selected_grid_cell.getSize() - 1; i >= 0; i-- ) {
                    if ( param && this.selected_grid_cell.getContent( i ).getUUID() === param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_UUID ] ) {
                        continue;
                    }
                    this.selected_grid_cell.getContent( i ).setPhase( SHINKEN.OBJECT.CONST.PHASE.RUNNING );
                    this.selected_grid_cell.getContent( i ).resetCounterByType( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.SELECTED );
                    this.selected_grid_cell.getContent( i ).resetCounterByType( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.OVERLAPPING );
                    this.selected_grid_cell.removeContentByIndex( i, true );
                }
                break;
            case "mouse_up":
                if ( !this.overlayed_frame ) {
                    return;
                }
                switch ( this.overlayed_frame.getObjectClassName() ) {
                    case "SHINKEN.OBJECT.OverlayFrameMove":
                        this.doActionAfter( "mouse_up_on_grid_cell", { [ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_UUID ]: this.selected_grid_cell.getContent( 0 ).getUUID(), [ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ]: param } );
                        return;
                }
                let element_impacted = this.overlayed_frame.doActionAfter( event_name, this.grid_elements );
                if ( !element_impacted ) {
                    this.resetOverlayFrame();
                    return;
                }
                var last_modification_mouse_up      = new SHINKEN.OBJECT.LastModification();
                const last_modification_extra_param = { "previous_nb_tile_in_height": this.nb_tiles_in_height.getValue() };
                let cells_impacted                  = this.overlayed_frame.doActionAfter( "make_mouse_up_action", element_impacted, this.grid_elements );
                switch ( this.overlayed_frame.getObjectClassName() ) {
                    case "SHINKEN.OBJECT.OverlayFrameMove":
                    case "SHINKEN.OBJECT.OverlayFrameZoneSelection":
                        this.updateSelectedList( element_impacted );
                        this.resetOverlayFrame();
                        return;
                    case "SHINKEN.OBJECT.GridCellResizer":
                        last_modification_mouse_up.setTypeModification( SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.RESIZE );
                        break;
                    case "SHINKEN.OBJECT.SpaceBoxDeletion":
                    case "SHINKEN.OBJECT.SpaceBoxCreation":
                        last_modification_mouse_up.setTypeModification( SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.MOVE );
                        break;
                }
                this.computeOverlappingAndResetOverlay();
                cells_impacted.forEach( current => {
                    last_modification_mouse_up.addSpecific( current.getLastModificationElement( SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.MOVE ) );
                } );
                last_modification_extra_param[ "next_nb_tile_in_height" ] = this.nb_tiles_in_height.getValue();
                last_modification_mouse_up.setExtraParam( last_modification_extra_param );
                this.getController().doActionAfter( "add_last_modification", { [ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ]: last_modification_mouse_up } );
                break;
            case "mouse_down":
                if ( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].target === this.getDomElement() && !param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].ctrlKey ) {
                    this.doActionAfter( "unselect_all_other_shinken_grid_cell" );
                }
                this.doActionAfter( "start_selection_by_zone", param );
                this.overlayed_frame.doActionAfter( event_name );
                break;
            case "open_form_for_widget_configuration":
                param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELLS_OBJECT ].forEach( current_grid_cell => {
                    current_grid_cell.doActionAfter( "open_form_for_widget_configuration" );
                } );
                break;
            case "widgets__have_been_configured__from_form":
            case "change__shinken_element__from_form":
                var last_modification_change                        = new SHINKEN.OBJECT.LastModification( param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION_TYPE ] );
                param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_OBJECT ] = this;
                for ( let i = 0, _size_i = param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_OBJECT_LIST ].length; i < _size_i; i++ ) {
                    last_modification_change.addSpecific( param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_OBJECT_LIST ][ i ].doActionAfter( event_name, param ) );
                }
                this.getController().doActionAfter( "add_last_modification", { [ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ]: last_modification_change } );
                break;
            case "confirm_add_new_widget":
                this.confirmAddNewWidget();
                break;
            case "mouse_move_on_grid_with_add_widget":
                this.scrollWhenAtEdgeOfView( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ] );
                this.moveGridCellToAdd( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ], param[ SHINKEN.OBJECT.RADIOMODE.PARAM.NAME ] );
                break;
            case "mouse_down_on_grid_cell":
                const _dom_resize_icon        = DOM.Service.findParentElementWithClass( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].target, "shinken-resize-cursor", 50 );
                const _dom_delete_widget_icon = DOM.Service.findParentElementWithClass( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].target, "shinken-delete-button", 50 );
                const _dom_config_widget_icon = DOM.Service.findParentElementWithClass( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].target, "shinken-config-button", 50 );
                const _cell_                  = this.grid_elements.doActionAfter( event_name, param );
                if ( !_dom_delete_widget_icon &&
                     this.selected_grid_cell &&
                     !param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].ctrlKey &&
                     !this.selected_grid_cell.containsUUID( _cell_.getUUID() ) ) {
                    this.doActionAfter( "unselect_all_other_shinken_grid_cell", param );
                }
                else if ( _dom_delete_widget_icon ) {
                    _cell_.setPhase( SHINKEN.OBJECT.CONST.PHASE.SELECTED );
                }
                
                this.updateSelectedList( [_cell_] );
                
                if ( _dom_resize_icon ) {
                    this.overlayed_frame = new SHINKEN.OBJECT.GridCellResizer( _dom_resize_icon.dataset.position, _cell_, this.tile_pixel_size );
                    this.overlayed_frame.init( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].clientX, param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].clientY, param[ "dom_relative_position_container_2024_07_18" ] );
                }
                else if ( _dom_delete_widget_icon ) {
                    param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].stopPropagation();
                    this.computeConfirmationPopup( "delete_selected_grid_cell", "delete_selected_grid_cell" );
                }
                else if ( _dom_config_widget_icon ) {
                    param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].stopPropagation();
                    if ( param[ SHINKEN_PAGE.CONST.EVENTS.PARAM.ACTIVE_WINDOW_NAME_IS_JSON ] ) {
                        this.getController().doActionAfter( "focus-on-json-forced", { [ SHINKEN_GRID.CONST.PARAM.EVENT.FOCUSED_WIDGET ]: _cell_ } );
                    }
                    else {
                        _cell_.doActionAfter( "initiate_configuration_of_widget" );
                        this.getController().doActionAfter( "show_form_for_widget_configuration", { [ SHINKEN_GRID.CONST.PARAM.EVENT.FOCUSED_WIDGET ]: _cell_, [ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_OBJECT ]: this } );
                    }
                }
                else {
                    if ( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].ctrlKey ) {
                        return this.doActionAfter( "mouse_down", param );
                    }
                    this.overlayed_frame = new SHINKEN.OBJECT.OverlayFrameMove();
                    this.overlayed_frame.init( param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].clientX, param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].clientY, param[ "dom_relative_position_container_2024_07_18" ] );
                }
                this.getController().doActionAfter( "ask_close_form_for_widget_configuration__if_needed", { [ SHINKEN_GRID.CONST.PARAM.EVENT.FOCUSED_WIDGET ]: _cell_ } );
                if ( this.overlayed_frame ) {
                    this.overlayed_frame.doActionAfter( "mouse_down" );
                }
                if ( _dom_delete_widget_icon || _dom_config_widget_icon ) {
                    return true;
                }
                break;
            case "mouse_up_on_grid_cell":
                if ( !this.overlayed_frame ) {
                    return;
                }
                switch ( this.overlayed_frame.getObjectClassName() ){
                    case "SHINKEN.OBJECT.GridCellResizer":
                        return this.doActionAfter( "mouse_up", param );
                }
                let cell = this.grid_elements.doActionAfter( event_name, param );
                if ( cell ) {
                    this.updateSelectedList( [cell] );
                }
                switch ( this.overlayed_frame.getObjectClassName() ) {
                    case "SHINKEN.OBJECT.OverlayFrameZoneSelection":
                        this.doActionAfter( "mouse_up" );
                        return;
                    case "SHINKEN.OBJECT.OverlayFrameMove":
                        this.overlayed_frame.doActionAfter( "mouse_up" );
                        break;
                }
                this.performNextActionBasedOnOverlap(); //USEFUL IN ORDER TO AVOID RESET IN LOOP
                this.computeOverlappingAndResetOverlay();
                break;
            case "mouse_move_selected_grid_cell":
                if ( !this.overlayed_frame ) {
                    return;
                }
                switch ( this.overlayed_frame.getObjectClassName() ) {
                    case "SHINKEN.OBJECT.GridCellResizer":
                        this.selected_grid_cell.getContent( 0 ).resetSquare();
                        this.computeOverlappingOfSelectedCells();
                        break;
                    case "SHINKEN.OBJECT.OverlayFrameZoneSelection":
                        break;
                    default:
                        this.updateSelectedCellsWithDelta();
                        break;
                }
                break;
            case "drag_n_drop_widget_stop_update_json":
                if ( !this.overlayed_frame ) {
                    return;
                }
                this.selected_grid_cell.getContents().forEach( cell => {
                    cell.doActionAfter( event_name );
                } );
                break;
            case "redo_last_modification":
                switch ( param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ].getTypeModification() ) {
                    case SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.DELETE:
                        param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ].getContents().forEach( modification_element => {
                            let current_element = this.grid_elements.getContentByUUID( modification_element.getModifiedObjectParam( SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_UUID ) );
                            this.deleteGridElements( [current_element], true );
                        } );
                        break;
                    case SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.CREATE:
                        var last_modification_redo = param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ].getContent( 0 );
                        let grid_cell_to_add       = this.grid_elements.addCell( last_modification_redo.getNext(), last_modification_redo.getParam( SHINKEN_GRID.CONST.PARAM.GRID_CELL.INDEX ), true );
                        grid_cell_to_add.setIndexLabelForced( last_modification_redo.getParam( SHINKEN_GRID.CONST.PARAM.GRID_CELL.INDEX_LABEL ) );
                        grid_cell_to_add.content.doActionAfter( "adding_new_widget_manually" );
                        grid_cell_to_add.computeHtml( this, this.dom_parent_with_scroll, true );
                        grid_cell_to_add.setPhase( SHINKEN.OBJECT.CONST.PHASE.EDITING );
                        this.getController().doActionAfter( "grid_cells_added_after__manual_action", { [ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELLS_OBJECT ]: [grid_cell_to_add] } );
                        break;
                    case SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.EDIT:
                    case SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.CONFIGURE:
                        param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_OBJECT ] = this;
                        this.grid_elements.doActionAfter( event_name, param, param_2, param_3 );
                        break;
                    default:
                        const grid_resize_height = this.getResizeHeightWithLastModification( event_name, param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ] );
                        this.applyResizeOfGrid( grid_resize_height );
                        this.grid_elements.doActionAfter( event_name, param, param_2, param_3 );
                        break;
                }
                break;
            case "undo_last_modification":
                switch ( param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ].getTypeModification() ) {
                    case SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.DELETE:
                        let grid_cells_added = [];
                        param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ].getContents().forEach( modification_element => {
                            let current_cell = this.grid_elements.addCell( modification_element.getPrevious(), modification_element.getParam( SHINKEN_GRID.CONST.PARAM.GRID_CELL.INDEX ), true );
                            current_cell.setIndexLabelForced( modification_element.getParam( SHINKEN_GRID.CONST.PARAM.GRID_CELL.INDEX_LABEL ) );
                            current_cell.computeHtml( this, this.dom_parent_with_scroll, true );
                            current_cell.setPhase( SHINKEN.OBJECT.CONST.PHASE.EDITING );
                            grid_cells_added.push( current_cell );
                        } );
                        this.getController().doActionAfter( "grid_cells_added_after__manual_action", { [ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELLS_OBJECT ]: grid_cells_added } );
                        break;
                    case SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.CREATE:
                        const element_uuid  = param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ].getContent( 0 ).getModifiedObjectParam( SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_UUID );
                        let current_element = this.grid_elements.getContentByUUID( element_uuid );
                        this.deleteGridElements( [current_element], true );
                        break;
                    case SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.CONFIGURE:
                    case SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.EDIT:
                        param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_OBJECT ] = this;
                        this.grid_elements.doActionAfter( event_name, param );
                        break;
                    default:
                        const grid_resize_height = this.getResizeHeightWithLastModification( event_name, param[ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ] );
                        this.applyResizeOfGrid( grid_resize_height );
                        this.grid_elements.doActionAfter( event_name, param, param_2, param_3 );
                        break;
                }
                break;
            case "on_key_up":
                param_2 = param_2 || param[ MANAGER.EVENT_MANAGER_V2.PARAM.EVENT ].keyCode;
                switch ( param_2 ) {
                    case SHINKEN.CONST.KEY_CODE.SUPPR:
                        if ( !this.selected_grid_cell.isEmpty() ) {
                            this.computeConfirmationPopup( "delete_selected_grid_cell", "delete_selected_grid_cell" );
                        }
                        break;
                    case SHINKEN.CONST.KEY_CODE.ENTER:
                        this.doActionAfter( "delete_selected_grid_cell" );
                        break;
                    case SHINKEN.CONST.KEY_CODE.ESCAPE:
                        SHINKEN.OBJECT.ConfirmationPopup_V3.getInstance().hide();
                        break;
                }
                break;
            case "update_focus_from_json_action":
                if ( this.last_plusated_element_uuid === param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_UUID ] ) {
                    return;
                }
                this.last_plusated_element_uuid = param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_UUID ];
                const grid_cell                 = this.grid_elements.getContentByUUID( param[ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_UUID ] );
                if ( grid_cell ) {
                    this.doActionAfter( "unselect_all_other_shinken_grid_cell" );
                    grid_cell.setPhase( SHINKEN.OBJECT.CONST.PHASE.SELECTED );
                    this.updateSelectedList( [grid_cell] );
                    grid_cell.askAddClass( "shinken-pulsate", 1600 );
                    DOM.Service.scrollIntoElementIfNotVisible( grid_cell.getDomElement() );
                    
                }
                break;
            case "remove_position_errors_of_cells":
                param.index_of_cells.forEach( index_cell_error => {
                    this.grid_elements.getContent( index_cell_error - 1 ).doActionAfter( event_name );
                } );
                break;
            case "remove_position_errors_of_cells_with_uuid":
                param[ "uuid_list" ].forEach( uuid => {
                    let current_cell = this.grid_elements.getContentByUUID( uuid );
                    if ( current_cell ) {
                        current_cell.doActionAfter( "remove_position_errors_of_cells_with_uuid" );
                        current_cell.resetCounterByType( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.OVERLAPPING );
                    }
                } );
                break;
            case "change_size_of_grid":
                this.applyResizeOfGrid( param[ SHINKEN_GRID.CONST.PARAM.GRID.RESIZE_HEIGHT ] );
                break;
            case "reset_grid_cell_to_add":
                if ( this.grid_cell_to_add ) {
                    this.grid_cell_to_add.remove( true );
                    this.grid_cell_to_add = null;
                }
                break;
            case "pulsate_widgets":
                let widget_uuids = param.split( ',' );
                widget_uuids.forEach( ( uuid ) => {
                    const _grid_cell = this.grid_elements.getContentByUUID( uuid );
                    if ( _grid_cell ) {
                        _grid_cell.askAddClass( "shinken-pulsate", 1600 );
                        DOM.Service.scrollIntoElementIfNotVisible( _grid_cell.getDomElement() );
                    }
                } );
                
                break;
            case "get_default_values_of_grid":
                return this.getDefaultValueOfGrid();
            case "change_virtual_add_grid_cell":
                if ( !this.isPhase( SHINKEN.OBJECT.CONST.PHASE.EDITING ) ) {
                    return;
                }
                this.moveGridCellToAdd( null, param[ SHINKEN.OBJECT.RADIOMODE.PARAM.NAME ] );
                break;
            default:
                this.grid_elements.doActionAfter( event_name, param, param_2, param_3 );
                break;
            
        }
    },
    getResizeHeightWithLastModification: function ( event_name, last_modification ) {
        const modification_extra_param = last_modification.getExtraParam();
        if ( !modification_extra_param ) {
            return 0;
        }
        const value_to_compare = event_name === "undo_last_modification" ? modification_extra_param[ "previous_nb_tile_in_height" ] : modification_extra_param[ "next_nb_tile_in_height" ];
        return value_to_compare - this.nb_tiles_in_height.getValue();
    },
    areSelectedCellsHasOverlapping     : function () {
        this.computeOverlappingOfSelectedCells();
        for ( const cell of this.selected_grid_cell.getContents() ) {
            if ( cell.hasCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.OVERLAPPING ) ) {
                return true;
            }
        }
        return false;
    },
    performNextActionBasedOnOverlap    : function () {
        const last_modification = new SHINKEN.OBJECT.LastModification( SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.MOVE );
        const event_name        = this.areSelectedCellsHasOverlapping() ? "grid_cell_action_after_mouse_up_reset" : "grid_cell_action_after_mouse_up";
        this.selected_grid_cell.doActionAfterContentChildren( event_name, { last_modification: last_modification } );
        this.getController().doActionAfter( "add_last_modification", { [ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ]: last_modification } );
    },
    //********************************************  SELECTED GRID CELL  *****************************************************//
    updateSelectedList  : function ( cell_array ) {
        if ( !cell_array ) {
            return;
        }
        if ( !this.selected_grid_cell ) {
            this.initSelectedGridCell();
        }
        let current;
        for ( let i = 0, _size_i = cell_array.length; i < _size_i; i++ ) {
            current = cell_array[ i ];
            if ( current.isPhase( SHINKEN.OBJECT.CONST.PHASE.SELECTED ) ) {
                this.selected_grid_cell.add( current );
            }
            else {
                this.selected_grid_cell.removeContent( current.getUUID(), true );
            }
        }
    },
    initSelectedGridCell: function () {
        this.selected_grid_cell = new SHINKEN.OBJECT.ShinkenObjectContainer();
        this.selected_grid_cell.initContents();
    },
    //********************************************  GETTER SETTER  *****************************************************//
    getUUID      : function () {
        return this.uuid;
    },
    getController: function () {
        return CONTROLLER.CommonUI;
    },
    //********************************************  COMPUTE  *****************************************************//
    computeMapping                   : function ( mapping ) {
        if ( this.grid_elements instanceof SHINKEN_GRID.Cells ) {
            this.grid_elements.computeMapping( mapping );
        }
    },
    computeOverlappingAndResetOverlay: function () {
        this.computeOverlappingOfSelectedCells();
        this.resetOverlayFrame();
    },
    computeOverlappingOfSelectedCells: function () {
        this.selected_grid_cell.getContents().forEach( cell => {
            this.grid_elements.computeOverlapping( cell );
        } );
    },
    computeSizes                     : function () {
        this.computeTileSize();
        this.computeFontSize();
        this.updateStyle();
        this.grid_elements.computeSizes( this );
    },
    computeTileSize                  : function () {
        this.tile_pixel_size = DOM.Service.getBoundingClientRect( this.getDomElement(), DOM.CONST.STYLE.WIDTH ) / this.getValue( SHINKEN_GRID.CONST.PARAM.GRID.TILES_MAX_X ) * 100 / this.zoom_percent;
    },
    computeContentWithChange         : function () {
        return this.grid_elements.computeContentWithChange();
    },
    computeCursorGridPosition        : function ( event ) {
        let _x             = event.clientX - DOM.Service.getBoundingClientRect( this.dom_element, DOM.CONST.STYLE.LEFT );
        let _y             = event.clientY - DOM.Service.getBoundingClientRect( this.dom_element, DOM.CONST.STYLE.TOP );
        let _on_one_widget = false;
        if ( !event.target.classList.contains( "shinken-grid" ) ) {
            let _dom_parent = DOM.Service.findParentElementWithClass( event.target, "shinken-grid-cell", 50 );
            if ( _dom_parent ) {
                _on_one_widget = true;
            }
        }
        this.mouse_position = {
            x            : Math.floor( _x / this.tile_pixel_size ),
            y            : Math.floor( _y / this.tile_pixel_size ),
            on_one_widget: _on_one_widget,
            is_out_grids : event.target.querySelector( ".shinken-grids" )
        };
    },
    //********************************************  POPUP DELETE GRID CELL  *****************************************************//
    updateComputeConfirmationPopupValidButton: function () {
        this.button_validate_deleted = new COMPONENT.ButtonWithCounter( "valid_popup__20240904", _( "confirmation_popup.delete_button", [DICTIONARY_COMMON_UI] ) );
        SHINKEN.OBJECT.ConfirmationPopup_V3.getInstance().setValidButton( this.button_validate_deleted );
    },
    buildHtmlTitleDeletePopup                : function () {
        return _( "confirmation_popup.deletable_content.grid.title", [DICTIONARY_COMMON_UI] );
    },
    buildHtmlDeleteConfirmationPopup         : function () {
        this.selected_grid_cell.sortContentByClassName();
        let to_return = DOM.Service.createElement( "div", { "class": "shinken-grid-delete-confirmation-popup" } );
        let container = DOM.Service.addElementTo( DOM.Service.createElement( "div", { "class": "shinken-container" } ), to_return );
        let header    = DOM.Service.addElementTo( DOM.Service.createElement( "div", { class: "shinken-layout-center-h-v shinken-layout-justify-between" } ), container );
        DOM.Service.addElementTo( DOM.Service.createElement( "div", { class: "shinken-header shinken-type-cell" }, _( "confirmation_popup.deletable_content.grid.type_column", [DICTIONARY_COMMON_UI] ) ), header );
        DOM.Service.addElementTo( DOM.Service.createElement( "div", { class: "shinken-header shinken-title-cell shinken-layout-expand-1" }, _( "confirmation_popup.deletable_content.grid.title_column", [DICTIONARY_COMMON_UI] ) ), header );
        DOM.Service.addElementTo( DOM.Service.createElement( "div", { class: "shinken-header shinken-action-cell" } ), header );
        let _table_element_to_delete_body = DOM.Service.addElementTo( DOM.Service.createElement( "div", { class: "shinken-container" } ), container );
        for ( let i = 0; i < this.selected_grid_cell.getSize(); i++ ) {
            DOM.Service.addElementTo( this.selected_grid_cell.getContent( i ).computeHtmlDeletePopup(), _table_element_to_delete_body );
            this.selected_grid_cell.getContent( i ).getCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.DELETED ).setParent( this.button_validate_deleted.getCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.TOTAL ) );
        }
        return to_return;
    },
    deleteGridElements                       : function ( list_to_remove, avoid_last_modification ) {
        const last_modification = new SHINKEN.OBJECT.LastModification( SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.DELETE );
        for ( let i = list_to_remove.length - 1; i >= 0; i-- ) {
            let modification_element = list_to_remove[ i ].getLastModificationElement( SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.DELETE );
            modification_element.setParam( SHINKEN_GRID.CONST.PARAM.GRID_CELL.INDEX_LABEL, list_to_remove[ i ].index_label );
            modification_element.setParam( SHINKEN_GRID.CONST.PARAM.GRID_CELL.INDEX, this.grid_elements.getMappingIndex( list_to_remove[ i ].getUUID() ) );
            last_modification.addSpecific( modification_element );
            this.grid_elements.removeContent( list_to_remove[ i ].getUUID() );
        }
        this.getController().doActionAfter( "grid_cells_deleted_by_popup", { [ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELLS_UUID ]: last_modification.getAllUuidOfImpactedElement() } );
        if ( !avoid_last_modification ) {
            this.getController().doActionAfter( "add_last_modification", { [ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ]: last_modification } );
        }
    },
    //********************************************  ADDING CELL  *****************************************************//
    createGridCellToAdd         : function ( type ) {
        this.grid_cell_to_add = this.grid_elements.getChildConstructor();
        this.grid_cell_to_add.setPhase( SHINKEN.OBJECT.CONST.PHASE.ADDING );
        this.grid_cell_to_add.setData( SHINKEN_GRID.CONST.PARAM.GRID_CELL.POSITION_X, { [ SHINKEN.OBJECT.CONST.PARAM.VALUE ]: "" } );
        this.grid_cell_to_add.setData( SHINKEN_GRID.CONST.PARAM.GRID_CELL.POSITION_Y, { [ SHINKEN.OBJECT.CONST.PARAM.VALUE ]: "" } );
        this.grid_cell_to_add.setData( SHINKEN_GRID.CONST.PARAM.GRID_CELL.WIDTH, { [ SHINKEN.OBJECT.CONST.PARAM.VALUE ]: this.getDefaultValueForElementToAdd( type, SHINKEN_GRID.CONST.PARAM.GRID_CELL.WIDTH ) } );
        this.grid_cell_to_add.setData( SHINKEN_GRID.CONST.PARAM.GRID_CELL.HEIGHT, { [ SHINKEN.OBJECT.CONST.PARAM.VALUE ]: this.getDefaultValueForElementToAdd( type, SHINKEN_GRID.CONST.PARAM.GRID_CELL.HEIGHT ) } );
        this.grid_cell_to_add.setData( SHINKEN_GRID.CONST.PARAM.GRID_CELL.CONTENT, this.getContentByTypeForElementToAdd( type ) );
        this.grid_cell_to_add.computeHtml( this, this.dom_parent_with_scroll, true );
        this.grid_cell_to_add.getDomElement().classList.add( "shinken-has-pointer-events-none" );
        DOM.Service.addElementTo( this.grid_cell_to_add.getDomElement(), this.getDomElement() );
        this.grid_cell_to_add.last_modication_data = type;
    },
    moveGridCellToAdd           : function ( event, type ) {
        const mouse_event = event || SHINKEN.TOOLS.MOUSE.getLastMouseEvent();
        if ( !mouse_event ) {
            return;
        }
        this.computeCursorGridPosition( mouse_event );
        if ( this.mouse_position.is_out_grids ) {
            this.doActionAfter( "reset_grid_cell_to_add" );
            return;
        }
        if ( !this.grid_cell_to_add ) {
            this.createGridCellToAdd( type );
        }
        let position_x = Math.round( this.mouse_position.x - this.grid_cell_to_add[ SHINKEN_GRID.CONST.PARAM.GRID_CELL.WIDTH ].getValue() / 2 );
        let position_y = Math.round( this.mouse_position.y - this.grid_cell_to_add[ SHINKEN_GRID.CONST.PARAM.GRID_CELL.HEIGHT ].getValue() / 2 );
        
        position_x = Math.min( Math.max( position_x, 0 ), this.nb_tiles_in_width.getValue() - this.grid_cell_to_add[ SHINKEN_GRID.CONST.PARAM.GRID_CELL.WIDTH ].getValue() );
        position_y = Math.min( Math.max( position_y, 0 ), this.nb_tiles_in_height.getValue() - this.grid_cell_to_add[ SHINKEN_GRID.CONST.PARAM.GRID_CELL.HEIGHT ].getValue() );
        
        this.grid_cell_to_add[ SHINKEN_GRID.CONST.PARAM.GRID_CELL.POSITION_X ].setUserValueAndValue( position_x );
        this.grid_cell_to_add[ SHINKEN_GRID.CONST.PARAM.GRID_CELL.POSITION_Y ].setUserValueAndValue( position_y );
        this.grid_cell_to_add.updateSize();
        this.grid_cell_to_add.computeSquare();
        this.grid_elements.computeOverlapping( this.grid_cell_to_add );
    },
    updateSelectedCellsWithDelta: function () {
        const delta_x = this.overlayed_frame.getDelta_x();
        const delta_y = this.overlayed_frame.getDelta_y();
        // We move all cell before compute the overlapping to avoid flickering on overlapping borders
        for ( let i = 0; i < this.selected_grid_cell.getSize(); i++ ) {
            this.selected_grid_cell.getContent( i ).doActionAfter( "mouse_scroll_selected_grid_cell", delta_x, delta_y );
        }
        for ( let i = 0; i < this.selected_grid_cell.getSize(); i++ ) {
            this.grid_elements.computeOverlapping( this.selected_grid_cell.getContent( i ) );
        }
    },
    confirmAddNewWidget         : function () {
        if ( !this.grid_cell_to_add ) {
            return;
        }
        if ( this.grid_cell_to_add.hasCounter( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.OVERLAPPING ) ) {
            return;
        }
        let params = {
            [ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_OBJECT ]         : this,
            [ SHINKEN_GRID.CONST.PARAM.EVENT.GRID_CELL_OBJECT ]    : this.grid_cell_to_add,
            [ SHINKEN_GRID.CONST.PARAM.EVENT.ADD_DIRECTLY_TO_GRID ]: true
        };
        
        var current_last_modification = new SHINKEN.OBJECT.LastModification( SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.CREATE );
        this.grid_cell_to_add.setPhase( SHINKEN.OBJECT.CONST.PHASE.EDITING );
        let grid_cell_added = this.grid_elements.addCellObject( this.grid_cell_to_add );
        this.grid_cell_to_add.doActionAfter( "adding_new_widget_manually", params );
        let modification_element = grid_cell_added.getLastModificationElement( SHINKEN.OBJECT.CONST.LAST_MODIFICATION.ACTION.CREATE );
        modification_element.setParam( SHINKEN_GRID.CONST.PARAM.GRID_CELL.INDEX_LABEL, grid_cell_added.index_label );
        modification_element.setParam( SHINKEN_GRID.CONST.PARAM.GRID_CELL.INDEX, this.grid_elements.getMappingIndex( grid_cell_added.getUUID() ) );
        current_last_modification.addSpecific( modification_element );
        
        this.getController().doActionAfter( "add_last_modification", { [ SHINKEN.OBJECT.CONST.LAST_MODIFICATION.PARAM.MODIFICATION ]: current_last_modification } );
        this.getController().doActionAfter( "confirm_add_new_widget_done", params );
        this.grid_cell_to_add.getDomElement().classList.remove( "shinken-has-pointer-events-none" );
        this.grid_cell_to_add = null;
    },
    //********************************************  EDITION CELLS OVER      ***********************************//
    initDomElementHighlightCell: function () {
        if ( this.dom_element_highlight_cell ) {
            return;
        }
        this.dom_element_highlight_cell = DOM.Service.createElement( "div", {
            class         : "shinken-grid-highlight-cell-container",
            "shi-tip-type": SHINKEN_TOOLTIP.TYPE.OFF_WHITE_ON_BLACK
        } );
        DOM.Service.addElementTo( this.dom_element_highlight_cell, this.getDomElement() );
    },
    updateHighLightedTile      : function ( force_hidden ) {
        if ( !this.mouse_position ) {
            return;
        }
        if ( this.mouse_position.is_out_grids ) {
            force_hidden = true;
        }
        this.initDomElementHighlightCell();
        if ( force_hidden ) {
            this.dom_element_highlight_cell.classList.add( "shinken-hidden" );
            SHINKEN_TOOLTIP.hideTooltip();
            return;
        }
        const _styles                     = {};
        _styles[ DOM.CONST.STYLE.TOP ]    = this.tile_pixel_size * this.mouse_position.y + DOM.Service.getBoundingClientRect( this.dom_element, DOM.CONST.STYLE.TOP );
        _styles[ DOM.CONST.STYLE.LEFT ]   = this.tile_pixel_size * this.mouse_position.x + DOM.Service.getBoundingClientRect( this.dom_element, DOM.CONST.STYLE.LEFT );
        _styles[ DOM.CONST.STYLE.HEIGHT ] = this.tile_pixel_size;
        _styles[ DOM.CONST.STYLE.WIDTH ]  = this.tile_pixel_size;
        
        
        DOM.Service.setStyles( this.dom_element_highlight_cell, _styles );
        let _tooltip_html = (this.mouse_position.on_one_widget) ? _( 'grid.mode_edition.form_visual.tooltip_on_hover_widget', [DICTIONARY_COMMON_UI] ) + " <br>" : '';
        _tooltip_html += "x = " + this.mouse_position.x + "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; & &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;y = " + this.mouse_position.y;
        
        DOM.Service.addAttribute( this.dom_element_highlight_cell, 'shi-tip-html', _tooltip_html );
        this.dom_element_highlight_cell.classList.remove( 'shinken-hidden' );
        SHINKEN_TOOLTIP.updateTooltip( this.dom_element_highlight_cell );
        
    },
    //********************************************  HTML   **************************************************//
    initGridDomElement             : function () {
        this.setDomElement( DOM.Service.createElement( "div", {
            id                     : "id-shinken-" + this.uuid,
            class                  : this.getClass(),
            style                  : "font-size:" + this.font_size + "px;",
            "data-visibility-state": this.visibility_state || ""
        } ) );
        if ( __has_edit_mode__ ) {
            DOM.Service.addClasses( this.dom_element, "shinken-parent-mouse-move-listener shinken-parent-mouse-leave-listener" );
        }
        this.computeTileSize();
    },
    computeHtml                    : function ( dom_parent_with_scroll ) {
        this.dom_parent_with_scroll = dom_parent_with_scroll;
        let to_return               = !this.getDomElement();
        if ( to_return ) {
            this.initGridDomElement();
            DOM.Service.setDataSet( this.dom_element, "gridUuid", this.uuid );
        }
        this.grid_elements.computeHtml( this, dom_parent_with_scroll );
        this.addCounterTargetDomElement( SHINKEN.OBJECT.COUNTER_V2_CONST.TYPE.SELECTED, this.getDomElement() );
        this.doActionAfter( "compute_html_done" );
        return to_return;
    },
    updateStyle                    : function () {
        DOM.Service.setStyles( this.dom_element, this.getComputedStyle() );
    },
    getComputedStyle               : function () {
        const _nb_tiles                         = this.isPhase( SHINKEN.OBJECT.CONST.PHASE.EDITING ) ? this.getValue( SHINKEN_GRID.CONST.PARAM.GRID.TILES_MAX_Y ) : this.max_nb_tiles;
        const _to_return                        = {};
        _to_return[ DOM.CONST.STYLE.HEIGHT ]    = this.tile_pixel_size * _nb_tiles;
        _to_return[ DOM.CONST.STYLE.FONT_SIZE ] = this.font_size;
        return _to_return;
    },
    computeFontSize                : function () {
        this.font_size = Math.round( DOM.Service.getBoundingClientRect( this.dom_element, DOM.CONST.STYLE.WIDTH ) / 85.7 );
    },
    resetOverlayFrame              : function () {
        if ( !this.overlayed_frame ) {
            return;
        }
        this.overlayed_frame.resetZoneToRemove();
        this.overlayed_frame.remove();
        this.overlayed_frame = null;
    },
    getDefaultValueForElementToAdd : function ( type, key ) {
        switch ( type ) {
            case SHINKEN_PAGE.VISUALFORM.COLLAPSE_FOLDER.WIDGETS.RADIO_MODES.ADD_WIDGET.WEATHER_ADD:
                return this.getDefaultValueFor( WIDGET.CONST.TYPE.WEATHER, key );
            case SHINKEN_PAGE.VISUALFORM.COLLAPSE_FOLDER.WIDGETS.RADIO_MODES.ADD_WIDGET.SEPARATOR_ADD:
                return this.getDefaultValueFor( WIDGET.CONST.TYPE.SEPARATOR, key );
        }
    },
    getDefaultValueFor             : function ( type_widget, key ) {
        let dict      = this.layouts[ type_widget + "_layout" ].getDataForJson( SHINKEN.OBJECT.JSON_EDITOR.FORMAT_DATA.VALUE_AND_DEFAULT );
        let to_return = SHINKEN.TOOLS.DICT.getValueWithComposedKey( dict, key ).value;
        if ( key === SHINKEN_GRID.CONST.PARAM.GRID_CELL.WIDTH ) {
            to_return = this.getComputedWidth( to_return );
        }
        return to_return;
    },
    getContentByTypeForElementToAdd: function ( type ) {
        let to_return;
        switch ( type ) {
            case SHINKEN_PAGE.VISUALFORM.COLLAPSE_FOLDER.WIDGETS.RADIO_MODES.ADD_WIDGET.WEATHER_ADD:
                to_return = {
                    [ WIDGET.CONST.PARAM.TYPE._KEY ]  : WIDGET.CONST.TYPE.WEATHER,
                    [ WIDGET.CONST.PARAM.ITEM._KEY ]  : {
                        [ WIDGET.CONST.PARAM.ITEM.TYPE._KEY ]: "",
                        [ WIDGET.CONST.PARAM.ITEM.UUID._KEY ]: "",
                        [ WIDGET.CONST.PARAM.ITEM.NAME._KEY ]: ""
                    },
                    [ WIDGET.CONST.PARAM.LAYOUT._KEY ]: this[ SHINKEN_GRID.CONST.PARAM.GRID.LAYOUTS ][ WEATHER.Grid.CONST.PARAM.DEFAULT_CONFIGURATION.WEATHER_WIDGET_LAYOUT._KEY ].content[ WIDGET.CONST.PARAM.LAYOUT._KEY ].getDataForJson( SHINKEN.OBJECT.JSON_EDITOR.FORMAT_DATA.VALUE_AND_FORCE_DEFAULT )
                };
                break;
            case SHINKEN_PAGE.VISUALFORM.COLLAPSE_FOLDER.WIDGETS.RADIO_MODES.ADD_WIDGET.SEPARATOR_ADD:
                to_return                                                                                                                                                                                                                                          = {
                    [ WIDGET.CONST.PARAM.TYPE._KEY ]  : WIDGET.CONST.TYPE.SEPARATOR,
                    [ WIDGET.CONST.PARAM.LAYOUT._KEY ]: this[ SHINKEN_GRID.CONST.PARAM.GRID.LAYOUTS ][ WEATHER.Grid.CONST.PARAM.DEFAULT_CONFIGURATION.SEPARATOR_WIDGET_LAYOUT._KEY ].content[ WIDGET.CONST.PARAM.LAYOUT._KEY ].getDataForJson( SHINKEN.OBJECT.JSON_EDITOR.FORMAT_DATA.VALUE_AND_FORCE_DEFAULT )
                };
                to_return[ WIDGET.CONST.PARAM.LAYOUT._KEY ][ WIDGET.CONST.PARAM.LAYOUT.INFORMATION_TO_DISPLAY._KEY ][ WIDGET.CONST.PARAM.LAYOUT.INFORMATION_TO_DISPLAY.TITLE_AREA._KEY ][ WIDGET.CONST.PARAM.LAYOUT.INFORMATION_TO_DISPLAY.TITLE_AREA.LABEL._KEY ] = _( "edit_mode.widget.new_widget_separator" );
                break;
        }
        return SHINKEN.OBJECT.DefaultConfigurationWithOwnPropertyGridPage.prototype.parseDictToDict_withValueKey( to_return );
    },
    getDefaultValueOfGrid          : function () {
    },
    //********************************************  UPDATE DATA   **************************************************//
    setData                  : function ( key, value ) {
        switch ( key ) {
            case SHINKEN_GRID.CONST.PARAM.GRID.TILES_MAX_X:
            case SHINKEN_GRID.CONST.PARAM.GRID.TILES_MAX_Y:
            case SHINKEN_GRID.CONST.PARAM.GRID.SEPARATION_GRID_ELEMENT_MIN:
            case SHINKEN_GRID.CONST.PARAM.GRID.SEPARATION_GRID_ELEMENT_MAX:
            case SHINKEN_GRID.CONST.PARAM.GRID.SEPARATION_GRID_ELEMENT_PERCENT:
            case SHINKEN_GRID.CONST.PARAM.GRID.TYPE:
                this.setDataKey( key, value );
                break;
            case SHINKEN_GRID.CONST.PARAM.GRID.GRID_ELEMENTS:
                this[ key ] = this.getChildConstructor( value );
                this[ key ].setCountersParent( this );
                this.addSynchronizedPhase( this[ key ] );
                this.own_children_keys.push( key );
                break;
            case SHINKEN_GRID.CONST.PARAM.GRID.LAYOUTS:
                this[ key ] = this.getLayoutConstructor( value );
                this.own_children_keys.push( key );
                break;
            default:
                this.setDataAsUnknownProperty( key, value );
                break;
        }
    },
    updateContentData        : function ( data ) {
        this.grid_elements.updateContentData( data );
    },
    getChildConstructor      : function ( data ) {
        return new SHINKEN_GRID.Cells( data );
    },
    getLayoutConstructor     : function ( data, uuid ) {
    },
    actionCurrentlyInProgress: function () {
        return !!this.overlayed_frame || SHINKEN.OBJECT.ConfirmationPopup_V3.getInstance().isVisible();
    },
    //********************************************  PHASE  *****************************************************//
    callbackForPhase: function () {
        if ( this.getCurrentPhase() !== this.getPreviousPhase() ) {
            this.updateStyle();
        }
        if ( this.getCurrentPhase() === SHINKEN.OBJECT.CONST.PHASE.EDITING ) {
            this.initSelectedGridCell();
        }
    },
    //********************************************  SCROLLING  **************************************************//
    scrollWhenAtEdgeOfView: function ( mouse_event ) {
        let dom_scrollable_container      = DOM.Service.findParentElementWithClass( this.getDomElement(), "shinken-grid-container", 50 );
        const mouse_position_y_in_grid    = this.parseRelativePixelY( mouse_event, dom_scrollable_container );
        const max_tile_in_height_in_pixel = this.parseGridToPixel_y( this.nb_tiles_in_height.getValue() );
        const position_y_visible          = this.getPositionYVisible( dom_scrollable_container );
        let scroll_height                 = 0;
        const delta                       = 20;
        
        if ( position_y_visible.max - mouse_position_y_in_grid < delta && mouse_position_y_in_grid + delta < max_tile_in_height_in_pixel ) {
            scroll_height = delta;
        }
        else if ( mouse_position_y_in_grid - position_y_visible.min < delta && mouse_position_y_in_grid - delta > 0 ) {
            scroll_height = -delta;
        }
        if ( scroll_height !== 0 ) {
            DOM.ScrollService.scrollVertically( dom_scrollable_container, scroll_height );
        }
    },
    getCurrentScrollTop   : function () {
        let scroll_element = DOM.Service.findParentElementWithClass( this.getDomElement(), "shinken-grid-container", 50 );
        return this.getPositionYVisible( scroll_element ).min;
    },
    setCurrentScrollTop   : function ( scroll_top ) {
        let scroll_element = DOM.Service.findParentElementWithClass( this.getDomElement(), "shinken-grid-container", 50 );
        DOM.ScrollService.scrollToTop( scroll_element, scroll_top );
    },
    parseGridToPixel_y    : function ( pos_grid ) {
        return pos_grid * this.tile_pixel_size;
    },
    parseRelativePixelY   : function ( mouse_event, dom_scrollable_container ) {
        return mouse_event.clientY - dom_scrollable_container.getBoundingClientRect().top + dom_scrollable_container.scrollTop;
    },
    getPositionYVisible   : function ( dom_scrollable_container ) {
        return {
            max: dom_scrollable_container.scrollTop + dom_scrollable_container.getBoundingClientRect().height,
            min: dom_scrollable_container.scrollTop
        };
    },
    applyResizeOfGrid     : function ( resize_height ) {
        if ( resize_height && resize_height !== 0 ) {
            const new_grid_height = Math.max( this.nb_tiles_in_height.getCfgValue(), this.nb_tiles_in_height.getValue() + resize_height );
            this.nb_tiles_in_height.setUserValueAndValue( new_grid_height );
            this.getController().doActionAfter( "change_grid_size_in_json", { new_grid_height_value: new_grid_height } );
            this.grid_elements.doActionAfter( "update_nb_tiles_in_height", { new_nb_tiles_in_height: new_grid_height } );
            this.updateStyle();
        }
    }
};
SHINKEN.TOOLS.CLASS.addPrototype( SHINKEN_GRID.Grid, SHINKEN.OBJECT.DefaultConfigurationWithOwnProperty );
SHINKEN.TOOLS.CLASS.addPrototype( SHINKEN_GRID.Grid, SHINKEN.OBJECT.PhaseInterface );
SHINKEN.TOOLS.CLASS.addPrototype( SHINKEN_GRID.Grid, SHINKEN.OBJECT.ConfirmationPopupInterface );
