# !/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.exceptions.business import ShinkenExceptionKeyError
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.component.abstract_component import AbstractComponent
from shinkensolutions.filters.list.items.list_filter_item_paginator import ListFilterItemPaginator
from shinkensolutions.filters.list.items.list_filter_item_sort import ListFilterItemSort
from shinkensolutions.filters.list.list_filter import ListFilter

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Type, List
    from shinkensolutions.filters.list.items.abstract_list_filter_item import AbstractListFilterItem


class LIST_FILTER_CONSTS:
    PAGINATION = 'page_settings'  # type: str
    SORT = 'sort'  # type: str


class FilterFactoryComponent(AbstractComponent):
    def __init__(self):
        self._default_ordered_list_filters = [LIST_FILTER_CONSTS.SORT, LIST_FILTER_CONSTS.PAGINATION]
        self._list_filter_item_types = {}  # type: Dict[str, Type[AbstractListFilterItem]]
    
    
    def init(self):
        # type: () -> None
        self._init_list_filter_item_types()
        for filter_type_name in self.get_ordered_list_filters():
            if filter_type_name not in self._list_filter_item_types:  # pragma: no cover
                raise ShinkenExceptionKeyError(text='Unknown list filter item type [ %s ]' % filter_type_name)
    
    
    def _init_list_filter_item_types(self):
        # type: () -> None
        self._list_filter_item_types.update({
            LIST_FILTER_CONSTS.PAGINATION: ListFilterItemPaginator,
            LIST_FILTER_CONSTS.SORT      : ListFilterItemSort
        })
    
    
    def make_list_filter(self, filter_dict):
        # type: (Dict[str, Dict[str, str]]) -> ListFilter
        
        list_filter = ListFilter()
        for filter_type_name in self.get_ordered_list_filters():
            filter_item_dict = filter_dict.get(filter_type_name, None)
            if filter_item_dict is not None:
                filter_item_type = self._list_filter_item_types[filter_type_name]
                filter_item = filter_item_type.from_dict(filter_item_dict)
                list_filter.add_filter_item_to_apply(filter_item)
        return list_filter
    
    
    def get_ordered_list_filters(self):
        # type: () -> List[str]
        # Must return all the filter type names to apply by default in make_list_filter()
        # The filters will be applied according to the list order
        return self._default_ordered_list_filters
