#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.
import functools
import logging
import time
from collections import namedtuple

from shinken.misc.os_utils import ShinkenAccessRightsError
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.pickledb import ShinkenPickleableMeta
from shinkensolutions.context_helper import nullcontext
from shinkensolutions.data_hub.data_hub_exception.data_hub_exception import DataHubNotReady, DataHubFatalException, DataHubItemNotFound, DataHubCorruptedData

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Tuple, Any, Dict, WrappedFunction, ContextManager
    from shinken.log import PartLogger
    from shinkensolutions.data_hub.data_hub_driver.abstract_data_hub_driver import AbstractDataHubDriver, AbstractDataHubDriverConfig


def handle_data_hub_fatal_error(func):
    # type: (WrappedFunction) -> WrappedFunction
    
    @functools.wraps(func)
    def wrapper(self, *args, **kwargs):
        # type: ('DataHub', Any, Any) -> Any
        try:
            return func(self, *args, **kwargs)
        except DataHubFatalException as err:
            self._logger_base_hub.error('The Data Hub stopped because of this fatal error : %s' % err)
            raise
    
    
    return wrapper


DataHubItem = namedtuple('DataHubItem', ['item_data', 'last_modification'])


class DataHubConfig(object, metaclass=ShinkenPickleableMeta):
    def __init__(self, data_hub_id, data_hub_category, data_type, data_id_key_name, driver_config, must_save_configuration=False):
        # type: (str, str, str, str, AbstractDataHubDriverConfig, bool) -> None
        self.data_hub_id = data_hub_id
        self.data_hub_category = data_hub_category  # This will be used to retrieve datahub from config file. Example : it is used by sanitize data to sanitize specifics datahub. Don't duplicate data_hub_category unless content must be treat equally
        self.data_type = data_type
        self.data_id_key_name = data_id_key_name
        self.driver_config = driver_config
        self.must_save_configuration = must_save_configuration
    
    
    def __getstate__(self):
        # type: () -> Dict[str, Any]
        state = self.__dict__.copy()
        return state
    
    
    def __setstate__(self, state):
        # type: (Dict[str, Any]) -> None
        
        # noinspection PyTypeChecker
        DataHubConfig.__init__(self, '', '', '', '', None)
        for prop, value in state.items():
            setattr(self, prop, value)


class DataHub:
    def __init__(self, logger, data_hub_id, data_type, data_id_key_name, data_hub_driver):
        # type: (PartLogger, str, str, str, AbstractDataHubDriver) -> None
        self._data_hub_id = data_hub_id
        self._data_type = data_type
        self._data_hub_driver = data_hub_driver
        self._data_id_key_name = data_id_key_name
        
        self._is_initialized = False
        
        self._logger = logger
        self._logger_base_hub = self._logger.get_sub_part('DATA HUB : %s' % data_hub_id)
        self._logger_hub_init = self._logger_base_hub.get_sub_part('INIT')
        self._logger_hub_save = self._logger_base_hub.get_sub_part('SAVE')
        self._logger_hub_load = self._logger_base_hub.get_sub_part('GET')
        self._logger_hub_retention = self._logger_base_hub.get_sub_part('SAVE RETENTION')
        self._logger_hub_remove = self._logger_base_hub.get_sub_part('REMOVE')
        self._logger_hub_last_mod = self._logger_base_hub.get_sub_part('GET LAST MODIF')
        self._logger_backup = self._logger_base_hub.get_sub_part('BACKUP')
        self._logger_restore = self._logger_base_hub.get_sub_part('RESTORE')
    
    
    def init(self):
        if not self._is_initialized:
            self._data_hub_driver.init()
            self._verify_right_access_on_all_data()
            self._is_initialized = True
    
    
    def get_id(self):
        return self._data_hub_id
    
    
    def _verify_right_access_on_all_data(self):
        self._data_hub_driver.check_right_access_issues_then_fix_them()
    
    
    def is_initialized(self):
        return self._is_initialized
    
    
    def check_data_in_hub(self, all_data_id: 'list[str]', *, log_level: int = logging.INFO) -> 'tuple[list[str], list[str]]':
        self._logger_hub_init.info('Starting check %s %s' % (len(all_data_id), self._data_type))
        
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        start = time.time()
        
        valid_data_id = []
        invalid_data_id = []
        for data_id in all_data_id:
            is_data_correct = self._data_hub_driver.is_data_correct(data_id)
            if is_data_correct:
                self._logger_hub_init.log(log_level, '%s with %s [ %s ] successfully check' % (self._data_type, self._data_id_key_name, data_id))
                valid_data_id.append(data_id)
            else:
                self._logger_hub_init.log(log_level, '%s with %s [ %s ] is invalid' % (self._data_type, self._data_id_key_name, data_id))
                invalid_data_id.append(data_id)
        
        elapsed = time.time() - start
        self._logger_hub_init.info('Check data in hub ended in [ %0.3f ] seconds' % elapsed)
        return valid_data_id, invalid_data_id
    
    
    def lock_context(self, data_id):
        # type: (str) -> ContextManager[Any]
        if not self.is_initialized():
            self._logger_hub_load.error('Cannot try to lock the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.lock_context(data_id)
        except NotImplementedError:
            return nullcontext()
    
    
    @handle_data_hub_fatal_error
    def get_data(self, data_id):
        # type: (str) -> Any
        if not self.is_initialized():
            self._logger_hub_load.error('Cannot get the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            data = self._data_hub_driver.read(data_id, log_error=True)
        except DataHubItemNotFound:
            self._logger_hub_load.info('The %s with %s [ %s ] was not found in the Data Hub' % (self._data_type, self._data_id_key_name, data_id))
            raise
        except ShinkenAccessRightsError as err:
            raise DataHubFatalException(err)
        except ValueError as err:
            self._logger_hub_load.error('The %s with %s [ %s ] could not be retrieved because its data is corrupted' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubCorruptedData(str(err))
        
        return data
    
    
    @handle_data_hub_fatal_error
    def get_data_and_last_modification(self, data_id):
        # type: (str) -> DataHubItem
        if not self.is_initialized():
            self._logger_hub_load.error('Cannot get the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            data, last_modification = self._data_hub_driver.read_and_get_last_modification_date(data_id, log_error=True)
        except DataHubItemNotFound:
            self._logger_hub_load.info('The %s with %s [ %s ] was not found in the Data Hub' % (self._data_type, self._data_id_key_name, data_id))
            raise
        except ShinkenAccessRightsError as err:
            raise DataHubFatalException(err)
        except ValueError as err:
            self._logger_hub_load.error('The %s with %s [ %s ] could not be retrieved because its data is corrupted' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubCorruptedData(str(err))
        
        return DataHubItem(data, last_modification)
    
    
    @handle_data_hub_fatal_error
    def get_last_modification_date(self, data_id):
        # type: (str) -> int
        if not self.is_initialized():
            self._logger_hub_last_mod.error('Cannot get the last modification time of %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            last_modification_date = self._data_hub_driver.get_last_modification_date(data_id)
        except DataHubItemNotFound:
            self._logger_hub_last_mod.info('The %s with %s [ %s ] was not found in the Data Hub' % (self._data_type, self._data_id_key_name, data_id))
            raise
        except ShinkenAccessRightsError as err:
            raise DataHubFatalException(err)
        except ValueError as err:
            self._logger_hub_last_mod.error('The %s with %s [ %s ] could not be retrieved because its data is corrupted' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubCorruptedData(str(err))
        
        return last_modification_date
    
    
    @handle_data_hub_fatal_error
    def save_data(self, data_id, data):
        # type: (str, Any) -> None
        if not self.is_initialized():
            self._logger_hub_save.error('Cannot save the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.write(data_id, data)
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def remove_data(self, data_id):
        # type: (str) -> None
        if not self.is_initialized():
            self._logger_hub_remove.error('Cannot remove the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.remove(data_id)
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def remove_data_and_lock(self, data_id):
        # type: (str) -> None
        if not self.is_initialized():
            self._logger_hub_remove.error('Cannot remove the %s with %s [ %s ] because the Data Hub is not initialized, please retry later' % (self._data_type, self._data_id_key_name, data_id))
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.hard_remove(data_id)
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def get_number_of_stored_data(self):
        # type: () -> int
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_number_of_stored_data()
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def get_all_data_id(self):
        # type: () -> List[str]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_all_data_id()
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def get_all_data_with_id(self):
        # type: () -> List[Tuple[str, Any]]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_all_data_with_id()
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def get_all_data(self):
        # type: () -> List[Any]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_all_data()
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def get_data_from_id_list(self, data_id_list):
        # type: (List[str]) -> List[Any]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.read_all(data_id_list)
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def get_data_with_id_from_id_list(self, data_id_list):
        # type: (List[str]) -> List[Tuple[str, Any]]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.read_all_with_id(data_id_list)
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def find_data_id(self, filters):
        # type: (Any) -> List[str]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.find_data_id(filters)
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def find_data(self, filters):
        # type: (Any) -> List[Any]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.find_data(filters)
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def find_data_with_id(self, filters):
        # type: (Any) -> List[Tuple[str, Any]]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.find_data_with_id(filters)
        except Exception as e:
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def destroy(self):
        # type: () -> None
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.destroy()
        except Exception as e:
            raise DataHubFatalException(str(e) if str(e) else getattr(e, 'strerror', e.__class__.__name__).decode('utf8', 'ignore'))
    
    
    @handle_data_hub_fatal_error
    def stop(self):
        # type: () -> None
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            self._data_hub_driver.stop()
        except Exception as e:
            raise DataHubFatalException(str(e) if str(e) else getattr(e, 'strerror', e.__class__.__name__).decode('utf8', 'ignore'))
    
    
    @handle_data_hub_fatal_error
    def get_total_size(self):
        # type: () -> int
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_total_size()
        except Exception as e:
            self._logger_hub_init.error(e)
            raise DataHubFatalException(str(e))
    
    
    @handle_data_hub_fatal_error
    def get_size_of(self, data_id):
        # type: (str) -> int
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        try:
            return self._data_hub_driver.get_size_of(data_id)
        except Exception as e:
            self._logger_hub_init.error(e)
            raise DataHubFatalException(str(e))
    
    
    def backup(self):
        # type: () -> Dict[str, Any]
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        backup = {}
        for data_id in self._data_hub_driver.get_all_data_id():
            data = self._data_hub_driver.read_raw(data_id)
            backup[data_id] = data
        return backup
    
    
    def restore(self, backup_data):
        # type: (Dict[str, Any]) -> None
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        for item_id, item_data in backup_data.items():
            self._data_hub_driver.restore(item_id, item_data)
    
    
    def sanitize_data(self, item_id, item_data):
        # type: (str, Any) -> None
        if not self.is_initialized():
            self._logger_hub_init.error('The data hub is not initialized')
            raise DataHubNotReady()
        
        self._data_hub_driver.sanitize(item_id, item_data)
