#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import atexit
import hashlib
import logging
import multiprocessing
import os
import random
import re
import shlex
import socket
import subprocess
import sys
import time

import threading
import pymongo
import six
from pymongo import MongoClient, MongoReplicaSetClient
from pymongo.uri_parser import parse_uri

from shinken.compat import bytes_to_unicode
from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.url_helper import BaseUrl
from shinkensolutions.ssh_mongodb.mongo_error import MongoConnectionError
from .database_statistics_thread import DatabaseStatistics
from ..data_hub.data_hub import DataHubConfig
from ..data_hub.data_hub_driver.abstract_data_hub_driver_file import DATA_HUB_FILE_DEFAULT_DIRECTORY
from ..data_hub.data_hub_driver.data_hub_driver_json import DataHubDriverConfigJson
from ..data_hub.data_hub_exception.data_hub_exception import DataHubItemNotFound
from ..data_hub.data_hub_factory.data_hub_factory import DataHubFactory
from ..data_hub.data_hub_meta_driver.data_hub_meta_driver_cleanup_directory import DataHubMetaDriverConfigCleanupDirectory

try:
    # noinspection SpellCheckingInspection
    import fcntl
except ImportError:
    # noinspection SpellCheckingInspection
    fcntl = None

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Union, List, Optional
    from shinkensolutions.data_hub.data_hub import DataHub
    from shinken.log import PartLogger

SSH_ADDRESS_ALREADY_IN_USE_ERROR = 'address_already_in_use'
SSH_TUNNEL_TIMEOUT_DEFAULT = 5


class ConnectionResult:
    id = 0
    
    
    def __init__(self, con: 'MongoClient | MongoReplicaSetClient', ssh_time: 'float', mongodb_time: 'float', local_port: 'int|None', use_ssh: 'bool', requester: 'str', connect_args: 'tuple|None' = None):
        cls = self.__class__
        cls.id += 1
        self.id = cls.id
        self._con = con
        self._ssh_time = ssh_time
        self._mongodb_time = mongodb_time
        self.local_port = local_port
        self.use_ssh = use_ssh
        self.requesters = {requester}
        self.connect_args = connect_args
        self.lock = threading.RLock()
        self.tunnel_keep_in_use_socket: 'socket.socket|None' = None
        
        if use_ssh:
            self.tunnel_keep_in_use_enable()
    
    
    def acquire(self):
        return self.lock.acquire()
    
    
    def release(self):
        self.lock.release()
    
    
    def __enter__(self):
        return self.acquire()
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.release()
    
    
    def add_requester(self, requester):
        self.requesters.add(requester)
    
    
    def get_connection(self) -> 'Union[MongoClient, MongoReplicaSetClient]':
        with self.lock:
            return self._con
    
    
    def set_connection(self, connection: 'MongoClient | MongoReplicaSetClient'):
        with self.lock:
            if self._con is not None:
                raise RuntimeError(f'Replacing unclosed connection[{self.id}] from requester {self.requesters}')
            self._con = connection
    
    
    def disconnect(self):
        with self.lock:
            if self._con:
                self._con.close()
                self._con = None
    
    
    def is_disconnected(self) -> bool:
        with self.lock:
            return self._con is None
    
    
    def get_ssh_tunnel_time(self):
        return self._ssh_time
    
    
    def get_mongodb_connection_time(self):
        return self._mongodb_time
    
    
    def network_is_reachable(self):
        if not self.use_ssh:
            return True
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.settimeout(5)
        try:
            sock.connect(('localhost', self.local_port))
            return True
        except Exception:
            return False
        finally:
            try:
                sock.close()
            except:
                pass
    
    
    def tunnel_keep_in_use_enable(self):
        if not self.use_ssh or (self.tunnel_keep_in_use_socket and self.tunnel_keep_in_use_socket.fileno() >= 0):
            return
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.tunnel_keep_in_use_socket = sock
        sock.settimeout(5)
        try:
            sock.connect(('localhost', self.local_port))
            return True
        except Exception:
            pass
    
    
    def tunnel_keep_in_use_disable(self):
        if not self.use_ssh or not self.tunnel_keep_in_use_socket:
            return
        try:
            if self.tunnel_keep_in_use_socket.fileno() >= 0:
                self.tunnel_keep_in_use_socket.close()
        except:
            pass
        finally:
            self.tunnel_keep_in_use_socket = None


class SshTunnelMongoManager:
    def __init__(self):
        self.connection_results: 'dict[str, ConnectionResult]' = {}
        self.mute_mode = False
        
        # We will remember all living ssh tunnels, so we can close them when need
        self._living_sub_processes: 'dict[str,subprocess.Popen]' = {}
        
        self._database_statistics_threads: 'Dict[str, DatabaseStatistics]' = {}
        
        self._need_data_hub: bool = True
        self._in_check = False
        self._data_hub: 'Optional[DataHub]' = None
        self._data_hub_creation_lock = multiprocessing.Lock()
        self._fork_logger: 'PartLogger|None' = None
        self._fork_lock = threading.RLock()
    
    
    def _init_data_hub(self, _logger: 'PartLogger') -> None:
        with self._data_hub_creation_lock:
            driver_config = DataHubDriverConfigJson(
                base_directory=DATA_HUB_FILE_DEFAULT_DIRECTORY.DEV_SHM,
                data_location_name='mongodb_database_statistics'
            )
            
            meta_driver_config = DataHubMetaDriverConfigCleanupDirectory(
                config_main_driver=driver_config,
                cleanup_interval=120
            )
            data_hub_config = DataHubConfig(
                data_hub_id='dev_shm_mongodb_database_statistics',
                data_hub_category='mongodb_database_statistics',
                data_type='DatabaseStatistics',
                data_id_key_name='name',
                driver_config=meta_driver_config
            )
            
            self._data_hub = DataHubFactory.build_and_init_data_hub(_logger.get_sub_part('DatabaseStatistics'), data_hub_config)
    
    
    def _create_database_statistics_threads(self, uri: str, connection_result: 'ConnectionResult', requester: str) -> None:
        if uri in self._database_statistics_threads and self._database_statistics_threads[uri].is_running():
            return
        self._database_statistics_threads[uri] = DatabaseStatistics(uri, connection_result, self._data_hub, 1, requester, get_statistics_interval=60)
        self._database_statistics_threads[uri].start_thread()
    
    
    def _update_connection_results(self, mongo_uri: str, connection_result: 'ConnectionResult', requester: str) -> None:
        self.connection_results[mongo_uri] = connection_result
        if self._need_data_hub:
            address = self._get_computed_address_from_mongo_uri(mongo_uri)
            
            self._create_database_statistics_threads(address if address else mongo_uri, connection_result, requester)
    
    
    def _clear_database_statistics_threads(self) -> None:
        for thread in self._database_statistics_threads.values():
            thread.stop()
        self._database_statistics_threads.clear()
    
    
    def set_fork_logger(self, logger: 'PartLogger'):
        self._fork_logger = logger.get_sub_part('MONGO FORK_CLEANUP')
    
    
    def reset_fork_logger(self):
        self._fork_logger = None
    
    
    # Before fork, we clear all our connection but not our ssh tunel
    def before_fork_cleanup(self) -> None:
        self._fork_lock.acquire()
        if not self.connection_results:
            return
        
        if self._fork_logger:
            self._fork_logger.info(f'Closing {len(self.connection_results)} mongo connections before fork. (will be re-opened automatically on usage)')
        self._clear_database_statistics_threads()
        
        for uri, connection_result in self.connection_results.items():
            try:
                connection_result.lock.acquire()
                # Close mongo connection (and related threads)
                connection_result.disconnect()
                
                if self._fork_logger and self._fork_logger.is_debug():
                    self._fork_logger.debug(f'Closed connection id:{connection_result.id} requested by {connection_result.requesters} to {uri} before fork')
            except Exception as e:
                if self._fork_logger:
                    self._fork_logger.warning(f'Failed to close connection id:{connection_result.id} before fork to {uri} with error {str(e)}')
                    self._fork_logger.print_stack(level=logging.WARNING)
    
    
    # In child after fork we clean up connections
    def after_fork_cleanup_in_child(self) -> None:
        self._living_sub_processes.clear()
        for uri, connection_result in self.connection_results.items():
            # Fork setup has closed all inet socket from parent, let's do some clean up
            connection_result.tunnel_keep_in_use_disable()
            connection_result.lock.release()
            if self._fork_logger and self._fork_logger.is_debug():
                self._fork_logger.debug(f'Released lock of connection id:{connection_result.id} requested by {connection_result.requesters} to {uri} after fork in child')
        self._fork_lock.release()
    
    
    def after_fork_cleanup_in_parent(self) -> None:
        for uri, connection_result in self.connection_results.items():
            try:
                # Restore connection to mongo
                connection_result.set_connection(self._do_connect(*connection_result.connect_args))
                self._update_connection_results(uri, connection_result, f'{connection_result.requesters}')
                if self._fork_logger and self._fork_logger.is_debug():
                    self._fork_logger.debug(f'Restored connection id:{connection_result.id} requested by {connection_result.requesters} to {uri} after fork in parent')
            except Exception as e:
                if self._fork_logger:
                    self._fork_logger.debug(f'Failed to restore connection id:{connection_result.id} requested by {connection_result.requesters} to {uri} after fork in parent with error {str(e)}')
                    self._fork_logger.print_stack(level=logging.WARNING)
            finally:
                # In all cases, lock must be released to allow auto reconnect to recreate connection if needed
                connection_result.lock.release()
        self._fork_lock.release()
    
    
    # This lib can be used in checks, and here we do nto want any logger or such things, Check will do it itself
    def set_mute(self) -> None:
        self.mute_mode = True
    
    
    def set_in_check(self) -> None:
        self._in_check = True
    
    
    def disable_data_hub(self) -> None:
        self._need_data_hub = False
    
    
    def set_ssh_tunnel_manager_for_check(self) -> None:
        self.disable_data_hub()
        self.set_in_check()
        self.set_mute()
    
    
    # For a specific uri we will have one id for the destination, and we will have 1 and only one tunnel
    # by destination
    def _get_destination_from_uri(self, uri, logger=None):
        # Parsing example :
        # parse_uri("mongodb://user:@example.com/my_database/?w=2")
        # {'username': 'user', 'nodelist': [('example.com', 27017)], 'database': 'my_database/', 'collection': None, 'password': '', 'options': {'w': 2}}
        if pymongo.version_tuple[0] >= 3:
            # noinspection PyArgumentList
            uri_datas = parse_uri(uri, warn=True)
        else:
            uri_datas = parse_uri(uri)
        
        # TODO: what to do with more than one node? how to manage this?
        nodes = uri_datas['nodelist']
        if len(nodes) != 1:
            if not self.mute_mode and logger:
                logger.error('Cannot extract destination server from the mongodb uri "%s". There must be one destination server to connect to.' % uri)
            return None
        node = nodes[0]
        addr, dest_port = node
        return addr, dest_port
    
    
    # We want a port that is 30000 < HASH PORT < 60000 based on the addr string
    @staticmethod
    def _get_tunnel_port(addr: str) -> int:
        h = hash(addr)
        if h < 0:
            h += sys.maxsize
        # h is here a big int, map it into the 10000 => 30000 space
        port = 10000 + divmod(h, 20000)[1]
        return port
    
    
    # We want a binding port, randomly from base port.
    # _get_new_binding_port_from_base_port port is 10K=>30K, so we can randomize in the next 30K to have
    # a random between 30K->60K
    @staticmethod
    def _get_new_binding_port_from_base_port(base_port: int) -> int:
        new_binding_port = base_port + random.randint(1, 30000)
        return new_binding_port
    
    
    # We will spawn an ssh background process.
    # It will wait for a connection in the next 30s
    # * If there i no connection in the 30s, it will just stop
    # * when the last connection will close, it will stop too
    def _spawn_background_ssh(self, uri, addr, dest_port, local_port_search, user, keyfile, max_ssh_retry, requestor, logger, tunnel_end_of_life_delay, ssh_tunnel_timeout):
        
        # the +1 is here to tell if the settings is 1, so we want 1 normal try and 1 retry
        nb_try = 1
        local_port_search['current'] = self._get_new_binding_port_from_base_port(local_port_search['base'])
        if not self.mute_mode:
            logger.info('Connection to %s with a ssh tunnel:' % uri)
        while True:
            current_binding_port = local_port_search['current']
            # note: lang=C + shell, so we can be sure we will have errors in english to grep them!
            # note2: -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no ==> do not look at host authorization
            # note3: -o BatchMode=yes => do not prompt password or such things
            # note4: -o ExitOnForwardFailure=true => if forward fail, exit!
            # note5: "root" user is strictly forbidden to prevent huge security holes
            # note6: -4 : ipv4 only, so it won't open only ipv6 if ipv4 is unbindable
            # note7: -o PreferredAuthentications=publickey : we do connect with key, so directly skip other methods
            # More info in : #SEF-2392
            cmd = '/usr/bin/ssh -4 -o PreferredAuthentications=publickey -o ExitOnForwardFailure=true -o BatchMode=yes -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no -L %d:localhost:%d ' % (current_binding_port, dest_port)
            
            if user is not None and user != "" and user != 'root':
                cmd += ' -l "%s"' % user
            
            if keyfile is not None and keyfile != "":
                cmd += ' -i "%s"' % os.path.expanduser(keyfile)
            
            # Protect requestor from non-shell characters
            requestor = re.sub(r'\W+', '-', requestor)
            cmd += ' "%s" "echo \"CONNECTED\";echo \" Mongo SSH Tunnel. Connection initial requestor is %s\";sleep %d"' % (addr, requestor, tunnel_end_of_life_delay)
            
            if not self.mute_mode:
                logger.info('   - searching a random local port available for the tunnel binding (trying %d): localhost:%s =(ssh tunnel)=> %s:22 =(mongodb)=> %s:%d (search try:%d)' %
                            (current_binding_port, current_binding_port, addr, addr, dest_port, nb_try))
            
            is_last_try = False
            if nb_try >= max_ssh_retry:
                is_last_try = True
            
            # Real error?
            nb_try += 1
            
            if six.PY3:
                cmd_clean = cmd
            else:
                cmd_clean = cmd.encode('ascii')
            
            p = subprocess.Popen(shlex.split(cmd_clean), stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=False, close_fds=True, preexec_fn=os.setsid, env={'LANG': 'C'})
            try:
                ssh_result = self._look_for_ssh_start_finish(p, addr, current_binding_port, logger, user=user, keyfile=keyfile, raise_on_unknow_error=is_last_try, ssh_tunnel_timeout=ssh_tunnel_timeout)
                is_connected = ssh_result['connected']
                # If we did connect, we are done
                if is_connected:
                    if not self.mute_mode:
                        logger.info('     - tunnel creation SUCCESS: localhost:%s =(ssh tunnel)=> %s:22 =(mongodb)=> %s:%d (search try:%d, ssh pid=%s)' %
                                    (current_binding_port, addr, addr, dest_port, nb_try, p.pid))
                    
                    # Remember the process, so we can clean it if we need in the future
                    self._living_sub_processes[f'{uri}:{current_binding_port}'] = p
                    break
                # Maybe it's just a problem of port already used, if so, increase
                ssh_error = ssh_result['error']
                # Maybe it's just an "already in use" error, if so just try with another port
                if ssh_error == SSH_ADDRESS_ALREADY_IN_USE_ERROR:
                    local_port_search['current'] = self._get_new_binding_port_from_base_port(local_port_search['base'])
                    if not self.mute_mode:
                        logger.info('     - the binding port %s was not free, trying another port (search try:%d)' % (current_binding_port, nb_try))
                    
                    continue
            except Exception as e:  # timeout
                if not self.mute_mode:
                    logger.warning('     - tunnel creation FAILED: localhost:%s =(ssh tunnel)=> %s:22 =(mongodb)=> %s:%d (search try:%d, ssh pid=%s) failed with error %s' % (current_binding_port, addr, addr, dest_port, nb_try, p.pid, str(e)))
                if nb_try > max_ssh_retry:
                    raise
    
    
    @staticmethod
    def _no_block_read(output):
        fd = output.fileno()
        fl = fcntl.fcntl(fd, fcntl.F_GETFL)
        fcntl.fcntl(fd, fcntl.F_SETFL, fl | os.O_NONBLOCK)
        try:
            return output.read()
        except:
            return ''
    
    
    def _kill_process(self, process):
        try:
            self._clear_database_statistics_threads()
            process.terminate()
            process.kill()
        except:  # already done
            pass
    
    
    def _look_for_ssh_start_finish(self, process, addr, current_binding_port, logger, user='shinken', keyfile='~/.ssh/id_rsa', raise_on_unknow_error=True, ssh_tunnel_timeout=SSH_TUNNEL_TIMEOUT_DEFAULT):
        start = time.time()
        if keyfile is None:
            keyfile = '~/.ssh/id_rsa'
        try:
            ssh_tunnel_timeout = float(ssh_tunnel_timeout)
        except Exception as exp:
            err = 'incorrect value ( %s ) for ssh_tunnel_value parameter ( %s )' % (ssh_tunnel_timeout, exp)
            if not self.mute_mode:
                logger.error(err)
            raise Exception(err)
        # Don't worry, we will exit, we just don't know if it will be fast or not
        while True:
            # Global command timeout (even if ssh have one, I prefer add ANOTHER one if process startup hangs for something)
            if ssh_tunnel_timeout > 0 and time.time() > (start + ssh_tunnel_timeout):
                warn = '     - the ssh tunnel to %s timed out to (after %ds) when trying to open it. Please check that your server is reachable and that the SSH daemon allows your connection.' % (addr, ssh_tunnel_timeout)
                if not self.mute_mode:
                    logger.warning(warn)
                self._kill_process(process)
                raise Exception(warn)
            
            # If the process is not dead, try to read what we can as stdout/stderr and look inside if we have a clue about finish or not
            process_finish = (process.poll() is not None)
            # print dir(process)
            if not process_finish:
                # Now read stdout/err, but as partial as process is still alive
                stdout = self._no_block_read(process.stdout)
                stderr = self._no_block_read(process.stderr)
            else:
                stdout, stderr = process.communicate()
            
            if stdout is None:
                stdout = ''
            stdout = bytes_to_unicode(stdout)
            lines = stdout.splitlines()
            for line in lines:
                # All we want is this line. It means that the ssh tunnel is done, and we did have a shell on the other side, so we are OK
                if 'CONNECTED' in line:
                    sock = None
                    # Try to really connect to it before give back the addr
                    try:
                        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
                        if ssh_tunnel_timeout > 0:
                            sock.settimeout(ssh_tunnel_timeout)
                        sock.connect(('localhost', current_binding_port))
                        sock.close()
                    except Exception as exp:  # The socket was not so good finally
                        try:
                            sock.close()
                        except:
                            pass
                        warn = '     - the ssh tunnel to %s timed out to (after %ds) when trying to test it (%s)' % (addr, ssh_tunnel_timeout, exp)
                        if not self.mute_mode:
                            logger.warning(warn)
                        self._kill_process(process)
                        return {'connected': False, 'error': warn}
                    return {'connected': True, 'error': None}
            
            # We look at stderr, it's where the truth lie
            if stderr is None:
                stderr = ''
            stderr = bytes_to_unicode(stderr)
            lines = stderr.splitlines()
            for line in lines:
                # Maybe there is just another ssh already started, not a problem in fact, the other process will allow us to connect
                if 'Address already in use' in line:  # SEF-9563 Thanks to ssh update, the message for this error changed, so we need to change it too
                    return {'connected': False, 'error': SSH_ADDRESS_ALREADY_IN_USE_ERROR}
                # Ok classic one: no ssh key on the other side, definitive error, exit from here
                elif 'Permission denied' in stderr:
                    err = '     - the ssh tunnel to %s failed because your local ssh key (%s) is not authorized on the distant server. You can use the command "ssh-copy-id -i %s %s@%s" from this local server to fix this. If any problem occurs, please refer to the Shinken Documentation. => %s' % \
                          (addr, keyfile, keyfile, user, addr, line)
                    if not self.mute_mode:
                        logger.error(err)
                    raise Exception(err)
            
            # If the process has finished, and we didn't catch a stderr we know, get a generic error
            if process_finish:
                err = '     - the ssh tunnel to %s failed to connect with the error: %s' % (addr, stderr)
                if raise_on_unknow_error:
                    if not self.mute_mode:
                        logger.error(err)
                    raise Exception(err)
                else:
                    return {'connected': False, 'error': err}
            
            # Still now finish? don't hammer the CPU
            time.sleep(0.01)
    
    
    @staticmethod
    def _do_connect(uri: str, replica_set: str, fsync: bool) -> 'MongoClient':
        if replica_set:
            con = MongoReplicaSetClient(uri, replicaSet=replica_set, fsync=fsync)
        else:
            con = MongoClient(uri, fsync=fsync)
        return con
    
    
    def _force_close_previous_ssh_process(self, uri):
        process = self._living_sub_processes.get(uri, None)
        if process:
            try:
                process.terminate()
                time.sleep(0.1)  # let the ssh process time to close
                process.kill()
            except:  # errors here are not a problem
                pass
            self._living_sub_processes.pop(uri, None)
    
    
    def close_all_tunnels(self) -> None:
        self._clear_database_statistics_threads()
        
        if len(self._living_sub_processes) == 0:
            return
        
        for process in list(self._living_sub_processes.values()):
            try:
                process.terminate()
            except:  # already done
                pass
        # We have some process, let them some ms to finish
        time.sleep(0.1)
        # But after some time, kill them all, no mercy
        for process in list(self._living_sub_processes.values()):
            try:
                process.kill()
            except:  # already done
                pass
        self._living_sub_processes.clear()
    
    
    def get_connection(
            self,
            uri: str,
            logger: 'PartLogger|None' = None,
            replica_set: str = '',
            fsync: bool = False,
            ssh_user: str = None,
            use_ssh: bool = False,
            ssh_keyfile=None,
            ssh_retry=1,
            requestor='(unknown requestor)',
            force_ssh_tunnel_recreation=False,
            tunnel_for='mongodb',
            tunnel_end_of_life_delay=30,
            ssh_tunnel_timeout=SSH_TUNNEL_TIMEOUT_DEFAULT
    ):
        if logger is None:
            logger = LoggerFactory.get_logger(requestor)
        logger = logger.get_sub_part('SSH TUNNEL')
        
        if self._need_data_hub and not self._data_hub:
            self._init_data_hub(logger)
        
        if force_ssh_tunnel_recreation:
            self._force_close_previous_ssh_process(uri)
        connection_result = self.connection_results.get(uri, None)
        if connection_result is not None:
            # Ok we have a connection, but maybe it's dead?
            try:
                connection_result.get_connection().admin.command('ping')
                # Ok we have a valid connection, give it
                if not self.mute_mode and logger.is_debug():
                    logger.debug(f'Reusing connection id:{connection_result.id} to {uri} from {connection_result.requesters} for {requestor}')
                connection_result.add_requester(requestor)
                
                return connection_result
            except Exception:
                if not self.mute_mode and logger.is_debug():
                    logger.debug(f'Closing connection id:{connection_result.id} to {uri} from {connection_result.requesters} as it is no more available')
                connection_result.disconnect()
                if connection_result.use_ssh:
                    connection_result.tunnel_keep_in_use_disable()
                    tunnel_key = f'{uri}:{connection_result.local_port}'
                    if self._living_sub_processes.get(tunnel_key, None):
                        if not self.mute_mode and logger.is_debug():
                            logger.debug(f'Closing SSH tunnel on port {connection_result.local_port} for connection id:{connection_result.id} to {uri} from {connection_result.requesters}')
                        self._force_close_previous_ssh_process(tunnel_key)
                self.connection_results.pop(uri, None)
                error_msg = 'The connection to the mongodb server %s is no more available, restart a new connection' % uri
                if not self.mute_mode:
                    logger.warning(error_msg)
        
        addr, dest_port = self._get_destination_from_uri(uri, logger)
        local_port_base = self._get_tunnel_port('%s:%s' % (addr, dest_port))
        local_port = None
        ssh_time = 0.0
        
        # Be sure there is a valid ssh waiting for our connection
        if use_ssh:
            before_ssh = time.time()
            local_port_search = {'base': local_port_base, 'current': local_port_base}
            self._spawn_background_ssh(uri, addr, dest_port, local_port_search, ssh_user, ssh_keyfile, ssh_retry, requestor, logger, tunnel_end_of_life_delay, ssh_tunnel_timeout)
            ssh_time = time.time() - before_ssh
            
            # The founded binding port can be different thant the base one
            local_port = local_port_search['current']
            
            # We change the mongo://addr/ => mongo://localhost/
            new_mongo_uri = uri.replace(addr, 'localhost')
            # If there was a specific port set in the uri, switch it
            if ':' in new_mongo_uri.replace('mongodb:', ''):  # avoid the first : in mongodb:
                new_mongo_uri = new_mongo_uri.replace(':%d' % dest_port, ':%d' % local_port)
            # If there was no such port, set it directly
            else:
                new_mongo_uri = new_mongo_uri.replace('localhost', 'localhost:%d' % local_port)
        else:
            new_mongo_uri = uri
        
        before_mongo = time.time()
        try:
            con = self._do_connect(new_mongo_uri, replica_set, fsync)
        except Exception:
            if use_ssh:
                # Mongo is not available ? so ... kill the ssh tunnel
                if not self.mute_mode:
                    logger.warning('Can\'t connect to mongo, close the SSH tunnel')
                self._force_close_previous_ssh_process(uri)
                raise MongoConnectionError('Mongo connection failure : localhost:%s ===(ssh tunnel)===> %s:22 ===(mongodb)===> %s:%d' % (local_port, addr, addr, dest_port))
            raise MongoConnectionError('Mongo connection failure to %s' % new_mongo_uri)
        
        mongodb_time = time.time() - before_mongo
        if use_ssh and not self.mute_mode:
            logger.info('   - SUCCESS mongo connection is OPENED with the SSH tunnel: localhost:%s =(ssh tunnel)=> %s:22 =(mongodb)=> %s:%d' % (local_port, addr, addr, dest_port))
        
        # Fill the final object
        connection_result = ConnectionResult(con, ssh_time, mongodb_time, local_port, use_ssh, requestor, connect_args=(new_mongo_uri, replica_set, fsync))
        self._update_connection_results(uri, connection_result, requestor)
        return connection_result
    
    
    def _get_mongo_connection_state(self, mongo_uri: str) -> 'tuple[bool, str]':
        address = self._get_computed_address_from_mongo_uri(mongo_uri) or mongo_uri
        if address not in self._database_statistics_threads:
            return False, 'Cannot retrieve mongo connection state'
        return self._database_statistics_threads[address].get_mongo_connection_state()
    
    
    @staticmethod
    def _compute_local_address(address: str) -> str:
        if address in BaseUrl.LOCAL_ADDRESSES:
            address = BaseUrl.get_local_ip() or BaseUrl.LOCAL_IP
        return address
    
    
    def _get_computed_address_from_mongo_uri(self, mongo_uri: str) -> str:
        address = self._get_destination_from_uri(mongo_uri.strip(), '')[0]
        if address:
            address = self._compute_local_address(address)
        return address
    
    
    def check_connexion_mongodb(self, mongo_uri: str) -> 'List[Dict[Union[bool, str]]]':
        if mongo_uri in self.connection_results:
            try:
                if self._data_hub:
                    address = self._get_computed_address_from_mongo_uri(mongo_uri)
                    if not address:
                        data = {
                            'statistics_collection_information': {
                                'status' : 'ERROR',
                                'message': 'Cannot extract destination server from the mongodb uri "%s". There must be one destination server to connect to.' % mongo_uri,
                            },
                            'stats'                            : {}
                        }
                    else:
                        uri_hash = hashlib.md5(address.encode('utf8', 'ignore')).hexdigest()
                        data = self._data_hub.get_data(uri_hash)
                        data['statistics_collection_information'] = {
                            'status' : 'OK',
                            'message': 'Statistics collected'
                        }
                else:
                    data = {
                        'databases_name'                   : [],
                        'pymongo_has_c'                    : {},
                        'bson_has_c'                       : {},
                        'statistics_collection_information': {
                            'status' : 'UNKNOWN',
                            'message': 'Data Hub is not initialized'
                        },
                        'stats'                            : {}
                    }
            except DataHubItemNotFound:
                data = {
                    'databases_name'                   : [],
                    'pymongo_has_c'                    : {},
                    'bson_has_c'                       : {},
                    'statistics_collection_information': {
                        'status' : 'UNKNOWN',
                        'message': 'Data Hub Item is not found'
                    },
                    'stats'                            : {}
                }
            except Exception as e:
                raise e
            is_connected, connection_state_message = self._get_mongo_connection_state(mongo_uri)
            data['status'] = 'OK' if is_connected else 'ERROR'
            data['is_connected'] = is_connected
            data['connect_msg'] = connection_state_message
            data['url'] = mongo_uri
            return [data]
        else:
            return []


mongo_by_ssh_mgr = SshTunnelMongoManager()
# When exiting, we need to close all living tunnels
atexit.register(mongo_by_ssh_mgr.close_all_tunnels)
