#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import itertools
import random
import sys
import threading
import time
from typing import cast
from weakref import proxy as weak_ref, ProxyType as WeakRef

from shinken.brok import Brok, BROK_VERSION_V020801P15
from shinken.brokerlink import BrokerLink, BrokerLinks
from shinken.check import NO_END_VALIDITY
from shinken.configuration_incarnation import PartConfigurationIncarnationContainer, PartConfigurationIncarnation
from shinken.log import PartLogger
from shinken.log import logger, get_chapter_string, LoggerFactory
from shinken.message import Message
from shinken.misc.filter import clear_user_access_cache, init_user_item_access_cache, get_element_full_name
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.broker.broker_contact import BrokerContact
from shinken.objects.broker.broker_host import BrokerHost
from shinken.objects.broker.broker_service import BrokerService
from shinken.objects.command import Command, Commands
from shinken.objects.config import Config
from shinken.objects.contact import Contact, Contacts
from shinken.objects.contactgroup import Contactgroup, Contactgroups
from shinken.objects.host import Hosts
from shinken.objects.host import VIEW_CONTACTS_DEFAULT_VALUE
from shinken.objects.hostgroup import Hostgroup, Hostgroups
from shinken.objects.notificationway import NotificationWay, NotificationWays
from shinken.objects.proxyitem import proxyitemsgraph
from shinken.objects.service import Services
from shinken.objects.servicegroup import Servicegroup, Servicegroups
from shinken.objects.state_id import STATE_ID, STATE_TYPE_ID
from shinken.objects.timeperiod import Timeperiod, Timeperiods
from shinken.pollerlink import PollerLink, PollerLinks
from shinken.reactionnerlink import ReactionnerLink, ReactionnerLinks
from shinken.receiverlink import ReceiverLink, ReceiverLinks
from shinken.schedulerlink import SchedulerLink, SchedulerLinks
from shinken.thread_helper import Thread
from shinken.toolbox.full_status_manipulator import FullStatusManipulator
from shinken.util import memory_trimming_disable_gc_collect_2, memory_trimming_allow_some_calls_to_gc_collect_2
from shinkensolutions.broker.monitoring_item_manager.filter import StaticCache
from shinkensolutions.indexed_collection.collection import IndexedCollection

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict, List, Union
    from shinken.objects.item import Item

CHAPTER_CONFIGURATION = get_chapter_string('CONFIGURATION')

# Only for scheduler mode
# Ok you are wondering why we don't add initial_broks_done? It's because the Livestatus modules need this part to do internal things.
# But don't worry, the vanilla regenerator will just skip it in all_done_linking :D
WANT_BROK_TYPE = (
    'program_status',
    'initial_host_status',
    'initial_hostgroup_status',
    'initial_service_status',
    'initial_servicegroup_status',
    'initial_contact_status',
    'initial_contactgroup_status',
    'initial_timeperiod_status',
    'initial_command_status'
)

INITIAL_BROKS_FOR_CONFIGURATION = (
    'initial_host_status',
    'initial_hostgroup_status',
    'initial_service_status',
    'initial_servicegroup_status',
    'initial_contact_status',
    'initial_contactgroup_status',
    'initial_timeperiod_status',
    'initial_command_status',
    'initial_scheduler_status',
    'initial_poller_status',
    'initial_reactionner_status',
    'initial_broker_status',
    'initial_receiver_status',
    'initial_broks_done',
    'proxy_items_graph',
    'program_status',
)


class RegeneratorFreshnessThread(Thread):
    def __init__(self, regenerator: 'Regenerator', lang: str = 'en', freshness_logger: 'PartLogger' = None) -> None:
        super(RegeneratorFreshnessThread, self).__init__(logger=freshness_logger)
        self.lang = lang
        self.loop_speed = None
        self.regenerator = regenerator
        self.random_step = random.random() * 10  # This will prevent multiple regenerator to check for freshness at the same time
        self.next_check_time: 'Optional[int]' = None
        self.min_check_interval: int = 60
        self.last_check_time: 'Optional[int]' = None
    
    
    def get_thread_name(self):
        return 'regenerator-freshness'
    
    
    def loop_turn(self):
        with self.regenerator.freshness_lock:
            now = int(time.time())
            
            if self.next_check_time and self.next_check_time <= now:
                self.next_check_time = None
                try:
                    self.last_check_time = now
                    # self.logger.debug('checking starts')
                    for item in itertools.chain(self.regenerator.hosts, self.regenerator.services):
                        end_of_validity = item.look_for_fake_unknown_state_due_to_missing_data_in_broker(self.regenerator, lang=self.lang, now=now)
                        if end_of_validity and (not self.next_check_time or (end_of_validity < self.next_check_time)):
                            self.next_check_time = end_of_validity
                    # self.logger.log_perf(now, '', 'checking done', min_time=0, info_time=5)
                except Exception:
                    self.logger.print_stack('checking raised an error')
            
            if self.next_check_time:
                next_loop = self.next_check_time - time.time()
                if next_loop < self.min_check_interval:
                    # self.logger.debug('next check should be in %.3fs, setting it to minimal delay: %ss' % (next_loop, self.min_freshness_check_interval))
                    next_loop = self.min_check_interval
                next_loop = next_loop + self.random_step
            else:
                next_loop = None
            # self.logger.debug('next check in %ss' % next_loop)
            # ok let's wait a bit, we are in a minute interval (at least) because we don't care about 1s delay here
            self.loop_speed = next_loop


# Class for a Regenerator. It will get broks, and 'regenerate' real objects from them
class Regenerator:
    def __init__(self, module_conf, import_in='(unknown)', with_indexed_items=False):
        
        # Our Real datas
        self.configs = {}
        self.currently_asking_full_broks = {}
        self.hosts = Hosts([])
        self.services = Services([])
        self.notificationways = NotificationWays([])
        self.contacts = Contacts([])
        self.hostgroups = Hostgroups([])
        self.servicegroups = Servicegroups([])
        self.contactgroups = Contactgroups([])
        self.timeperiods = Timeperiods([])
        self.commands = Commands([])
        self.schedulers = SchedulerLinks([])
        self.pollers = PollerLinks([])
        self.reactionners = ReactionnerLinks([])
        self.brokers = BrokerLinks([])
        self.receivers = ReceiverLinks([])
        # From now we only look for realms names
        self.realms = set()
        self.tags = {}
        self.services_tags = {}
        
        # Indexed data
        self.with_indexed = with_indexed_items
        self.indexed_items = IndexedCollection()
        self.indexed_items_2 = {}
        self.indexed_items.add_index('contact_name', self.get_contact_names, on_list=True)
        self.indexed_items.add_index('type', Regenerator.get_type)
        self.indexed_items.add_index('realm', lambda _h: _h.get_realm())
        
        # And in progress one
        self.inp_hosts = {}
        self.inp_services = {}
        self.inp_hostgroups = {}
        self.inp_servicegroups = {}
        self.inp_contactgroups = {}
        
        # Flag to say if our data came from the scheduler or not
        # (so if we skip *initial* broks)
        self.in_scheduler_mode = False
        
        # The Queue where to launch message, will be fill from the broker
        self.from_module_to_main_daemon_queue = None
        
        # We will have default value overridden by the module configurations
        self.default_properties_values = getattr(module_conf, 'configuration_default_value', {})
        self.minimal_time_before_an_element_become_missing_data = getattr(module_conf, 'minimal_time_before_an_element_become_missing_data', 0)
        
        # Sorted cache of all host and services
        self.all_hosts_and_services = []
        
        # Regenerator can have a specific thread that look at objects freshness
        self.freshness_thread: 'Optional[RegeneratorFreshnessThread]' = None
        self.freshness_lock = threading.Condition(threading.RLock())
        
        self.last_configuration_handle_uuid = None
        self.time_start_load_new_configuration = None
        self.part_configuration_incarnation_container = PartConfigurationIncarnationContainer()
        
        # Know in which module we are loaded
        self.import_in = import_in
        
        self.logger = LoggerFactory.get_logger().get_sub_part('REGENERATOR')
        self.logger_freshness = self.logger.get_sub_part('STATE-FRESHNESS', len('STATE-FRESHNESS'))
        
        # Flags
        self._have_received_initial_broks = False
        self.detail_timer = {}
        self.item_counter = {
            'all'             : 0,
            'hosts'           : 0,
            'clusters'        : 0,
            'checks'          : 0,
            'contacts'        : 0,
            'notificationways': 0,
            'hostgroups'      : 0,
            'servicegroups'   : 0,
            'contactgroups'   : 0,
            'timeperiods'     : 0,
            'commands'        : 0,
            'realms'          : 0,
        }
    
    
    @staticmethod
    def memory_management_init():
        memory_trimming_disable_gc_collect_2()
    
    
    def have_received_initial_broks(self):
        return self._have_received_initial_broks
    
    
    def get_configuration_incarnation_uuid(self) -> str:
        return self.part_configuration_incarnation_container.get_configuration_loaded_uuid()
    
    
    def get_last_part_configuration_incarnation(self) -> 'PartConfigurationIncarnation':
        return self.part_configuration_incarnation_container.last_part_configuration_incarnation_loaded
    
    
    def get_default_property_value(self, cls, property_name, missing=''):
        if cls not in self.default_properties_values:
            return missing
        if property_name not in self.default_properties_values[cls]:
            return missing
        return self.default_properties_values[cls][property_name]
    
    
    # Load an external queue for sending messages
    def load_external_queue(self, from_module_to_main_daemon_queue):
        self.from_module_to_main_daemon_queue = from_module_to_main_daemon_queue
    
    
    def want_brok(self, brok):
        if self.in_scheduler_mode:
            # If we are in a scheduler mode, some broks are dangerous, so we will skip them
            return brok.type not in WANT_BROK_TYPE
        return True
    
    
    def _filter_broks_by_configuration(self, brok: 'Brok') -> bool:
        brok_type = brok.type
        part_configuration_incarnation = brok.part_configuration_incarnation
        if brok_type in INITIAL_BROKS_FOR_CONFIGURATION and part_configuration_incarnation:
            can_accept_brok, cause = self.part_configuration_incarnation_container.check_part_is_loaded(part_configuration_incarnation)
            if not can_accept_brok:
                # logger.debug('[regenerator] brok [%s] refused because : %s' % (brok_type, cause))
                
                # We log only for brok_type 'initial_broks_done' for not spam log
                if brok_type == 'initial_broks_done':
                    self.logger.info(f'No need to reload the configuration part because {cause}')
                return False
            
            if brok_type == 'program_status':
                self.part_configuration_incarnation_container.add_part_received(part_configuration_incarnation)
            
            can_accept_brok = self.part_configuration_incarnation_container.check_configuration_incarnation_came_from_last_configuration_incarnation_received(part_configuration_incarnation.get_configuration_incarnation())
            if not can_accept_brok:
                # self.logger.debug(
                #     'We reject the configuration of %s because this configuration (make at %s from Arbiter %s) is older than the last configuration we received (make at %s from Arbiter %s).' % (
                #         part_configuration_incarnation.scheduler_name,
                #         PartLogger.format_time(part_configuration_incarnation.get_creation_date()),
                #         part_configuration_incarnation.get_configuration_incarnation().get_author(),
                #         PartLogger.format_time(self.part_configuration_incarnation_container.last_configuration_incarnation_received.get_creation_date()),
                #         self.part_configuration_incarnation_container.last_configuration_incarnation_received.get_author()
                #     )
                # )
                if brok_type == 'initial_broks_done':
                    self.logger.info(
                        'We reject the configuration of %s because this configuration (make at %s from Arbiter %s) is older than the last configuration we received (make at %s from Arbiter %s).' % (
                            part_configuration_incarnation.scheduler_name,
                            PartLogger.format_time(part_configuration_incarnation.get_creation_date()),
                            part_configuration_incarnation.get_configuration_incarnation().get_author(),
                            PartLogger.format_time(self.part_configuration_incarnation_container.last_configuration_incarnation_received.get_creation_date()),
                            self.part_configuration_incarnation_container.last_configuration_incarnation_received.get_author()
                        )
                    )
                
                return False
        
        return True
    
    
    def manage_brok(self, brok):
        # Look for a manager function for a brok, and call it
        manage = getattr(self, 'manage_' + brok.type + '_brok', False)
        
        managed = manage and self.want_brok(brok) and self._filter_broks_by_configuration(brok)
        if managed:
            return manage(brok)
    
    
    @staticmethod
    def update_element(element, data):
        # Update all, but not the id as we can't follow it between restarts
        if 'id' in data:
            del data['id']
        
        # We let the freshness thread know we did update this element
        setattr(element, 'last_broker_data_update', int(time.time()))
        for prop in data:
            prop_value = data[prop]
            if isinstance(prop_value, str):
                prop_value = sys.intern(prop_value)
            setattr(element, prop, prop_value)
        
        # Invalidation of list cache (see webui-enterprise/plugins/tabular/tabular.py test_remote_tabular)
        if hasattr(element, 'list_cache_var'):
            # delattr(element, 'list_cache_var')
            element.list_cache_var = {}
        
        elt_type = getattr(element, 'my_type', None)
        
        # Some extra stuff to do on Host & Service about full status, if available
        if elt_type in ['host', 'service']:
            full_status = data.get('current_full_status', None)
            if full_status is not None:
                full_status_time = data.get('update_time_full_status', None)
                state_id = FullStatusManipulator.get_state_from_full_status(full_status=full_status, full_status_change_time=full_status_time)
                # MISSING DATA received in full status from Scheduler, as regenerator only knows UNKNOWN state yet,
                # we convert state value for this element
                if state_id == 4:
                    element.last_state_change = full_status_time
                    element.state_type = 'HARD'
                    element.state_type_id = 1
                    element.state = 'UNKNOWN'
                    if element.got_business_rule:
                        element.bp_state = 3
                    else:
                        element.state_id = 3
    
    
    # We have a thread that is doing freshness checks and modification if need
    def launch_freshness_thread(self, lang='en'):
        self.freshness_thread = RegeneratorFreshnessThread(regenerator=self, lang=lang, freshness_logger=self.logger_freshness)
        self.freshness_thread.start_thread()
    
    
    def stop_freshness_thread(self):
        if self.freshness_thread:
            self.logger_freshness.debug('disabling thread')
            self.freshness_thread.stop()
            self.logger_freshness.debug('thread ended')
            self.freshness_thread = None
    
    
    def create_reversed_list(self):
        self.hosts.create_reversed_list()
        self.hostgroups.create_reversed_list()
        self.contacts.create_reversed_list()
        self.contactgroups.create_reversed_list()
        self.notificationways.create_reversed_list()
        self.services.create_reversed_list()
        self.servicegroups.create_reversed_list()
        self.timeperiods.create_reversed_list()
        self.commands.create_reversed_list()
    
    
    def all_done_linking(self, part_configuration_incarnation):
        memory_trimming_disable_gc_collect_2()
        try:
            self._all_done_linking(part_configuration_incarnation)
        finally:
            memory_trimming_allow_some_calls_to_gc_collect_2(2)
    
    
    # Now we get all data about an instance, link all this stuff :)
    def _all_done_linking(self, part_configuration_incarnation: 'PartConfigurationIncarnation') -> None:
        part_configuration_id = part_configuration_incarnation.get_part_id()
        scheduler_name = part_configuration_incarnation.scheduler_name
        has_already_loaded_a_part_configuration = self.get_configuration_incarnation_uuid() == part_configuration_incarnation.get_uuid()
        
        start = time.time()
        # check if the instance is really defined, so got ALL the init phase
        if part_configuration_id not in self.configs.keys():
            self.logger.error('The instance %s is not fully given, bailout' % part_configuration_incarnation)
            return
        
        logger_regenerator = self.logger.get_sub_part('scheduler=%s' % scheduler_name, register=False).get_sub_part('LOADING')
        logger_perf = logger_regenerator.get_sub_part('PERF')
        self.part_configuration_incarnation_container.add_part_loaded(part_configuration_incarnation)
        logger_regenerator.info('Loading configuration part : %s' % part_configuration_incarnation)
        
        # Try to load the in progress list and make them available for finding
        try:
            inp_hosts = self.inp_hosts[part_configuration_id]
            inp_hosts.create_reversed_list()
            inp_hostgroups = self.inp_hostgroups[part_configuration_id]
            inp_contactgroups = self.inp_contactgroups[part_configuration_id]
            inp_services = self.inp_services[part_configuration_id]
            inp_services.create_reversed_list()
            inp_servicegroups = self.inp_servicegroups[part_configuration_id]
        except Exception as exp:
            self.logger.error('Warning in all done:  %s ' % exp)
            self.logger.print_stack()
            return
        
        # We want to know if all local checks have really valid hosts, so first index them
        local_conf_host_name_idx = {host.get_name() for host in inp_hosts}
        for service in inp_services:
            service.part_configuration_incarnation = part_configuration_incarnation
            host_name = service.host_name
            if host_name not in local_conf_host_name_idx:
                self.logger.error('IMPORTANT ERROR: the shard %d does have an error: the check %s do not have a known host (%s).' % (part_configuration_id, service.service_description, service.host_name))
                self.logger.error('In order to solve this, we need to ask again to the scheduler the element definitions.')
                self.logger.error('*' * 30)
                del self.inp_hosts[part_configuration_id]
                del self.inp_hostgroups[part_configuration_id]
                del self.inp_contactgroups[part_configuration_id]
                del self.inp_services[part_configuration_id]
                del self.inp_servicegroups[part_configuration_id]
                
                del self.configs[part_configuration_id]
                self._ask_new_fill_brok(part_configuration_id)
                return
        
        t1 = time.time()
        logger_perf.info('Creating logger and load all new raw element ------------------------------ : %.2f' % (t1 - start))
        # We need to delete the reversed list here
        # Because we may override id so that the reversed_list is inconsistent (used in find by name)
        # Inconsistency comes from a mix of host group declaration :
        # within a host or with to define hostgroups statement
        # No worry, we recreate it just after!
        delattr(self.hostgroups, 'reversed_list')
        
        # Merge hostgroups with real ones
        for incoming_hostgroup in inp_hostgroups:
            hostgroup_name = incoming_hostgroup.hostgroup_name
            hostgroup = self.hostgroups.find_by_name(hostgroup_name)
            if not hostgroup:
                self.hostgroups[incoming_hostgroup.id] = incoming_hostgroup
        # We can declare hostgroups done
        self.hostgroups.create_reversed_list()
        
        t2 = time.time()
        logger_perf.info('Merging incoming hostgroup with already existing ones --------------------- : %.2f' % (t2 - t1))
        t3 = time.time()
        
        # Now link hosts with hostgroups, and commands
        for host in inp_hosts:
            new_hostgroups = []
            host_hostgroups = cast(str, host.hostgroups)
            for hostgroup_name in host_hostgroups.split(','):
                hostgroup = self.hostgroups.find_by_name(hostgroup_name)
                if hostgroup:
                    new_hostgroups.append(hostgroup)
            host.hostgroups = new_hostgroups
            
            # Now link commands
            self.linkify_a_command(host, 'check_command')
            self.linkify_a_command(host, 'event_handler')
            
            # Now link timeperiods
            self.linkify_a_timeperiod_by_name(host, 'notification_period')
            self.linkify_a_timeperiod_by_name(host, 'check_period')
            self.linkify_a_timeperiod_by_name(host, 'maintenance_period')
            
            # And link contacts too
            self.linkify_contacts(host, 'contacts')
            self.linkify_contacts(host, 'view_contacts')
            self.linkify_contacts(host, 'notification_contacts')
            self.linkify_contacts(host, 'edition_contacts')
            
            # Linkify tags
            for tag in host.tags:
                if tag not in self.tags:
                    self.tags[tag] = 0
                self.tags[tag] += 1
            
            # find the old host and remove associated service
            old_host = self.hosts.find_by_name(host.get_name())
            if old_host:
                for service in old_host.services:
                    host_name = service.host_name
                    service_description = service.service_description
                    # First remove the old service if present
                    self.services.remove_by_name_and_hostname(host_name, service_description)
            
            # Remove the old host is present
            self.hosts.remove_by_name(host.get_name())
            
            self.hosts[host.id] = host
        
        self.hosts.create_reversed_list()
        self.services.remove_orphan_services()
        
        t4 = time.time()
        logger_perf.info('Linking hosts => hostgroups / command / timeperiod / contacts ------------- : %.2f' % (t4 - t3))
        
        # We need to delete the reversed list here
        # Because we may override id so that the reversed_list
        # is inconsistent (used in find by name)
        # Inconsistency comes from a mix of service group declaration :
        # within a service or with the define servicegroup statement
        # No worry, we recreate it just after!
        delattr(self.servicegroups, 'reversed_list')
        
        # Merge service groups with real ones
        for incoming_service_group in inp_servicegroups:
            service_group_name = incoming_service_group.servicegroup_name
            service_group = self.servicegroups.find_by_name(service_group_name)
            # If the service group already exist, just add the new services into it
            if not service_group:
                self.servicegroups[incoming_service_group.id] = incoming_service_group
        # We can declare service groups done
        self.servicegroups.create_reversed_list()
        
        t5 = time.time()
        logger_perf.info('Merging incoming service groups with already existing ones ---------------- : %.2f' % (t5 - t4))
        
        sum1 = 0.0
        sum2 = 0.0
        sum3 = 0.0
        sum4 = 0.0
        
        # Now link services with hosts, services groups, and commands
        self.services.set_hosts_for_index(self.hosts)
        for service in inp_services:
            t52 = time.time()
            new_service_groups = []
            service_servicegroups = cast(str, service.servicegroups)
            for service_group_name in service_servicegroups.split(','):
                service_group = self.servicegroups.find_by_name(service_group_name)
                if service_group:
                    new_service_groups.append(service_group)
            service.servicegroups = new_service_groups
            
            # Now link with host
            host_name = service.host_name
            service_host = self.hosts.find_by_name(host_name)
            service.host = weak_ref(service_host) if service_host else service_host
            if service.host:
                service.host.services.append(weak_ref(service))
            
            sum1 += (time.time() - t52)
            
            t53 = time.time()
            # Now link Command() objects
            self.linkify_a_command(service, 'check_command')
            self.linkify_a_command(service, 'event_handler')
            
            # Now link timeperiods
            self.linkify_a_timeperiod_by_name(service, 'notification_period')
            self.linkify_a_timeperiod_by_name(service, 'check_period')
            self.linkify_a_timeperiod_by_name(service, 'maintenance_period')
            
            sum2 += (time.time() - t53)
            
            t54 = time.time()
            # And link contacts too
            self.linkify_contacts(service, 'contacts')
            
            # Linkify services tags
            for tag in service.tags:
                if tag not in self.services_tags:
                    self.services_tags[tag] = 0
                self.services_tags[tag] += 1
            
            sum3 += (time.time() - t54)
            
            t55 = time.time()
            
            # We can really declare this host OK now
            host_name = service.host_name
            service_description = service.service_description
            # First remove the old service if present
            self.services.remove_by_name_and_hostname(host_name, service_description)
            self.services.add_service_with_index(service)
            sum4 += (time.time() - t55)
            # logger_perf.debug('Creation the service : %s' % service.get_full_name())
        
        t6 = time.time()
        logger_perf.info('Linking services => host / servicegroups / command / timeperiod / contact - : %.2f' % (t6 - t5))
        logger_perf.debug('Sub times: 1:%.2f  2:%.2f  3:%.2f  4:%.2f' % (sum1, sum2, sum3, sum4))
        
        # Link services into service groups
        # First clean members
        for service_group in self.servicegroups:
            service_group.members = []
        # Then set the services into the service groups
        for service in self.services:
            for service_group in service.servicegroups:
                service_group.members.append(service)
        
        t7 = time.time()
        logger_perf.info('Linking service groups => services ---------------------------------------- : %.2f' % (t7 - t6))
        
        # Same for hosts
        # Link hosts into hostgroups
        # First clean members
        for hostgroup in self.hostgroups:
            hostgroup.members = []
        # Then set the services into the service groups
        for host in self.hosts:
            for hostgroup in host.hostgroups:
                hostgroup.members.append(host)
        
        t72 = time.time()
        logger_perf.info('Linking hostgroups => hosts ----------------------------------------------- : %.2f' % (t72 - t7))
        
        # Add realm of these hosts. Only the first is useful
        for host in inp_hosts:
            self.realms.add(host.realm)
            break
        
        t8 = time.time()
        logger_perf.info('Build realm list ---------------------------------------------------------- : %.2f' % (t8 - t72))
        
        # Now we can link all impacts/source problem list but only for the new ones here of course
        for host in inp_hosts:
            self.linkify_host_and_hosts(host, 'parents')
            self.linkify_host_and_hosts(host, 'childs')
            self.linkify_dict_srv_and_hosts(host, 'parent_dependencies')
            self.linkify_dict_srv_and_hosts(host, 'child_dependencies')
        
        t9 = time.time()
        logger_perf.info('Linking host => host & check dependencies ( parents / childs ) ------------ : %.2f' % (t9 - t8))
        
        # Now services too
        for service in inp_services:
            self.linkify_dict_srv_and_hosts(service, 'parent_dependencies')
            self.linkify_dict_srv_and_hosts(service, 'child_dependencies')
        
        t91 = time.time()
        logger_perf.info('Linking checks => host & check dependencies ( parents / childs ) ---------- : %.2f' % (t91 - t9))
        
        # Linking TIMEPERIOD exclude with real ones now
        for timeperiod in self.timeperiods:
            new_exclude = []
            for timeperiod_exclude in timeperiod.exclude:
                timeperiod_exclude_name = timeperiod_exclude.timeperiod_name
                tag = self.timeperiods.find_by_name(timeperiod_exclude_name)
                if tag:
                    new_exclude.append(tag)
            timeperiod.exclude = new_exclude
        
        t10 = time.time()
        logger_perf.info('Linking timeperiod => excluded timeperiod --------------------------------- : %.2f' % (t10 - t91))
        
        # clean old objects
        if has_already_loaded_a_part_configuration:
            t101 = time.time()
            logger_regenerator.info('Cleanup of contacts from previous configuration has already been done with update of contacts from another Scheduler.')
        else:
            to_del = {contact.id for contact in self.contacts if contact.configuration_incarnation != part_configuration_incarnation.get_configuration_incarnation()}
            
            for host in self.hosts:
                # We need to unlink only item from previous configuration
                if host.part_configuration_incarnation != part_configuration_incarnation:
                    self.unlinking_contact(host, to_del, 'contacts')
                    self.unlinking_contact(host, to_del, 'view_contacts')
                    self.unlinking_contact(host, to_del, 'notification_contacts')
                    self.unlinking_contact(host, to_del, 'edition_contacts')
            
            for service in self.services:
                # We need to unlink only item from previous configuration
                if service.part_configuration_incarnation != part_configuration_incarnation:
                    self.unlinking_contact(service, to_del, 'contacts')
            
            for c_id in to_del:
                del self.contacts[c_id]
            self.contacts.create_reversed_list()
            t101 = time.time()
            logger_perf.info('Cleanup of contacts from previous configuration --------------------------------- : %.2f' % (t101 - t10))
        
        # Merge contactgroups with real ones
        for incoming_contactgroup in inp_contactgroups:
            contactgroup_name = incoming_contactgroup.contactgroup_name
            contactgroup = self.contactgroups.find_by_name(contactgroup_name)
            # If the contactgroup already exist, just add the new
            # contacts into it
            if not contactgroup:
                self.contactgroups[incoming_contactgroup.id] = incoming_contactgroup
        
        # We can declare contactgroups done
        self.contactgroups.create_reversed_list()
        
        t11 = time.time()
        logger_perf.info('Merging incoming contactgroups with already existing ones ----------------- : %.2f' % (t11 - t101))
        
        # Same for contacts
        # Link contacts into contactgroups
        # First clean members
        for contactgroup in self.contactgroups:
            contactgroup.members = []
        # Then set the contacts into the contact groups
        for contact in self.contacts:
            # We must set our contact into the real contactgroups. but there are 2 cases:
            # * it's our first start/shard, our inp_cg are already the self.contactgroups
            # * it's not our first start/shard, then our inp_cg will be dropped, and we just get its name to get the self.contactgroups one
            for inp_cg in contact.contactgroups:
                self.contactgroups.find_by_name(inp_cg.get_name()).members.append(contact)
        
        t12 = time.time()
        logger_perf.info('Linking contactgroups => contacts ----------------------------------------- : %.2f' % (t12 - t11))
        
        self.all_hosts_and_services = []
        self.all_hosts_and_services.extend(self.hosts)
        self.all_hosts_and_services.extend(self.services)
        self.all_hosts_and_services.sort(key=get_element_full_name)
        
        t13 = time.time()
        logger_perf.info('Creating hosts and checks list -------------------------------------------- : %.2f' % (t13 - t12))
        
        init_user_item_access_cache(self)
        
        t14 = time.time()
        logger_perf.info('Creating user access index ------------------------------------------------ : %.2f' % (t14 - t13))
        
        del self.inp_hosts[part_configuration_id]
        del self.inp_hostgroups[part_configuration_id]
        del self.inp_contactgroups[part_configuration_id]
        del self.inp_services[part_configuration_id]
        del self.inp_servicegroups[part_configuration_id]
        
        t15 = time.time()
        logger_perf.info('clean old objects --------------------------------------------------------- : %.2f' % (t15 - t14))
        
        hosts_and_services_to_refresh = [i for i in self.all_hosts_and_services if i.part_configuration_incarnation == part_configuration_incarnation]
        StaticCache.refresh(hosts_and_services_to_refresh, part_configuration_incarnation.get_uuid())
        t16 = time.time()
        logger_perf.info('refresh static cache for %8i hosts and services ---------------------- : %.2f' % (len(hosts_and_services_to_refresh), t16 - t15))
        
        nb_clusters = len([h for h in self.hosts if h.got_business_rule])
        self.item_counter = {
            'all'             : len(self.hosts) + len(self.services),
            'hosts'           : len(self.hosts) - nb_clusters,
            'clusters'        : nb_clusters,
            'checks'          : len(self.services),
            'contacts'        : len(self.contacts),
            'notificationways': len(self.notificationways),
            'hostgroups'      : len(self.hostgroups),
            'servicegroups'   : len(self.servicegroups),
            'contactgroups'   : len(self.contactgroups),
            'timeperiods'     : len(self.timeperiods),
            'commands'        : len(self.commands),
            'realms'          : len(self.realms),
        }
        
        logger_regenerator.info('Configuration size')
        logger_regenerator.info('- hosts -------------- : %d ' % self.item_counter['hosts'])
        logger_regenerator.info('- clusters ----------- : %d ' % self.item_counter['clusters'])
        logger_regenerator.info('- checks ------------- : %d ' % self.item_counter['checks'])
        logger_regenerator.info('- contacts ----------- : %d ' % self.item_counter['contacts'])
        logger_regenerator.info('- notificationways --- : %d ' % self.item_counter['notificationways'])
        logger_regenerator.info('- hostgroups --------- : %d ' % self.item_counter['hostgroups'])
        logger_regenerator.info('- servicegroups ------ : %d ' % self.item_counter['servicegroups'])
        logger_regenerator.info('- contactgroups ------ : %d ' % self.item_counter['contactgroups'])
        logger_regenerator.info('- timeperiods -------- : %d ' % self.item_counter['timeperiods'])
        logger_regenerator.info('- commands ----------- : %d ' % self.item_counter['commands'])
        logger_regenerator.info('- realms ------------- : %d ' % self.item_counter['realms'])
        logger_regenerator.info('The configuration with shard_id=%d was fully loaded in %.3fs' % (part_configuration_id, (time.time() - start)))
    
    
    @staticmethod
    def _get_contact_list(item):
        return [contact.get_name().lower() for contact in (item.view_contacts if item.my_type == 'host' else item.host.view_contacts)]
    
    
    def _get_contact_list_default_everyone(self, item):
        view_contacts = item.view_contacts if item.my_type == 'host' else item.host.view_contacts
        if view_contacts:
            return self._get_contact_list(item)
        else:
            return [VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE]
    
    
    def _build_indexed_items_collection(self: 'Regenerator') -> None:
        if not self.with_indexed:
            return
        
        start_time = time.time()
        
        reindex_count = 0
        reindex_time = 0
        find_time = 0
        for item in self.all_hosts_and_services:
            uuid = item.get_instance_uuid()
            
            t0 = time.time()
            item_in_indexed_items = self.indexed_items_2.get(uuid, None)
            find_time += time.time() - t0
            if self.has_change_for_item_in_indexed_items(item, item_in_indexed_items):
                t0 = time.time()
                if item_in_indexed_items:
                    self.indexed_items.remove({'instance_uuid': uuid})
                self.indexed_items.append(item)
                self.indexed_items_2[uuid] = item
                reindex_count += 1
                reindex_time += time.time() - t0
        
        self.logger.info('generated indexed %s item list cache in %.3fs, with %s in this regenerator with %d total elements in indexed_items reindex_time:%.3fs find_time:%.3fs' % (
            reindex_count, time.time() - start_time, len(self.all_hosts_and_services), self.indexed_items.counter, reindex_time, find_time))
    
    
    def has_change_for_item_in_indexed_items(self, item, item_in_indexed_items):
        if not item_in_indexed_items:
            return True
        if item_in_indexed_items is item:
            return False  # Watch it it is False
        
        if item.get_name() != item_in_indexed_items.get_name():
            return True
        if item.get_realm() != item_in_indexed_items.get_realm():
            return True
        if self.get_type(item) != self.get_type(item_in_indexed_items):
            return True
        if self.get_contact_names(item) != self.get_contact_names(item_in_indexed_items):
            return True
        return False
    
    
    @staticmethod
    def get_type(item):
        return 'cluster' if (item.get_item_type() == 'host' and item.got_business_rule) else item.get_item_type()
    
    
    def get_contact_names(self, item):
        _def_access = self.get_default_property_value('host', 'view_contacts', missing=VIEW_CONTACTS_DEFAULT_VALUE.NOBODY)
        return self._get_contact_list_default_everyone(item) if _def_access == VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE else self._get_contact_list(item)
    
    
    @staticmethod
    def _build_user_filter_for_indexed_lookup(user: 'Contact') -> 'Dict[str, Dict]':
        if getattr(user, 'is_admin', False):
            return {}
        else:
            return {'contact_name': {'$in': [user.get_name(), VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE]}}
    
    
    def indexed_items_lookup(self: 'Regenerator', lookup_filter: 'Dict', user: 'Contact') -> 'List[Union[BrokerHost,BrokerService]]':
        if not self.with_indexed:
            raise NotImplementedError
        
        search_filter = lookup_filter.copy()
        search_filter.update(self._build_user_filter_for_indexed_lookup(user))
        return self.indexed_items.find(search=search_filter)
    
    
    # We look for o.prop (CommandCall) and we link the inner Command() object with our real ones
    def linkify_a_command(self, o, prop):
        cc = getattr(o, prop, None)
        # if the command call is void, bypass it
        if not cc:
            setattr(o, prop, None)
            return
        cmdname = cc.command
        c = self.commands.find_by_name(cmdname)
        cc.command = weak_ref(c) if c else c
    
    
    # We look at o.prop and for each command we relink it
    def linkify_commands(self, o, prop):
        v = getattr(o, prop, None)
        if not v:
            # If do not have a command list, put a void list instead
            setattr(o, prop, [])
            return
        
        for cc in v:
            cmdname = cc.command
            if isinstance(cmdname, str):
                c = self.commands.find_by_name(cmdname)
                cc.command = weak_ref(c) if c else c
    
    
    # We look at the timeperiod() object of o.prop and we replace it with our true one
    def linkify_a_timeperiod(self, o, prop):
        t = getattr(o, prop, None)
        if not t:
            setattr(o, prop, None)
            return
        tpname = t.timeperiod_name
        tp = self.timeperiods.find_by_name(tpname)
        setattr(o, prop, weak_ref(tp) if tp else tp)
    
    
    # same than before, but the value is a string here
    def linkify_a_timeperiod_by_name(self, o, prop):
        tpname = getattr(o, prop, None)
        if not tpname:
            setattr(o, prop, None)
            return
        tp = self.timeperiods.find_by_name(tpname)
        setattr(o, prop, weak_ref(tp) if tp else tp)
    
    
    @staticmethod
    def unlinking_contact(item: 'Item', contacts_id_to_unlink: 'set[int]', property_name: str):
        item_value = getattr(item, property_name)
        
        if not item_value:
            return
        
        new_item_value = [item for item in item_value if item.id not in contacts_id_to_unlink]
        setattr(item, property_name, new_item_value)
    
    
    # We look at o.prop and for each contacts in it,  we replace it with true object in self.contacts
    def linkify_contacts(self, o, prop):
        v = getattr(o, prop)
        
        if not v:
            return
        
        new_v = []
        for cname in v:
            c = self.contacts.find_by_name(cname)
            if c:
                new_v.append(weak_ref(c))
        setattr(o, prop, new_v)
    
    
    # We got a service/host dict, we want to get back to a flat list
    def linkify_dict_srv_and_hosts(self, o, prop):
        v = getattr(o, prop)
        
        if not v:
            setattr(o, prop, [])
        
        new_v = []
        
        for name in v['services']:
            elts = name.split('/')
            hname = elts[0]
            sdesc = '/'.join(elts[1:])
            s = self.services.find_srv_by_name_and_hostname(hname, sdesc)
            if s:
                if not isinstance(s, WeakRef):
                    s = weak_ref(s)
                new_v.append(s)
        for hname in v['hosts']:
            h = self.hosts.find_by_name(hname)
            if h:
                new_v.append(weak_ref(h))
        setattr(o, prop, new_v)
    
    
    def linkify_host_and_hosts(self, o, prop):
        v = getattr(o, prop)
        
        if not v:
            setattr(o, prop, [])
        
        new_v = []
        for hname in v:
            h = self.hosts.find_by_name(hname)
            if h:
                new_v.append(weak_ref(h))
        setattr(o, prop, new_v)
    
    
    # ------- Brok management part
    
    def initial_status_freshness_check(self, item: 'Union[BrokerHost, BrokerService]') -> None:
        if not self.freshness_thread:
            return
        with self.freshness_lock:
            lang = self.freshness_thread.lang
            item.look_for_fake_unknown_state_due_to_missing_data_in_broker(self, lang, time.time())
    
    
    def before_after_hook(self, brok, item):
        # This can be used by derived classes to compare the data in the brok with the object which will be updated by these data.
        # For example, it is possible to find out in this method whether the state of a host or service has changed.
        
        if 'next_schedule' in brok.type:
            return
        
        if 'host_' not in brok.type and 'service_' not in brok.type:
            return
        
        # TODO: MISSING DATA start time mismatch between regenerator and Event/SLA modules, cf. SEF-9992
        
        # Scheduler data hacking, like :
        # * state conversion (MISSING DATA -> UNKNOWN),
        # * data sanitize (when in MISSING DATA, we have to correct missing_data_activation_time, state_validity_en_time and state_validity_period)
        # * make full_status_change_time prevailing on last_state_change
        # * last_state_change handling after MISSING DATA state
        brok_version = brok.data.get('brok_version', 0)
        full_status_change_time = None
        new_state_from_scheduler = None
        full_status = None
        if brok_version >= BROK_VERSION_V020801P15:
            full_status_change_time = brok.data.get('full_status_change_time', brok.data.get('update_time_full_status', None))
            
            # Full status time prevail on last_state_change, if available
            if full_status_change_time:
                brok.data['last_state_change'] = full_status_change_time
            
            full_status = brok.data.get('current_full_status', None)
            new_state_from_scheduler = FullStatusManipulator.get_state_from_full_status(full_status=full_status, full_status_change_time=full_status_change_time)
            
            # Make regenerator behave the same way
            # - when MISSING DATA occurs in Broker,
            # - when it comes from Scheduler
            if new_state_from_scheduler == STATE_ID.MISSING_DATA:
                # WARNING: when full_status from Scheduler returns MISSING DATA state,
                # the following data will not be reliable due to reschedule
                # we must correct these in Broker (here)
                brok.data['missing_data_activation_time'] = 0
                brok.data['state_validity_end_time'] = 0
                brok.data['state_validity_period'] = NO_END_VALIDITY
                
                # As regenerator only knows UNKNOWN status yet,
                # we convert state value for this element
                brok.data['state_type'] = 'HARD'
                brok.data['state_type_id'] = STATE_TYPE_ID.HARD
                brok.data['state'] = 'UNKNOWN'
                if brok.data['got_business_rule']:
                    brok.data['bp_state'] = STATE_ID.UNKNOWN
                else:
                    brok.data['state_id'] = STATE_ID.UNKNOWN
        
        missing_data_activation_time = brok.data.get('missing_data_activation_time', 0)
        if missing_data_activation_time > brok.data['last_state_change'] > brok.data['state_validity_end_time'] > 0:
            # We may have received a context change between end of status and missing data activation time,
            # we set a possible end time (>= start time)
            brok.data['state_validity_end_time'] = missing_data_activation_time
        now = time.time()
        
        # When we were in MISSING DATA, handle received Scheduler update accordingly
        last_missing_data_start_in_broker = getattr(item, 'last_missing_data_start_in_broker', 0)
        if last_missing_data_start_in_broker > 0:
            
            if full_status_change_time:
                last_change_from_scheduler = full_status_change_time
            else:
                last_change_from_scheduler = brok.data.get('last_state_change', 0)
            
            # Status from Scheduler has already expired,
            # - stay in "MISSING DATA from Broker"
            # - keep MISSING DATA start time from Broker
            if 0 < missing_data_activation_time < now:
                brok.data['last_state_change'] = last_missing_data_start_in_broker
                if 'full_status_change_time' in brok.data:
                    brok.data['full_status_change_time'] = last_missing_data_start_in_broker
                if 'current_full_status' in brok.data:
                    brok.data['current_full_status'] = FullStatusManipulator.set_missing_data_state_in_full_status(brok.data['current_full_status'])
                brok.data['state_type'] = 'HARD'
                brok.data['state_type_id'] = STATE_TYPE_ID.HARD
                brok.data['state'] = 'UNKNOWN'
                if brok.data['got_business_rule']:
                    brok.data['bp_state'] = STATE_ID.UNKNOWN
                else:
                    brok.data['state_id'] = STATE_ID.UNKNOWN
                return
            # A new status is available from Scheduler, we quit "MISSING DATA from Broker"
            # When status in Scheduler is also MISSING DATA, keep start period from Broker if we have no context (as if the state just "continues")
            # Else, shift start period in Broker to now if Scheduler status started before MISSING DATA period in Broker
            elif (new_state_from_scheduler is None or new_state_from_scheduler != STATE_ID.MISSING_DATA or new_state_from_scheduler != full_status) and last_missing_data_start_in_broker > last_change_from_scheduler:
                brok.data['last_state_change'] = time.time()
            item.last_missing_data_start_in_broker = 0
        # We are not/no more in MISSING DATA, keep last_state_change updated by Broker :
        # suppose a more recent local (ie: in Broker) last_state_change is due to a previous leaving of "MISSING DATA from Broker"
        elif getattr(item, 'last_state_change', 0) > brok.data.get('last_state_change', 0):
            brok.data['last_state_change'] = item.last_state_change
        
        # MISSING DATA management is fine tuned here (freshness thread wake up, last_state_change, ...)
        with self.freshness_lock:
            # self.logger_freshness.debug('brok_type:[%s] elt:[%s] state:[%s] mdat:[%s]' % (brok.type, item.get_dbg_name(), brok.data['state_id'], self.logger.format_time(missing_data_activation_time)))
            
            if self.freshness_thread and missing_data_activation_time > 0:
                # Compute freshness thread next wake up time, and update it if needed
                freshness_thread = self.freshness_thread
                # self.logger_freshness.debug('brok_type:[%s] expire in %.3fs' % (brok.type, missing_data_activation_time - now))
                
                if item.got_business_rule:
                    item_state = getattr(item, 'bp_state', 0)
                else:
                    item_state = getattr(item, 'state_id', 0)
                obsolete_missing_data_activation_time = missing_data_activation_time < now and item_state == STATE_ID.UNKNOWN
                sooner_next_check_time = freshness_thread.min_check_interval + (freshness_thread.last_check_time if freshness_thread.last_check_time else now)
                next_missing_data_activation_time = missing_data_activation_time if missing_data_activation_time > sooner_next_check_time else sooner_next_check_time
                
                if not obsolete_missing_data_activation_time and (not freshness_thread.next_check_time or (freshness_thread.next_check_time > next_missing_data_activation_time)):
                    freshness_thread.next_check_time = next_missing_data_activation_time
                    freshness_thread.notify()
                    # self.logger_freshness.debug('brok_type:[%s] will change next wake up in %s' % (brok.type, next_missing_data_activation_time - time.time()))
    
    
    # ------ INITIAL PART
    def manage_program_status_brok(self, b):
        # We do not want to modify elements while switching elements
        # NOTE: we do not lock all access, because we will only change atomic elements on the elements
        with self.freshness_lock:
            self.do_manage_program_status_brok(b)
    
    
    def manage_asking_initial_broks_brok(self, b):
        data = b.data
        instance_id = data['instance_id']
        self.currently_asking_full_broks[instance_id] = True
    
    
    def do_manage_program_status_brok(self, b):
        data = b.data
        instance_id = data['instance_id']
        part_configuration_incarnation: 'PartConfigurationIncarnation' = b.part_configuration_incarnation
        self.part_configuration_incarnation_container.add_part_received(part_configuration_incarnation)
        logger_regenerator = self.logger.get_sub_part('scheduler=%s' % part_configuration_incarnation.scheduler_name, register=False)
        logger_regenerator.info('Creating new configuration for %s ' % part_configuration_incarnation)
        
        config = Config()
        self.update_element(config, data)
        
        # If we were asking for this config, remove it from protection list
        if instance_id in self.currently_asking_full_broks:
            del self.currently_asking_full_broks[instance_id]
        
        # Clean all in_progress things.
        self.inp_hosts[instance_id] = Hosts([])
        self.inp_services[instance_id] = Services([])
        self.inp_hostgroups[instance_id] = Hostgroups([])
        self.inp_servicegroups[instance_id] = Servicegroups([])
        self.inp_contactgroups[instance_id] = Contactgroups([])
        
        # And we save it
        self.configs[instance_id] = config
        
        # Clean the old 'hard' objects
        
        # We should clean all previously added hosts and services
        to_del_h = [h for h in self.hosts if h.instance_id == instance_id]
        to_del_srv = [s for s in self.services if s.instance_id == instance_id]
        
        # Clean hosts from hosts and hostgroups
        for h in to_del_h:
            del self.hosts[h.id]
        
        # Now clean all hostgroups too
        for hg in self.hostgroups:
            # Exclude from members the hosts with this inst_id
            hg.members = [h for h in hg.members if h.instance_id != instance_id]
        
        for s in to_del_srv:
            del self.services[s.id]
        
        # Now clean service groups
        for sg in self.servicegroups:
            sg.members = [s for s in sg.members if s.instance_id != instance_id]
        
        # We now regenerate reversed list so the client will find only real objects
        self.create_reversed_list()
        
        # TODO: Warning this may have a severe cost, to be analyzed
        self.all_hosts_and_services = []
        self.all_hosts_and_services.extend(self.hosts)
        self.all_hosts_and_services.extend(self.services)
        self.all_hosts_and_services.sort(key=get_element_full_name)
        
        logger_regenerator.info('clean %d hosts from previous configuration' % len(to_del_h))
        logger_regenerator.info('clean %d checks from previous configuration' % len(to_del_srv))
        
        # DEBUG pour trouver les références sur un hôte (voir plus bas)
        # test_host = to_del_h[0] if to_del_h else None
        
        del to_del_h
        del to_del_srv
        clear_user_access_cache()
        
        # DEBUG pour trouver les références sur un objet
        # if test_host:
        #     import gc
        #     try:
        #         referrers = gc.get_referrers(test_host)
        #         logger_regenerator.error('REF[%s] => %s' % (test_host.get_full_name(), referrers))
        #         for item in referrers:
        #             if isinstance(item, Host) or isinstance(item, Service):
        #                 name = item.get_full_name()
        #                 if not name:
        #                     name = item.get_instance_uuid()
        #                 if not name:
        #                     name = '%s / %s' % (item, item.id)
        #                 logger_regenerator.error('REF[%s] > %s' % (test_host.get_full_name(), name))
        #             elif isinstance(item, list):
        #                 logger_regenerator.error('REF[%s] > LIST:%s' % (test_host.get_full_name(), item))
        #             else:
        #                 logger_regenerator.error('REF[%s] > type:%s' % (test_host.get_full_name(), type(item)))
        #
        #     except:
        #         logger_regenerator.print_stack()
        #         pass
        # else:
        #     logger_regenerator.error('REF[None] > ')
    
    
    @staticmethod
    def manage_proxy_items_graph_brok(b):
        son_to_fathers = b.data['proxy_items_graph_son_to_fathers']
        proxyitemsgraph.update_from_son_to_fathers(son_to_fathers)
    
    
    def manage_initial_host_status_brok(self, brok: Brok, skip_useless_in_configuration=True) -> None:
        data = brok.data
        inst_id = data['instance_id']
        
        # Try to get the "in progress" hosts
        try:
            inp_hosts = self.inp_hosts[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_host_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        host = BrokerHost({}, skip_useless_in_configuration=skip_useless_in_configuration)
        host.list_cache_var = {}
        host.part_configuration_incarnation = brok.part_configuration_incarnation
        self.before_after_hook(brok, host)
        self.update_element(host, data)
        self.initial_status_freshness_check(host)
        
        # We need to rebuild Downtime and Comment relationship
        for dtc in host.downtimes:
            dtc.ref = host
        
        # Ok, put in the "in progress" hosts
        inp_hosts[host.id] = host
        # logger.debug('Creating the host %s as id %s' % (h.get_name(), h.id))
    
    
    def manage_initial_hostgroup_status_brok(self, b):
        data = b.data
        inst_id = data['instance_id']
        
        # Try to get the inp progress Hostgroups
        try:
            inp_hostgroups = self.inp_hostgroups[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_hostgroup_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        # With void members
        hg = Hostgroup([])
        
        # populate data
        self.update_element(hg, data)
        
        # We will link hosts into hostgroups later so now only save it
        inp_hostgroups[hg.id] = hg
    
    
    def manage_initial_service_status_brok(self, brok: Brok, skip_useless_in_configuration=True) -> None:
        data = brok.data
        inst_id = data['instance_id']
        
        # Try to get the "in progress" services
        try:
            inp_services = self.inp_services[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_service_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        service = BrokerService({}, skip_useless_in_configuration=skip_useless_in_configuration)
        service.list_cache_var = {}
        service.part_configuration_incarnation = brok.part_configuration_incarnation
        self.before_after_hook(brok, service)
        self.update_element(service, data)
        self.initial_status_freshness_check(service)
        
        # We need to rebuild Downtime and Comment relationship
        for dtc in service.downtimes:
            dtc.ref = service
        
        # Ok, put in the "in progress" services
        inp_services[service.id] = service
    
    
    # We create a servicegroup in our in progress part we will link it after
    def manage_initial_servicegroup_status_brok(self, b):
        data = b.data
        sgname = data['servicegroup_name']
        inst_id = data['instance_id']
        
        # Try to get the inp progress
        try:
            inp_servicegroups = self.inp_servicegroups[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_servicegroup_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        self.logger.debug('Creating a servicegroup: %s in instance %d' % (sgname, inst_id))
        
        # With void members
        sg = Servicegroup([])
        
        # populate data
        self.update_element(sg, data)
        
        # We will link hosts into hostgroups later
        # so now only save it
        inp_servicegroups[sg.id] = sg
    
    
    # For Contacts, it's a global value, so 2 cases:
    # We got it -> we update it
    # We don't -> we create it
    # In both cases we need to relink it
    def manage_initial_contact_status_brok(self, brok: 'Brok'):
        if 'manage_initial_contact_status_brok' not in self.detail_timer:
            self.detail_timer['manage_initial_contact_status_brok'] = {
                'build element'         : 0,
                'build notificationways': 0,
                'build contactgroups'   : 0,
                'build reversed list'   : 0,
            }
        detail_timer = self.detail_timer['manage_initial_contact_status_brok']
        
        contact_to_reindex = []
        notificationway_to_reindex = []
        
        t0 = time.time()
        data = brok.data
        cname = data['contact_name']
        inst_id = data['instance_id']
        
        contact: 'BrokerContact' = self.contacts.find_by_name(cname)
        if contact:
            self.update_element(contact, data)
            contact.configuration_incarnation = brok.part_configuration_incarnation.get_configuration_incarnation()
        else:
            contact = BrokerContact({}, skip_useless_in_configuration=True, configuration_incarnation=brok.part_configuration_incarnation.get_configuration_incarnation())
            self.update_element(contact, data)
            self.contacts[contact.id] = contact
        
        contact_to_reindex.append(contact)
        t1 = time.time()
        detail_timer['build element'] += t1 - t0
        
        # Delete some useless contact values
        contact.host_notification_commands = []
        contact.service_notification_commands = []
        contact.host_notification_period = []
        contact.service_notification_period = []
        
        # Now manage notification ways too same than for contacts. We create or update
        contact_notificationways = contact.notificationways
        new_notificationways = []
        for contact_notificationway in contact_notificationways:
            notificationway_name = contact_notificationway.notificationway_name
            notificationway = self.notificationways.find_by_name(notificationway_name)
            if not notificationway:
                notificationway = NotificationWay([])
                self.notificationways[notificationway.id] = notificationway
            
            # Now update it
            for prop in NotificationWay.properties:
                if hasattr(contact_notificationway, prop):
                    setattr(notificationway, prop, getattr(contact_notificationway, prop))
            new_notificationways.append(notificationway)
            notificationway_to_reindex.append(notificationway)
            
            # Linking the notification way
            # With commands
            self.linkify_commands(notificationway, 'host_notification_commands')
            self.linkify_commands(notificationway, 'service_notification_commands')
            
            # Now link timeperiods
            self.linkify_a_timeperiod(notificationway, 'host_notification_period')
            self.linkify_a_timeperiod(notificationway, 'service_notification_period')
        
        contact.notificationways = new_notificationways
        t2 = time.time()
        detail_timer['build notificationways'] += t2 - t1
        
        # linkify contacts with newly receive contactgroups and
        # at the end (in all_done_linking) they will be re-linkify with the linkified groups
        new_contact_groups = []
        for contact_group_name in contact.contactgroups:
            contact_group = self.inp_contactgroups[inst_id].find_by_name(contact_group_name)
            if contact_group:
                new_contact_groups.append(contact_group)
        contact.contactgroups = new_contact_groups
        
        t3 = time.time()
        detail_timer['build contactgroups'] += t3 - t2
        
        # build index for new contacts and notificationways
        self.contacts.reindex(contact_to_reindex)
        self.notificationways.reindex(notificationway_to_reindex)
        
        t4 = time.time()
        detail_timer['build reversed list'] += t4 - t3
    
    
    # From now we only create an hostgroup with unlink data in the in prepare list. We will link all of them at the end.
    def manage_initial_contactgroup_status_brok(self, b):
        data = b.data
        inst_id = data['instance_id']
        
        # Try to get the inp progress Contactgroups
        try:
            inp_contactgroups = self.inp_contactgroups[inst_id]
        except Exception as exp:
            logger.error('Error in manage_initial_contactgroup_status_brok : [%s]' % exp)
            logger.print_stack()
            return
        
        # With void members
        cg = Contactgroup([])
        
        # populate data
        self.update_element(cg, data)
        
        # We will link contacts into contactgroups later
        # so now only save it
        inp_contactgroups[cg.id] = cg
    
    
    # For Timeperiods we got 2 cases: do we already got the command or not.
    # if got: just update it
    # if not: create it and declare it in our main commands
    def manage_initial_timeperiod_status_brok(self, b):
        data = b.data
        tpname = data['timeperiod_name']
        
        tp = self.timeperiods.find_by_name(tpname)
        if tp:
            self.update_element(tp, data)
        else:
            tp = Timeperiod({})
            self.update_element(tp, data)
            self.timeperiods[tp.id] = tp
            # We add a timeperiod, we update the reversed list
            self.timeperiods.create_reversed_list()
    
    
    # For command we got 2 cases: do we already got the command or not.
    # if got: just update it
    # if not: create it and declare it in our main commands
    def manage_initial_command_status_brok(self, b):
        data = b.data
        cname = data['command_name']
        
        c = self.commands.find_by_name(cname)
        if c:
            # print 'Already existing command', cname, 'updating it'
            self.update_element(c, data)
        else:
            # print 'Creating a new command', cname
            c = Command({})
            self.update_element(c, data)
            self.commands[c.id] = c
            # Ok, we can regenerate the reversed list so
            self.commands.create_reversed_list()
    
    
    def manage_initial_scheduler_status_brok(self, b):
        data = b.data
        scheduler_name = data['scheduler_name']
        realm = data['realm']
        self.realms.add(realm)
        sched = SchedulerLink({})
        self.update_element(sched, data)
        self.schedulers[scheduler_name] = sched
    
    
    def manage_initial_poller_status_brok(self, b):
        data = b.data
        poller_name = data['poller_name']
        poller = PollerLink({})
        self.update_element(poller, data)
        self.pollers[poller_name] = poller
    
    
    def manage_initial_reactionner_status_brok(self, b):
        data = b.data
        reactionner_name = data['reactionner_name']
        reac = ReactionnerLink({})
        self.update_element(reac, data)
        self.reactionners[reactionner_name] = reac
    
    
    def manage_initial_broker_status_brok(self, b):
        data = b.data
        broker_name = data['broker_name']
        broker = BrokerLink({})
        self.update_element(broker, data)
        self.brokers[broker_name] = broker
    
    
    def manage_initial_receiver_status_brok(self, b):
        data = b.data
        receiver_name = data['receiver_name']
        receiver = ReceiverLink({})
        self.update_element(receiver, data)
        self.receivers[receiver_name] = receiver
    
    
    # This brok is here when the WHOLE initial phase is done. So we got all data, we can link all together :)
    def manage_initial_broks_done_brok(self, b):
        part_configuration_incarnation = b.part_configuration_incarnation
        # We do not want to modify elements while switching elements
        # NOTE: we do not lock all access, because we will only change atomic elements on the  elements
        with self.freshness_lock:
            self.all_done_linking(part_configuration_incarnation)
            self._have_received_initial_broks = True
    
    
    # ------- Status Update part
    def _ask_new_fill_brok(self, instance_id, scheduler_name='unknown'):
        # Maybe we did already ask for a regeneration, so don't hammer it
        if instance_id in self.currently_asking_full_broks:
            self.logger.debug('Skipping asking for instance id %d full brok information, because we already did it' % instance_id)
            return
        
        if self.from_module_to_main_daemon_queue is None:
            return
        
        logger_regenerator = self.logger.get_sub_part('scheduler=%s' % scheduler_name, register=False).get_sub_part('NEED DATA')
        logger_regenerator.info('Requesting initial broks from the broker about the scheduler \'%s\' (managing shard %s).' % (scheduler_name, instance_id))
        msg = Message(id=0, type='NeedData', data={'full_instance_id': instance_id}, source=self.import_in)
        self.from_module_to_main_daemon_queue.put(msg)
        self.currently_asking_full_broks[instance_id] = True
    
    
    # A scheduler send us a 'I'm alive' brok.
    # If we never heard about this one, we got some problem and we ask him some initial data :)
    def manage_update_program_status_brok(self, b):
        data = b.data
        instance_id = data['instance_id']
        
        # If we got an update about an unknown instance, cry and ask for a full version!
        if instance_id not in self.configs.keys():
            scheduler_name = data['instance_name']
            self._ask_new_fill_brok(instance_id, scheduler_name=scheduler_name)
            return
        
        # Ok, good conf, we can update it
        config = self.configs[instance_id]
        self.update_element(config, data)
    
    
    def manage_update_host_status_brok(self, b):
        # There are some properties that should not change and are already linked so just remove them
        clean_prop = [
            'check_command',
            'hostgroups',
            'contacts',
            'view_contacts',
            'notification_contacts',
            'edition_contacts',
            'notification_period',
            'contact_groups',
            'check_period',
            'event_handler',
            'maintenance_period',
            'realm',
            'escalations'
        ]
        
        # some are only use when a topology change happened
        topology_change = b.data['topology_change']
        if not topology_change:
            other_to_clean = ['childs', 'parents', 'child_dependencies', 'parent_dependencies']
            clean_prop.extend(other_to_clean)
        
        data = b.data
        for prop in clean_prop:
            del data[prop]
        
        hname = data['host_name']
        h = self.hosts.find_by_name(hname)
        
        if h:
            # If we are updating the host with old instance_id, we will have several issues, so just skip this
            # Note: this must be a old brok, because initial broks are always before this brok
            if h.instance_id != data['instance_id']:
                return
            
            self.before_after_hook(b, h)
            self.update_element(h, data)
            
            # If the topology change, update it
            if topology_change:
                logger.debug('[regenerator] Topology change for host %s and hostparent: %s' % (h.get_name(), h.parent_dependencies))
                self.linkify_host_and_hosts(h, 'parents')
                self.linkify_host_and_hosts(h, 'childs')
                self.linkify_dict_srv_and_hosts(h, 'parent_dependencies')
                self.linkify_dict_srv_and_hosts(h, 'child_dependencies')
            
            # Relink downtimes
            for dtc in h.downtimes:
                dtc.ref = h
    
    
    def manage_update_service_status_brok(self, b):
        # There are some properties that should not change and are already linked
        # so just remove them
        clean_prop = ['check_command', 'servicegroups',
                      'contacts', 'notification_period', 'contact_groups',
                      'check_period', 'event_handler',
                      'maintenance_period', 'escalations']
        
        # some are only use when a topology change happened
        toplogy_change = b.data['topology_change']
        if not toplogy_change:
            other_to_clean = ['child_dependencies', 'parent_dependencies']
            clean_prop.extend(other_to_clean)
        
        data = b.data
        for prop in clean_prop:
            del data[prop]
        
        hname = data['host_name']
        sdesc = data['service_description']
        s = self.services.find_srv_by_name_and_hostname(hname, sdesc)
        if s:
            # If we are updating the check with old instance_id, we will have several issues, so just skip this
            # Note: this must be a old brok, because initial broks are always before this brok
            if s.instance_id != data['instance_id']:
                return
            
            self.before_after_hook(b, s)
            self.update_element(s, data)
            
            # If the topology change, update it
            if toplogy_change:
                self.linkify_dict_srv_and_hosts(s, 'parent_dependencies')
                self.linkify_dict_srv_and_hosts(s, 'child_dependencies')
            
            # Relink downtimes with the service
            for dtc in s.downtimes:
                dtc.ref = s
    
    
    def manage_update_broker_status_brok(self, b):
        data = b.data
        broker_name = data['broker_name']
        try:
            s = self.brokers[broker_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    def manage_update_receiver_status_brok(self, b):
        data = b.data
        receiver_name = data['receiver_name']
        try:
            s = self.receivers[receiver_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    def manage_update_reactionner_status_brok(self, b):
        data = b.data
        reactionner_name = data['reactionner_name']
        try:
            s = self.reactionners[reactionner_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    def manage_update_poller_status_brok(self, b):
        data = b.data
        poller_name = data['poller_name']
        try:
            s = self.pollers[poller_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    def manage_update_scheduler_status_brok(self, b):
        data = b.data
        scheduler_name = data['scheduler_name']
        
        realm = data['realm']
        self.realms.add(realm)
        try:
            s = self.schedulers[scheduler_name]
            self.update_element(s, data)
        except Exception:
            pass
    
    
    # ------- Check result and schedule part
    def manage_host_check_result_brok(self, brok: Brok) -> None:
        data = brok.data
        host_name = data['host_name']
        
        host = self.hosts.find_by_name(host_name)
        if not host:
            return
        
        # Only a Scheduler with the same part_configuration_incarnation given in the initial_host brok can update the host.
        if host.part_configuration_incarnation != brok.part_configuration_incarnation:
            return
        self.before_after_hook(brok, host)
        self.update_element(host, data)
    
    
    # this brok should arrive within a second after the host_check_result_brok
    def manage_host_next_schedule_brok(self, brok: Brok) -> None:
        self.manage_host_check_result_brok(brok)
    
    
    # A service check have just arrived, we UPDATE data info with this
    def manage_service_check_result_brok(self, brok: Brok) -> None:
        data = brok.data
        host_name = data['host_name']
        service_description = data['service_description']
        service = self.services.find_srv_by_name_and_hostname(host_name, service_description)
        if not service:
            return
        
        # Only a Scheduler with the same part_configuration_incarnation given in the initial_service brok can update the host.
        if service.part_configuration_incarnation != brok.part_configuration_incarnation:
            return
        self.before_after_hook(brok, service)
        self.update_element(service, data)
    
    
    # A service check update have just arrived, we UPDATE data info with this
    def manage_service_next_schedule_brok(self, brok: Brok) -> None:
        self.manage_service_check_result_brok(brok)
